/* 
 * hci_internal.c --  Structs and defines used by hci.c and vendor specific
 *                    hci functions
 *
 * Copyright (C) 2000  Axis Communications AB
 *
 * Johan Zander <johan.zander@axis.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: hci_internal.h,v 1.1 2000/11/17 15:17:29 johanz Exp $
 *
 */

#ifndef HCI_INTERNAL_H
#define HCI_INTERNAL_H

/****************** INCLUDE FILES SECTION ***********************************/

#ifdef __KERNEL__
#include <linux/types.h>
#include <linux/bluetooth/btcommon.h>
#else
#include "local.h"
#include "btcommon.h"
#endif

/****************** DEBUG CONSTANT AND MACRO SECTION ************************/

/* Defines for the debug macros */
#if DEBUG_HCI_QUEUEING
#define D_QUEUE(fmt...) printk(HCI_DBG_STR fmt)
#else
#define D_QUEUE(fmt...) 
#endif

#if DEBUG_HCI_INDATA
#define D_INDATA(fmt...) printk(fmt)
#else
#define D_INDATA(fmt...)
#endif

/* Defines for the debug macros */
#if DEBUG_HCI_RECEIVING_PROCESS
#define D_REC(fmt...) printk(HCI_DBG_STR fmt)
#else
#define D_REC(fmt...)
#endif

#if DEBUG_HCI_SEND_PROCESS
#define D_SND(fmt...) printk(HCI_DBG_STR fmt)
#else
#define D_SND(fmt...)
#endif

#if DEBUG_HCI_CTRL
#define D_CTRL(fmt...) printk(HCI_DBG_STR fmt)
#else
#define D_CTRL(fmt...)
#endif

#if DEBUG_DATAFLOW
#define PRINTPKT(str, data, len) print_data(str, data, len)
#else
#define PRINTPKT(str, data, len)
#endif

#if DEBUG_HCI_CMD
#define D_CMD(fmt...) printk(HCI_DBG_STR fmt)
#else
#define D_CMD(fmt...)
#endif
		
/****************** CONSTANT AND MACRO SECTION ******************************/

/* How many inbuffers we have got, the size of them, and how many command
   buffers we have got */
#define NBR_OF_HCI_INBUFFERS 7
#define HCI_IN_SIZE 17000 //2048
#define NBR_CMD_BUFS 10

/* Defines for the packet types */
#define CMD_PKT 0x01
#define ACL_PKT 0x02
#define SCO_PKT 0x03
#define EVENT_PKT 0x04
#define ERROR_PKT 0x05
#define NEG_PKT 0x06

/* Define the different OpCode Group Field (OGF) values */
#define HCI_LC 0x01   /* Link Control Command */
#define HCI_LP 0x02   /* Link Policy Command */
#define HCI_HC 0x03   /* Host Controller and Baseband Commands */
#define HCI_IP 0x04   /* Informational Parameters */
#define HCI_SP 0x05   /* StatusParameters */
#define HCI_TC 0x06   /* Test Commands */
#define MANUFACTURER_SPEC 0x3f

/* Define the different OpCode Command Field (OCF) values */

/* Link control commands (HCI_LC) */
#define INQUIRY 0x01
#define INQUIRY_CANCEL 0x02
#define PERIODIC_INQUIRY_MODE 0x3
#define EXIT_PERIODIC_INQUIRY_MODE 0x4
#define CREATE_CONNECTION 0x05
#define DISCONNECT 0x06
#define ADD_SCO_CONNECTION 0x07
#define ACCEPT_CONNECTION_REQUEST 0x09
#define REJECT_CONNECTION_REQUEST 0x0A
#define LINK_KEY_REQUEST_REPLY 0xB
#define LINK_KEY_REQUEST_NEGATIVE_REPLY 0xC
#define PIN_CODE_REQUEST_REPLY 0xD
#define PIN_CODE_REQUEST_NEGATIVE_REPLY 0xE
#define CHANGE_CONNECTION_PACKET_TYPE 0x0F
#define AUTHENTICATION_REQUESTED 0x11
#define SET_CONNECTION_ENCRYPTION 0x13
#define CHANGE_CONNECTION_LINK_KEY 0x15
#define MASTER_LINK_KEY 0x17
#define REMOTE_NAME_REQUEST 0x19
#define READ_REMOTE_SUPPORTED_FEATURES 0x1B
#define READ_REMOTE_VERSION_INFORMATION 0x1D
#define READ_CLOCK_OFFSET 0x1F

/* Link Policy commands (HCI_LP) */
#define HOLD_MODE 0x1
#define SNIFF_MODE 0x3
#define EXIT_SNIFF_MODE 0x4
#define PARK_MODE 0x5
#define EXIT_PARK_MODE 0x6
#define QOS_SETUP 0x7
#define ROLE_DISCOVERY 0x9 
#define SWITCH_ROLE 0xB
#define READ_LINK_POLICY_SETTINGS 0xC
#define WRITE_LINK_POLICY_SETTINGS 0xD

/* Host-Controller and Baseband Commands (HCI_HC) */
#define RESET 0x03
#define FLUSH 0x08
#define READ_STORED_LINK_KEY 0x0D
#define CHANGE_LOCAL_NAME 0x13
#define WRITE_SCAN_ENABLE 0x1A
#define WRITE_PAGESCAN_ACTIVITY 0x1C
#define WRITE_INQUIRYSCAN_ACTIVITY 0x1E
#define WRITE_AUTHENTICATION_ENABLE 0x20
#define WRITE_CLASS_OF_DEVICE 0x24
#define SET_HOST_CONTROLLER_TO_HOST_FLOW_CONTROL 0x31
#define HOST_BUFFER_SIZE 0x33
#define HOST_NBR_OF_COMPLETED_PACKETS 0x35
#define READ_LINK_SUPERVISION_TO 0x36
#define WRITE_LINK_SUPERVISION_TO 0x37

/* Informational Parameters (HCI_IP) */
#define READ_LOCAL_VERSION_INFORMATION 0x1 
#define READ_LOCAL_SUPPORTED_FEATURES 0x3
#define READ_BUFFER_SIZE 0x05
#define READ_COUNTRY_CODE 0x7
#define READ_BD_ADDR 0x09

/* Status Parameters (HCI_SP) */
#define READ_FAILED_CONTACT_COUNTER 0x1
#define RESET_FAILED_CONTACT_COUNTER 0x2
#define GET_LINK_QUALITY 0x3
#define READ_RSSI 0x5

/* Testing commands (HCI_TC) */
#define READ_LOOPBACK_MODE 0x01
#define WRITE_LOOPBACK_MODE 0x02
#define ENABLE_DEVICE_UNDER_TEST_MODE 0x03

/* Defines of the different events that is sent from the Host Controler
   to the Host */
#define INQUIRY_COMPLETE 0x01
#define INQUIRY_RESULT 0x02
#define CONNECTION_COMPLETE 0x03
#define CONNECTION_REQUEST 0x04
#define DISCONNECTION_COMPLETE 0x05
#define AUTHENTICATION_COMPLETE 0x6
#define REMOTE_NAME_REQUEST_COMPLETE 0x7
#define ENCRYPTION_CHANGE 0x8
#define CHANGE_CONNECTION_LINK_KEY_COMPLETE 0x9
#define MASTER_LINK_KEY_COMPLETE 0xA
#define READ_REMOTE_SUPPORTED_FEATURES_COMPLETE 0xB
#define READ_REMOTE_VERSION_INFORMATION_COMPLETE 0xC
#define QOS_SETUP_COMPLETE 0xD
#define COMMAND_COMPLETE 0x0E
#define COMMAND_STATUS 0x0F
#define HARDWARE_ERROR 0x10
#define FLUSH_OCCURRED 0x11
#define ROLE_CHANGED 0x12
#define NBR_OF_COMPLETED_PACKETS 0x13
#define MODE_CHANGE 0x14
#define RETURN_LINK_KEYS 0x15
#define PIN_CODE_REQUEST 0x16
#define LINK_KEY_REQUEST_EVENT 0x17
#define LINK_KEY_NOTIFICATION 0x18
#define LOOPBACK_COMMAND 0x19
#define DATA_BUFFER_OVERFLOW 0x1A
#define MAX_SLOTS_CHANGE 0x1B
#define READ_CLOCK_OFFSET_COMPLETE 0x1C
#define CONNECTION_PACKET_TYPE_CHANGED 0x1D
#define QOS_VIOLATION 0x1E
#define PAGE_SCAN_MODE_CHANGE 0x1F
#define PAGE_SCAN_REPETITION_MODE_CHANGE 0x20

/* Vendor specific events */
#define VENDOR_EVENT 0xFF

/* The flags in the HCI header */
#define L2CAP_FRAME_START 0x02
#define L2CAP_FRAME_CONT 0x01

#define HCI_HDR_LEN 1
#define ACL_HDR_LEN 4
#define SCO_HDR_LEN 3
#define CMD_HDR_LEN 3
#define EVENT_HDR_LEN 2

/****************** TYPE DEFINITION SECTION *********************************/

/* Struct defining a HCI command packet, the meaning of the type field is
   decribed in part H:4 chapter 2 in the baseband core specification. The
   other fields are described in section H:1 chapter 4.4.1 in the baseband
   core specification */ 

typedef struct cmd_pkt {
	u32 type:8;
        u32 ocf:10;
        u32 ogf:6;
        u32 len:8;
        
        u8 data[256];
} cmd_pkt;

/* Struct used to keep track of the current number of buffers, and the sizes
   of the buffers in the bluetooth module. There are two buffers for data in
   the modul, ACL buffers for ordinary data and SCO buffers for audion data.
   The cmd_num, describes how many more command packets that can be sent to
   the module. */

typedef struct host_controller_buffers{
	s32 acl_len;
	s32 sco_len;
	s32 acl_num;
	s32 sco_num;
	s32 cmd_num;
} host_controller_buffers;

/* The hci_in_buffer is used for storing hci packtes until a complete L2CAP
   packet has been received. When a complete L2CAP packet is received and
   processed by the L2CAP layer the empty variable is set to TRUE */

typedef struct hci_in_buffer {
	u8 *buf_ptr;
	u32 count;
	u32 empty;
	u32 hci_hdl;
	u32 nbr_of_hci_pkt;
	u32 l2cap_len;
	u8 buf[HCI_IN_SIZE];
} hci_in_buffer;

enum bt_states{UNIT_ACTIVE, UNIT_PASSIVE, NOT_CONNECTED};

typedef struct hci_con {
	enum bt_states state;
	u8 bd[6];
	u16 con_hdl;
	u8 name[248];
} hci_con;

/* Used as a control object for the entire HCI layer. The trash_buffer is
   only used as an emergency buffer to avoid a program crash if something has
   gone wrong with the hci_in_buffers */

/* FIXME: Do we really need the trash_buffer? */

typedef struct hci_controller {
	host_controller_buffers hc_buf;
	hci_in_buffer hci_in_buf[NBR_OF_HCI_INBUFFERS];
	hci_in_buffer trash_buffer;
	u32 nbr_of_connections;
	u8 local_bd[6];
        s32 active_connection;
	hci_con con[MAX_NBR_OF_CONNECTIONS];
} hci_controller;

/****************** EXPORTED FUNCTION DECLARATION SECTION *******************/

/*
   Called from both hci.c and hci_vendor.c.
   Implemented in hci.c
*/
void release_cmd_timer(void);
void start_cmd_timer(void);
s32 send_cmd(u8 *cmd, u8 len);

/*
   Called from hci.c.
   Implemented in hci_vendor.c
*/
void process_vendor_return_param(u32 ocf, u8* r_val);
void process_vendor_event(u8 *buf, u32 len, u32 event_code);

#endif
/****************** END OF FILE hci_internal.h ******************************/
