/* 
 * btcommon.h -- Contains common object structs / defines for bluetooth stack
 *
 * Copyright (C) 2000  Axis Communications AB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: btcommon.h,v 1.39 2000/11/24 13:41:02 johanz Exp $
 *
 */

#ifndef BTCOMMON_H
#define BTCOMMON_H

/****************** INCLUDE FILES SECTION ***********************************/

#ifdef __KERNEL__
#include <linux/types.h>
#include <linux/bluetooth/btdebug.h>
#include <linux/bluetooth/btconfig.h>
#include <linux/timer.h>
#else
#include "local.h"
#include "btdebug.h"
#include "btconfig.h"
#include <sys/unistd.h> /* for the sleep ..*/
#endif

/* Defines common to all files */

#define FALSE 0
#define TRUE 1

#ifndef __KERNEL__
#define kmalloc(size,prio) malloc((size))
#define kfree(obj) free((obj))
#define wake_up_interruptible(p)
#define interruptible_sleep_on(p) sleep(1)
#endif

#define MIN(a,b) ((a) < (b) ? (a) : (b))
#define MAX(a,b) ((a) > (b) ? (a) : (b))

#define CHAR2INT10(c1,c0) (((u32)((c1) & 0x03) << 8) + (u32)((c0) & 0xff))
#define CHAR2INT12(c1,c0) (((u32)((c1) & 0x0f) << 8) + (u32)((c0) & 0xff))
#define CHAR2INT16(c1,c0) (((u32)((c1) & 0xff) << 8) + (u32)((c0) & 0xff))
#define CHAR2INT32(c3,c2,c1,c0) (((u32)((c3) & 0xff) << 24) + \
				 ((u32)((c2) & 0xff) << 16) + \
				 ((u32)((c1) & 0xff) << 8) + (u32)((c0) &0xff))
/* extract line number from tty */
#define GET_TTYLINE(tty) (MINOR(tty->device) - tty->driver.minor_start)

/****************** Defines realted to the bluetooth implementation *********/

#define SERIAL_PORT_PROFILE 0
/* FIXME: Will this be specified in the spec somewhere... */ 

/* The BC flag in the HCI header */
#define NO_BROADCAST 0
#define ACTIVE_BROADCAST 1
#define PICONET_BROADCAST 2

#define MAX_NBR_OF_CONNECTIONS 7

/****************** Defines realated to the management of the bt-driver *****/

#define BT_NR_DATAPORTS 7
#define BT_NR_CTRLPORTS 1
#define BT_NR_PORTS (BT_NR_CTRLPORTS + BT_NR_DATAPORTS)

/* Define the ioctls to the bt driver */
#define BT_IOC_MAGIC 'B' /* Use B as a magic number */
#define BT_IOC_MAXNR 255
#define BTCONNECT _IOW(BT_IOC_MAGIC, 0, bt_connection)
#define BTDISCONNECT _IOW(BT_IOC_MAGIC, 1, s32)
#define BTSETSPEED _IOW(BT_IOC_MAGIC, 2, s32)
#define BTINITSTACK _IO(BT_IOC_MAGIC, 3)
#define BTSETSERTTY _IO(BT_IOC_MAGIC, 4)
#define BTSETMODEMDUMMY _IOW(BT_IOC_MAGIC, 5, s32)
#define BTSHUTDOWN _IO(BT_IOC_MAGIC, 6)
#define BTREADREMOTEBDADDR _IOR(BT_IOC_MAGIC, 0x7, u8[6])

/* ioctls executing HCI commands */
#define HCIINQUIRY _IOWR(BT_IOC_MAGIC, 10, inquiry_results)
#define HCIREADLOCALBDADDR _IOR(BT_IOC_MAGIC, 0x11, u8[6])
#define HCIWRITESCANENABLE _IOW(BT_IOC_MAGIC, 0x12, u32)
#define HCIWRITEPAGESCANACTIVITY _IOW(BT_IOC_MAGIC, 0x13, u32[2])
#define HCISETLOCALNAME _IOW(BT_IOC_MAGIC, 0x14, u8[248])
#define HCIENABLEDUT _IO(BT_IOC_MAGIC, 0x15)

/* ioctls vendor specific HCI commands */
#define HCISETBAUDRATE _IOW(BT_IOC_MAGIC, 0x20, s32)
#define HCIWRITEBDADDR _IOW(BT_IOC_MAGIC, 0x21, u8[6])
#define HCISENDRAWDATA _IOW(BT_IOC_MAGIC, 0x22, u8[64])

/* other ioctls used for testing */
#define BTSENDTESTDATA _IOW(BT_IOC_MAGIC, 0xf0, s32[2])
#define HCITESTCONNECTREQ _IOW(BT_IOC_MAGIC, 0xf1, u8[6])

/* NOTE !
 * N_BT should be defined in /include/asm/termios.h 
 * However, if you are compiling this source standalone
 * the following define can be useful. If this number is taken 
 * simply change it to something not taken.
 */

#ifndef N_BT
#define N_BT 15 /* hopefully available :) */
#endif

#define MAX_BT_TMP_BUF 4096

/* Bluetooth HW defines used to setup different HW and to cope with 
   limitations in HW. Uncomment one to use the specified hardware.
   Must be changed in btd.h as well. */

//define GENERIC_HW
#define ERICSSON_HW
//#define DIGIANSWER_HW
//#define CSR_HW
//#define NO_HW

/****************** TYPE DEFINITION SECTION *********************************/

/* The BT transmit object used when sending data */
typedef struct bt_tx_buf {
	u32 magic;
	u32 hci_hdl; /* The destination hci connection handle */
	u8 pb_flag; /* Packet boundary flag */
	u8 bc_flag; /* Broadcast flag */

	/* 'Filled in' data length  (up to this point) when creating object.
	   Also used when sending data to mark how much that has been sent
	   hci_send_data calculates what to send by : 
	   to_send=tx_buf->data+tx_buf->subscr_len-tx->cur_len */
	s32 cur_len; 
	/* Total length of the data area not including this buffer hdr  */
	s32 subscr_len; 
	s32 flushed;  /* If set, discard at send level */
	u8 data[0]; /* The actual data segment to send */
} bt_tx_buf;

/* SDP layer object */
typedef struct sdp_tx_buf{
	u8 hci_hdr[5]; 
	u8 l2cap_hdr[4];
	u8 frame[0];
} sdp_tx_buf;

/* RFComm layer object */
typedef struct rfcomm_tx_buf{
	u8 hci_hdr[5];
	u8 l2cap_hdr[4];
	u8 frame[0];
} rfcomm_tx_buf;

/* TCS layer object */
typedef struct tcs_tx_buf{
	u8 hci_hdr[5];
	u8 l2cap_hdr[4];
	u8 frame[0];
} tcs_tx_buf;

/* L2CAP layer object */
typedef struct l2cap_tx_buf{
	u8 hci_buf[5];  
	u8 frame[0];  
} l2cap_tx_buf;

#define BT_TX_HDRSIZE sizeof(bt_tx_buf)
#define L2CAP_TX_HDRSIZE sizeof(l2cap_tx_buf)
#define RFCOMM_TX_HDRSIZE sizeof(rfcomm_tx_buf)
#define SDP_TX_HDRSIZE sizeof(sdp_tx_buf)

/****************** Some typedefs used in the l2cap_con struct **************/

typedef u8 BD_ADDR[6];

typedef u16 CID;

typedef struct flow {
	u8 type; /* = 0x3 or 0x4 ??? */
	u8 len; /* = 22 (0x16) */
	u8 flags; /* default 0 */
	u8 service; /* default 0x01 (best effort) */
	u32 token_rate;
	u32 bucket_size; /* bytes */
	u32 peak; /* bps */
	u32 latency; /* ms */ 
	u32 delay; /* ms */
} flow;
 
/****************** Definintion of the L2CAP connection object **************/


/* FIXME -- move all layer specific structs into each layers .h file 
   and fix include order ! */

/* used to resend lost l2cap commands */
typedef struct l2cap_cmd_backup {
	u8 type; /* conreq/confreq/discreq */	
	u8 nbr_resends;
	/* params for config req */
	u16 in_mtu;
	u16 flush_timeout;
	u16 link_to;
	flow *outflow; /* danger if this is deallocated in upper layers */
	u16 inforeq_type;
} l2cap_cmd_backup;

typedef struct l2cap_con {
	s32 magic;
	BD_ADDR remote_bd; /* 6 bytes */
	u16 hci_hdl;

	/* FIXME - add multiple HCI handles for group 
	   management functionality */

	u16 local_cid;
	u16 remote_cid;
	s32 current_state;
	u16 psm;
	u16 local_mtu;

	/* remote options temp stored when receiving config req */
	u16 remote_mtu;
	u16 flush_timeout;
	flow remote_qos;

	u8 initiator;

	u8 link_up;  
	u8 sig_id_sent; /* last sent command id */
	u8 sig_id_rcv; /* last received command id*/
	

	u8 ping_sent;
	u8 inforeq_sent;

	s32 reconfiguring; /* indicates if we are currently reconfiguring */
	
	s32 conf_req_ready; /* indicates if we received pos rsp on our req */
	s32 conf_rsp_ready; /* indicates if we replied pos on a config req */

#ifdef __KERNEL__
	struct timer_list rtx_timer;
	struct timer_list ertx_timer;
#else
	/* timers for usermode is in l2cap.c */
#endif
	struct l2cap_cmd_backup last_cmd;  /* used to resend signals */

	/* stats packets sent/received/lost etc*/

	void *upper_con; /* upper layer connection object */

	struct l2cap_con *prev;
	struct l2cap_con *next; /* Next connection in list */
} l2cap_con;

/* Function pointers to upper layers, confirm and indication functions */
typedef struct protocol_layer {
	u32 psm;
	/* add psm value and skip registered */
	void (*con_ind)(l2cap_con *con);
	void (*conf_ind)(l2cap_con *con);
	void (*disc_ind)(l2cap_con *con);

	void (*con_pnd)(l2cap_con *con, s32 status);
	void (*con_cfm)(l2cap_con *con, s32 result);
	void (*conf_cfm)(l2cap_con *con, s32 result);
	void (*disc_cfm)(l2cap_con *con);

	void (*receive_data)(l2cap_con *con, u8 *data, u32 len);
	struct protocol_layer *next_layer; 
} protocol_layer;

/****************** RFCOMM connection object ****************************/

enum rfcomm_states{DISCONNECTED, CONNECTING, NEGOTIATING, CONNECTED, DISCONNECTING, FLOW_STOPPED};

typedef struct dlci_struct{
	enum rfcomm_states state;
	u8 credits;
	u8 initiated;
	u8 initiator;
	u16 mtu;
} dlci_struct;

typedef struct rfcomm_con{
	u32 magic;
	s32 line;
	u8 credit_flow;
	dlci_struct dlci[62];
	u8 initiator;
	u8 server_chn;
	l2cap_con *l2cap;
} rfcomm_con;

/****************** BT connection object ********************************/

typedef struct bt_connection{
	u8 bd[6];
	u32 psm;
	u16 layer_specific;
} bt_connection;


/****************** Other structures used in the stack **********************/

enum bt_session_states { BT_INACTIVE, BT_LOWERCONNECTED, 
			 BT_UPPERCONNECTED, BT_ACTIVE };

/* tty->driver_data... */
typedef struct bt_session
{
	struct tty_struct *upper_tty;
	rfcomm_con *rfcomm;
	u8 dlci;
	s32 state;
} bt_session;

typedef struct bt_ctrl_struct {
	struct bt_session session[BT_NR_PORTS];
	struct tq_struct *tx_queue;
	s32 nbr_active;
	s32 nbr_upper;
	s32 tty_last_unthrottled; /* indicated which line that was woken 
				     up last time buffers were below the 
				     unthrottle level */
} bt_ctrl_struct;

typedef struct bt_stat_struct
{
	s32 bytes_sent;
	s32 bytes_received;
} bt_stat_struct;

extern bt_stat_struct bt_stat;

typedef struct inquiry_results {
	u32 nbr_of_units;
	u8 bd_addr[0];
} inquiry_results;

typedef struct serport_profile_info {
	s32 rfcomm_serv_chan;
	u32 serport_name_length;
	u8 serport_name[64];
} serport_profile_info;


extern s32 bt_current_hw;


#endif
/****************** END OF FILE btcommon.h **********************************/
