;;**************************************************************************
;; FILE NAME : md
;;
;; DESCRIPTION : GCC instruction patterns for the GNU C Compiler Cris version.
;;
;;------------------------------------------------------------------------
;;          Copyright (c) 1991, 1995, Axis Communications, Lund, Sweden
;;
;; HISTORY
;;
;; DATE     NAME     CHANGES
;; ----     ----     -------
;; 91****   RW       Initiated
;; 911220   RW       Alpha release (0.01.1)
;; 930127   HP       Last bug found for CRIS first version.
;; 930211   HP       Other bugs found and solved.
;; 930402   HP       Annoying or.d with <=65535 && <=255 changed to or.b
;;                   && or.w.  Maybe same with andsi3 ?
;; 930429   HP       Updated to cris ver 0.32
;; 930907   HP       Only if small-arit (TARGET_SMALL_ARIT) use
;;                   arithmetic in HI and QI modes.  This may or may
;;                   not give better code when mixing modes a lot.
;;                   Maybe TARGET_SMALL_ARIT only should affect sub/add/cmp.
;; 950320   HP       Changed all "To" constraints to use "!" so's
;;                   insns matching r r n (must be reloaded) will NOT
;;                   get reloaded into r r o instead of r 0 n.
;;                   (If they will, they will end up in the "constant
;;                   pool" which is not desirable, and whats more,
;;                   the constant pool handling has a fatal bug in 2.6.3). 
;;                    Really, the trick about rejecting BOTH operands
;;                   if one is "<the_other>" as in "0" is not very
;;                   good.  Other stuff tend to be reloaded, for a
;;                   seemingly less cost. 
;; 951228   HP        This is still needed in 2.7.2 (previous comment
;;                   for 2.6.3)
;; 960216   HP        After a serious discussion with Richard Kenner,
;;                    <kenner@vlsi1.ultra.nyu.edu> (the guy who
;;                   answered at this time to bug-gcc) about a
;;                   reloader bug, ALL define_insn patterns with
;;                   operands with unspecified modes have been "expanded".
;;                    I now believe a "define_insn" (with or without
;;                   name) with operands with unspecified modes are
;;                   subject to reloader problems; resulting in "abort()".
;;                    I hope Kenner (or other) finally agrees AT LEAST
;;                   to the need to put in a note about this in the manual...
;; 960923   HP        Using a SUBREG on operand 0 of any zero_extract
;;                   construct is not correct, not even according to
;;                   the gcc porting manual, and definitely not
;;                   according to other targets.  This bug bit when
;;                   compiling lex.c::yylex() of gcc-2.1, causing a
;;                   SIGSEGV on a sunos5.  The code must have rotted,
;;                   this should have failed many tests ago!
;;                    Also, there is only need for one unnamed btst
;;                   pattern;  REG_P() had to be changed to REG_S_P()
;;                   (defined in cris.h), accepting SUBREGs
;;                   surrounding the REG expressions for operand
;;                   checking in unnamed-patterns-conditions. 
;;                    All regression tests are ok: (c-1.41 nps560,
;;                   ipps, cd); nps560 had one extra btstq insn due to
;;                   inconsistent subreg checking in final.c::final_scan_insn()
;; 961129   HP        Snapshot ss-961122 uncovered a bug in one of the "split"
;;                   patterns.
;;
;;**************************************************************************

;; This file is part of GNU CC.

;; GNU CC is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU CC is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU CC; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;- Instruction patterns.  When multiple patterns apply,
;;- the first one in the file is chosen.
;;-
;;- See file "rtl.def" for documentation on define_insn, match_*, et. al.
;;-
;;- cpp macro #define NOTICE_UPDATE_CC in file tm.h handles condition code
;;- updates for most instructions.

;;
;; We need an attribute to define whether an instruction can be put in
;; a branch-delay slot or not. This depends on the instruction 
;; (all instructions except jumps and branches are allowed) and the
;; addressing mode (must not be prefix or referring to pc).
;;  In short, instructions must be 16 bit and not refer/alter pc.
;;
;; The default for this attribute is not-slottable, so we dont have to
;; set it explicitly for all different branches and hairy address
;; modes with special insn patterns. The default that is defined here
;; could depend on the kind of insn and the addressing mode, but that
;; would need more attributes and hairier code below.
;;
;;  The method used is to use the constraint to guide whether the insn
;; is slottable or not.  This is the shortest way to win and
;; (hopefully) the most understandable.
;;  There is an extra constraint, Q, which recognizes: indirect reg,
;; except when reg is pc.
;;;;  You need to check the particular instruction if it has a valid
;; "slottable" quick-immediate operand, and include it in the
;; constraint pattern for the slottable pattern.
;;
;; Yes/no means if the insn is slottable or not. Branch means that the
;; insn is a branch insn (which is not slottable).

(define_attr "slottable" "no,yes,branch" (const_string "no"))

;; A branch is delayed one cycle; the instruction just after the
;; branch is read and executed.  That instruction is not annulled
;; neither if the branch is taken or not.  Only some instructions
;; are allowed.
;;
;; Note: some attempts have been done to make gcc avoid putting
;; redundant insns into delay slots.  Since delayed-branch scheduling
;; is done before removal of reduntant test insns, this may sometimes
;; move a redundant test insn into a delay slot.  But on the average
;; it does not make sense to avoid test insns in delay slots.
;;  If you have new ideas about this, uncomment the code below and
;; rewrite redundant() in aux-output.c .
;;  Maybe all this will be unnecessary if gcc is changed to be more
;; optimal for machines with delay slots.
;;  For now it doesnt even attempt to rearrange the insns to remove
;; test insns (regardless of delay slots).
;; ??? Is this still true for 2.7.2?
;;
(define_delay (eq_attr "slottable" "branch")
  [
;;   (and
    (eq_attr "slottable" "yes")
;;    (ne (symbol_ref "(TARGET_SLOTALL || !redundant(insn))")
;;      (const_int 0)))
   (nil) (nil)])
;;
;; NOTE: besides the attributes, there also has to be a %# at the end
;; of a "delayed" instruction output pattern
;; (see "jump" => "ba %l0%#"), so print_operand can catch it and print  
;; a "nop" if necessary. (The method is snatched from sparc.md)
;;

;
; Test insns.
;

; DImode
;
; Allow register and offsettable mem operands only.
(define_insn "tstdi"
  [(set (cc0)
        (match_operand:DI 0 "nonimmediate_operand" "r,o"))]
  ""
  "test.d %M0\;ax\;test.d %H0")

;
;; Some test insns with side-effect on the mem addressing go here.
;;
;; test [rx=ry+o] trying to avoid when rx is ry
;;
;;;;; See note on cmp-insns with side-efects (or lack of them)

;
; Normal named test patterns from SI on.
;
(define_insn "tstsi"
  [(set (cc0)
        (match_operand:SI 0 "nonimmediate_operand" "r,Q>,m"))]
  ""
  "test.d %R0"
  [(set_attr "slottable" "yes,yes,no")])

(define_insn "tsthi"
  [(set (cc0)
        (match_operand:HI 0 "nonimmediate_operand" "r,Q>,m"))]
  ""
  "test.w %R0"
  [(set_attr "slottable" "yes,yes,no")])

(define_insn "tstqi"
  [(set (cc0)
        (match_operand:QI 0 "nonimmediate_operand" "r,Q>,m"))]
  ""
  "test.b %R0"
  [(set_attr "slottable" "yes,yes,no")])

;
; It seems that the position of the sign-bit && the fact that 0.0 is
; all-bits 0 makes "tstsf" a straight-forward implementation.
; Note that the same is not true for a comparison.
;  We disregard the possibility of -0 (which really is legal in ieee 754).
; Remember that when coding float-lib functions!
;
; In future: check this one for next_cc0_user to determine if zero or
; negative (or both, then special precautions) is used. -0 is both
; zero and negative.  Should we really ignore the -0 case and let it
; result in negative AND non-zero? 
;
;
;; Pattern commented out because of the -0 case.  Is kept here because
;; we should take care of that case as a special; check if we use the
;; zero, non-zero, positive or negative part of the condition afterwards.
;
;(define_insn "tstsf"
;  [(set (cc0)
;        (match_operand:SF 0 "nonimmediate_operand" "r,Q>,m"))]
;  ""
;  "test.d %R0"
;  [(set_attr "slottable" "yes,yes,no")])

;
; Compare insns.
;

;
; We could optimize the sizes of the immediate operands for various
; cases, but that isnt worth it because of the very little usage of
; DI-mode as anything else but a structure/block-mode. Just do the
; obvious stuff for the defined constraint letters. 
;
; For the same non-worth reason, extra cse for address operands is not done here.
;
(define_insn "cmpdi"
  [(set (cc0)
        (compare (match_operand:DI 0 "nonimmediate_operand" "r,r,r,r,r,r,o")
                 (match_operand:DI 1 "general_operand"      "K,I,P,n,r,o,r")))]
  ""
  "@
   cmpq %1,%M0\;ax\;cmpq 0,%H0
   cmpq %1,%M0\;ax\;cmpq -1,%H0
   cmp%e1.%z1 %1,%M0\;ax\;cmpq %H1,%H0
   cmp.d %M1,%M0\;ax\;cmp.d %H1,%H0
   cmp.d %M1,%M0\;ax\;cmp.d %H1,%H0
   cmp.d %M1,%M0\;ax\;cmp.d %H1,%H0
   cmp.d %M0,%M1\;ax\;cmp.d %H0,%H1")

;;
;; Compare insns with side effect addressing mode
;;
;; cmp.S [rx=ry+i],rz;
;; cmp.S [%3=%1+%2],%0
;;
;; Not possible since the reloader DOES NOT ACCEPT cc-changing insns
;; with side-effects other than setting the condition codes.  The
;; reason is that the reload stage MAY cause another insn to be output
;; after the main instruction, causing the condition codes to be
;; invalid.  (This does not apply to the CRIS case, since move to
;; memory does not change the condition code, but we have no way to
;; force this case.
;;
;; All "old" compare-with-side-effect patterns removed.

;
; We now have cmps and cmpu (compare reg w. sign/zero extended mem).
; Most useful with 8/16-bit constants ([pc+]), but make a pattern and
; let gcc see it. Avoid sideeffect patterns, though (see above).
;
; QI-mode
(define_insn ""
  [(set (cc0)
        (compare
         (match_operand:SI 0 "register_operand" "r,r")
         (match_operator:SI 2 "extend_operator"
                         [(match_operand:QI 1 "memory_operand" "Q>,m")])))]
  ""
  "cmp%e2.%s1 %1,%0"
  [(set_attr "slottable" "yes,no")])

; HI-mode
(define_insn ""
  [(set (cc0)
        (compare
         (match_operand:SI 0 "register_operand" "r,r")
         (match_operator:SI 2 "extend_operator"
                         [(match_operand:HI 1 "memory_operand" "Q>,m")])))]
  ""
  "cmp%e2.%s1 %1,%0"
  [(set_attr "slottable" "yes,no")])

;
; Swap operands, we dont know the canonical look (if any).
;
; QI-mode
(define_insn ""
  [(set (cc0)
        (compare
         (match_operator:SI 2 "extend_operator"
                            [(match_operand:QI 0 "memory_operand" "Q>,m")])
         (match_operand:SI 1 "register_operand" "r,r")))]
  ""
  "cmp%e2.%s0 %0,%1" ; notice_update_cc knows about swapped operands
                     ; to compares.
  [(set_attr "slottable" "yes,no")])

; HI-mode
(define_insn ""
  [(set (cc0)
        (compare
         (match_operator:SI 2 "extend_operator"
                            [(match_operand:HI 0 "memory_operand" "Q>,m")])
         (match_operand:SI 1 "register_operand" "r,r")))]
  ""
  "cmp%e2.%s0 %0,%1" ; notice_update_cc knows about swapped operands
                     ; to compares.
  [(set_attr "slottable" "yes,no")])

;
; The normal compare patterns, from SI on.
;
(define_insn "cmpsi"
  [(set (cc0)
        (compare
         (match_operand:SI 0 "nonimmediate_operand" "r,r,r,r,Q>,Q>,r,r,m,m")
         (match_operand:SI 1 "general_operand"     "I,r,Q>,M,M,r,P,g,M,r")))]
  ""
  "@
   cmpq %1,%0
   cmp.d %1,%0
   cmp.d %1,%0
   test.d %0
   test.d %0
   cmp.d %0,%1
   cmp%e1.%z1 %1,%0
   cmp.d %R1,%0
   test.d %R0
   cmp.d %R0,%1" 
  [(set_attr "slottable" "yes,yes,yes,yes,yes,yes,no,no,no,no")])

(define_insn "cmphi"
  [(set (cc0)
        (compare (match_operand:HI 0 "nonimmediate_operand" "r,r,Q>,Q>,r,m,m")
                 (match_operand:HI 1 "general_operand"     "r,Q>,M,r,g,M,r")))]
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "@
   cmp.w %1,%0
   cmp.w %1,%0
   test.w %0
   cmp.w %0,%1
   cmp.w %R1,%0
   test.w %R0
   cmp.w %R0,%1"
  [(set_attr "slottable" "yes,yes,yes,yes,no,no,no")])  ;; no quick for this one

(define_insn "cmpqi"
  [(set (cc0)
        (compare (match_operand:QI 0 "nonimmediate_operand" "r,r,r,Q>,Q>,r,m,m")
                 (match_operand:QI 1 "general_operand"      "r,Q>,M,M,r,g,M,r")))]
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "@
   cmp.b %1,%0
   cmp.b %1,%0
   test.b %0
   test.b %0
   cmp.b %0,%1
   cmp.b %R1,%0
   test.b %R0
   cmp.b %R0,%1"
  [(set_attr "slottable" "yes,yes,yes,yes,yes,no,no,no")]) ;; no quick for this one

;
; Pattern that use the btst insn.
; It is useful for "if (i & val)" constructs, where val is an exact
; power of 2, or if val+1 is a power of two (i.e. we check for a bunch
; of ones starting at bit 0).
;
; SI-mode (will cover all other modes since gcc automatically extends subregs)
(define_insn ""
  [(set (cc0)
        (zero_extract (match_operand:SI 0 "nonmemory_operand" "r,r,r,r,r,r,n")
                      (match_operand:SI 1 "const_int_operand" "K,n,K,n,K,n,n")
                      (match_operand:SI 2 "nonmemory_operand" "M,M,K,n,r,r,r")))]
  ;; Either it is a single bit, or consecutive ones starting at 0.
  "GET_CODE(operands[1]) == CONST_INT
   && (operands[1] == const1_rtx || operands[2] == const0_rtx)
   && (REG_S_P(operands[0])
       || (operands[1] == const1_rtx && REG_S_P(operands[2])
           && GET_CODE(operands[0]) == CONST_INT
           && exact_log2(INTVAL(operands[0])) >= 0))"
;
; The latter "&&" variant above should be caught by some kind of
; canonicalization, but we can easily do it.  It results from
; expressions of the type: 
; power_of_2_value & (1 << y)
;
; Since there may be (buggy) tests in C on bits (in constant position)
; beyond the size of a word, handle that by assuming those bits are 0. 
;
  "@
   btstq (%1-1),%0
   test.d %0
   btstq %2,%0
   clearf nz
   btst %2,%0
   clearf nz
   cmpq %p0,%2"
 [(set_attr "slottable" "yes")])

;
; Move insns.
;

; No pattern for movdi is needed, gcc can split into movsi:s by
; itself.  

;
; Side-effect patterns for move.S1 [rx=ry+rx.S2],rw
; and move.S1 [rx=ry+i],rz
;  Then movs.S1 and movu.S1 for both modes.
;
;
; Move.S1 [rx=ry+rz.S],rw avoiding when rx is ry, or rw is rx
;
; QI-mode
(define_insn ""
  [(set (match_operand:QI 0 "register_operand" "=r,r")
        (mem:QI (plus:SI (mult:SI (match_operand:SI 1 "register_operand" "r,r")
                                  (match_operand:SI 2 "const_int_operand" "n,n"))
                         (match_operand:SI 3 "register_operand" "r,r"))))
   (set (match_operand:SI 4 "register_operand" "=3,r")
        (plus:SI (mult:SI (match_dup 1)
                          (match_dup 2))
                 (match_dup 3)))]

  "side_effect_mode_ok(MULT,operands,4,3,1,2,0)"

  "@
   #
   move.%s0 [%4=%3+%1%T2],%0")

; HI-mode
(define_insn ""
  [(set (match_operand:HI 0 "register_operand" "=r,r")
        (mem:HI (plus:SI (mult:SI (match_operand:SI 1 "register_operand" "r,r")
                                  (match_operand:SI 2 "const_int_operand" "n,n"))
                         (match_operand:SI 3 "register_operand" "r,r"))))
   (set (match_operand:SI 4 "register_operand" "=3,r")
        (plus:SI (mult:SI (match_dup 1)
                          (match_dup 2))
                 (match_dup 3)))]

  "side_effect_mode_ok(MULT,operands,4,3,1,2,0)"

  "@
   #
   move.%s0 [%4=%3+%1%T2],%0")

; SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r")
        (mem:SI (plus:SI (mult:SI (match_operand:SI 1 "register_operand" "r,r")
                                  (match_operand:SI 2 "const_int_operand" "n,n"))
                         (match_operand:SI 3 "register_operand" "r,r"))))
   (set (match_operand:SI 4 "register_operand" "=3,r")
        (plus:SI (mult:SI (match_dup 1)
                          (match_dup 2))
                 (match_dup 3)))]

  "side_effect_mode_ok(MULT,operands,4,3,1,2,0)"

  "@
   #
   move.%s0 [%4=%3+%1%T2],%0")

;
; move.S1 [rx=ry+i],rz avoiding when rx is ry, or rz is rx
; (and i may be a register too)
;
; QI-mode
(define_insn ""
  [(set (match_operand:QI 0 "register_operand" "=r,r,r")
        (mem:QI (plus:SI (match_operand:SI 1 "bdap_operand" "%r,r,r")
                         (match_operand:SI 2 "bdap_operand" "r>Ri,r,>Ri"))))
   (set (match_operand:SI 3 "register_operand" "=1,r,r")
        (plus:SI (match_dup 1)
                 (match_dup 2)))]

  "side_effect_mode_ok(PLUS,operands,3,1,2,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[2]) != CONST_INT
             || INTVAL(operands[2]) > 127
             || INTVAL(operands[2]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'J')))
        return \"#\";
     return \"move.%s0 [%3=%1%S2],%0\";")

; HI-mode
(define_insn ""
  [(set (match_operand:HI 0 "register_operand" "=r,r,r")
        (mem:HI (plus:SI (match_operand:SI 1 "bdap_operand" "%r,r,r")
                         (match_operand:SI 2 "bdap_operand" "r>Ri,r,>Ri"))))
   (set (match_operand:SI 3 "register_operand" "=1,r,r")
        (plus:SI (match_dup 1)
                 (match_dup 2)))]

  "side_effect_mode_ok(PLUS,operands,3,1,2,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[2]) != CONST_INT
             || INTVAL(operands[2]) > 127
             || INTVAL(operands[2]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'J')))
        return \"#\";
     return \"move.%s0 [%3=%1%S2],%0\";")

; SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r,r")
        (mem:SI (plus:SI (match_operand:SI 1 "bdap_operand" "%r,r,r")
                         (match_operand:SI 2 "bdap_operand" "r>Ri,r,>Ri"))))
   (set (match_operand:SI 3 "register_operand" "=1,r,r")
        (plus:SI (match_dup 1)
                 (match_dup 2)))]
  
  "side_effect_mode_ok(PLUS,operands,3,1,2,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[2]) != CONST_INT
             || INTVAL(operands[2]) > 127
             || INTVAL(operands[2]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'J')))
        return \"#\";
     return \"move.%s0 [%3=%1%S2],%0\";")

;
; Other way round; to memory.
;
; move.s rx,[ry=rx+rw.S]
;
; QI-mode
(define_insn ""
  [(set (mem:QI (plus:SI (mult:SI (match_operand:SI 0 "register_operand" "r,r")
                                  (match_operand:SI 1 "const_int_operand" "n,n"))
                         (match_operand:SI 2 "register_operand" "r,r")))
        (match_operand:QI 3 "register_operand" "r,r"))
   (set (match_operand:SI 4 "register_operand" "=2,r")
        (plus:SI (mult:SI (match_dup 0)
                          (match_dup 1))
                 (match_dup 2)))]

  "side_effect_mode_ok(MULT,operands,4,2,0,1,3)"

  "@
   #
   move.%s3 %3,[%4=%2+%0%T1]")

; HI-mode
(define_insn ""
  [(set (mem:HI (plus:SI (mult:SI (match_operand:SI 0 "register_operand" "r,r")
                                  (match_operand:SI 1 "const_int_operand" "n,n"))
                         (match_operand:SI 2 "register_operand" "r,r")))
        (match_operand:HI 3 "register_operand" "r,r"))
   (set (match_operand:SI 4 "register_operand" "=2,r")
        (plus:SI (mult:SI (match_dup 0)
                          (match_dup 1))
                 (match_dup 2)))]

  "side_effect_mode_ok(MULT,operands,4,2,0,1,3)"

  "@
   #
   move.%s3 %3,[%4=%2+%0%T1]")

; SI-mode
(define_insn ""
  [(set (mem:SI (plus:SI (mult:SI (match_operand:SI 0 "register_operand" "r,r")
                                  (match_operand:SI 1 "const_int_operand" "n,n"))
                         (match_operand:SI 2 "register_operand" "r,r")))
        (match_operand:SI 3 "register_operand" "r,r"))
   (set (match_operand:SI 4 "register_operand" "=2,r")
        (plus:SI (mult:SI (match_dup 0)
                          (match_dup 1))
                 (match_dup 2)))]

  "side_effect_mode_ok(MULT,operands,4,2,0,1,3)"

  "@
   #
   move.%s3 %3,[%4=%2+%0%T1]")

;
; move.s rx,[ry=rz+i]
;
; QI-mode
(define_insn ""
  [(set (mem:QI (plus:SI (match_operand:SI 0 "bdap_operand" "%r,r,r")
                         (match_operand:SI 1 "bdap_operand" "r>Ri,r,>Ri")))
        (match_operand:QI 2 "register_operand" "r,r,r"))
   (set (match_operand:SI 3 "register_operand" "=0,r,r")
        (plus:SI (match_dup 0)
                 (match_dup 1)))]

  "side_effect_mode_ok(PLUS,operands,3,0,1,-1,2)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[1]) != CONST_INT
             || INTVAL(operands[1]) > 127
             || INTVAL(operands[1]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[1]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[1]),'J')))
        return \"#\";
     return \"move.%s2 %2,[%3=%0%S1]\";")

; HI-mode
(define_insn ""
  [(set (mem:HI (plus:SI (match_operand:SI 0 "bdap_operand" "%r,r,r")
                         (match_operand:SI 1 "bdap_operand" "r>Ri,r,>Ri")))
        (match_operand:HI 2 "register_operand" "r,r,r"))
   (set (match_operand:SI 3 "register_operand" "=0,r,r")
        (plus:SI (match_dup 0)
                 (match_dup 1)))]

  "side_effect_mode_ok(PLUS,operands,3,0,1,-1,2)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[1]) != CONST_INT
             || INTVAL(operands[1]) > 127
             || INTVAL(operands[1]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[1]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[1]),'J')))
        return \"#\";
     return \"move.%s2 %2,[%3=%0%S1]\";")

; SI-mode
(define_insn ""
  [(set (mem:SI (plus:SI (match_operand:SI 0 "bdap_operand" "%r,r,r")
                         (match_operand:SI 1 "bdap_operand" "r>Ri,r,>Ri")))
        (match_operand:SI 2 "register_operand" "r,r,r"))
   (set (match_operand:SI 3 "register_operand" "=0,r,r")
        (plus:SI (match_dup 0)
                 (match_dup 1)))]

  "side_effect_mode_ok(PLUS,operands,3,0,1,-1,2)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[1]) != CONST_INT
             || INTVAL(operands[1]) > 127
             || INTVAL(operands[1]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[1]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[1]),'J')))
        return \"#\";
     return \"move.%s2 %2,[%3=%0%S1]\";")


;
; Clear patterns with sideeffect to memory.  It is hard to share this
; pattern among modes, so there will be one for each mode.

;
;  clear.d [ry=rx+rw.s2]
;
(define_insn ""
  [(set (mem:SI (plus:SI (mult:SI (match_operand:SI 0 "register_operand" "r,r")
                                  (match_operand:SI 1 "const_int_operand" "n,n"))
                      (match_operand:SI 2 "register_operand" "r,r")))
        (const_int 0))
   (set (match_operand:SI 3 "register_operand" "=2,r")
        (plus:SI (mult:SI (match_dup 0)
                          (match_dup 1))
                 (match_dup 2)))]

  "side_effect_mode_ok(MULT,operands,3,2,0,1,-1)"

  "@
   #
   clear.d [%3=%2+%0%T1]")
;
; clear.d [ry=rz+i]
;
(define_insn ""
  [(set (mem:SI (plus:SI (match_operand:SI 0 "bdap_operand" "%r,r,r")
                         (match_operand:SI 1 "bdap_operand" "r>Ri,r,>Ri")))
        (const_int 0))
   (set (match_operand:SI 2 "register_operand" "=0,r,r")
        (plus:SI (match_dup 0)
                 (match_dup 1)))]
  
  "side_effect_mode_ok(PLUS,operands,2,0,1,-1,-1)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[1]) != CONST_INT
             || INTVAL(operands[1]) > 127
             || INTVAL(operands[1]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[1]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[1]),'J')))
        return \"#\";
     return \"clear.d [%2=%0%S1]\";")

;
;  clear.w [ry=rx+rw.s2]
;
(define_insn ""
  [(set (mem:HI (plus:SI (mult:SI (match_operand:SI 0 "register_operand" "r,r")
                                  (match_operand:SI 1 "const_int_operand" "n,n"))
                         (match_operand:SI 2 "register_operand" "r,r")))
        (const_int 0))
   (set (match_operand:SI 3 "register_operand" "=2,r")
        (plus:SI (mult:SI (match_dup 0)
                          (match_dup 1))
                 (match_dup 2)))]

  "side_effect_mode_ok(MULT,operands,3,2,0,1,-1)"

  "@
   #
   clear.w [%3=%2+%0%T1]")

;
; clear.w [ry=rz+i]
;
(define_insn ""
  [(set (mem:HI (plus:SI (match_operand:SI 0 "bdap_operand" "%r,r,r")
                         (match_operand:SI 1 "bdap_operand" "r>Ri,r,>Ri")))
        (const_int 0))
   (set (match_operand:SI 2 "register_operand" "=0,r,r")
        (plus:SI (match_dup 0)
                 (match_dup 1)))]

  "side_effect_mode_ok(PLUS,operands,2,0,1,-1,-1)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[1]) != CONST_INT
             || INTVAL(operands[1]) > 127
             || INTVAL(operands[1]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[1]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[1]),'J')))
        return \"#\";
     return \"clear.w [%2=%0%S1]\";")

;
;  clear.b [ry=rx+rw.s2]
;
(define_insn ""
  [(set (mem:QI (plus:SI (mult:SI (match_operand:SI 0 "register_operand" "r,r")
                                  (match_operand:SI 1 "const_int_operand" "n,n"))
                         (match_operand:SI 2 "register_operand" "r,r")))
        (const_int 0))
   (set (match_operand:SI 3 "register_operand" "=2,r")
        (plus:SI (mult:SI (match_dup 0)
                          (match_dup 1))
                 (match_dup 2)))]

  "side_effect_mode_ok(MULT,operands,3,2,0,1,-1)"

  "@
   #
   clear.b [%3=%2+%0%T1]")

;
; clear.b [ry=rz+i]
;
(define_insn ""
  [(set (mem:QI (plus:SI (match_operand:SI 0 "bdap_operand" "%r,r,r")
                         (match_operand:SI 1 "bdap_operand" "r>Ri,r,>Ri")))
        (const_int 0))
   (set (match_operand:SI 2 "register_operand" "=0,r,r")
        (plus:SI (match_dup 0)
                 (match_dup 1)))]

  "side_effect_mode_ok(PLUS,operands,2,0,1,-1,-1)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[1]) != CONST_INT
             || INTVAL(operands[1]) > 127
             || INTVAL(operands[1]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[1]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[1]),'J')))
        return \"#\";
     return \"clear.b [%2=%0%S1]\";")

;
; Normal move patterns from SI on.
;

(define_insn "movsi"
  [(set
    (match_operand:SI 0 "general_operand"  "=r,r,r,Q>,r,Q>,g,r,r,g")
    (match_operand:SI 1 "general_operand"    "r,Q>,M,M,I,r,M,n,g,r"))]
  ""
  "*
{
  extern rtx same_ref_as_saved();

  /* Better to have c-switch here; must optimize size of move insn.
     The alternative is to try to find more constraint letters... */
  switch(which_alternative) {

  case 8:
    /* Take care of reducing effect of unoptimized address cse.
       This call takes care of saving for move const+int,r
       where const is a symbol or label.
     */
    operands[1] = same_ref_as_saved(operands[0],operands[1]);

  case 9: case 0: case 1: case 5:
    return \"move.d %R1,%R0\";
  case 2: case 3: case 6:
    return \"clear.d %R0\";

    /* Constants -32..31 except 0. */
  case 4:
    return \"moveq %1,%0\";

    /* We can win a bit on constants -32768..-33,32..65535 */
  case 7:
    if (INTVAL(operands[1]) > 0 && INTVAL(operands[1]) < 65536) {
      if (INTVAL(operands[1]) < 256)
        return \"movu.b %1,%0\";
      return \"movu.w %1,%0\";
    }
    if (INTVAL(operands[1]) >= -32768 && INTVAL(operands[1]) < 32768) {
      if (INTVAL(operands[1]) >= -128 && INTVAL(operands[1]) < 128)
        return \"movs.b %1,%0\";
      return \"movs.w %1,%0\";
    }
    return \"move.d %1,%0\";

  default:
    return \"BOGUS: %1 -> %0\";
  }
}"
  [(set_attr "slottable" "yes,yes,yes,yes,yes,yes,no,no,no,no")])

;
; Extend operations with side-effect from mem to register
;
; movs/movu (always from mem to register)
;
; [rx=ry+rz.S]
;
; QI-mode -> HI-mode
(define_insn ""
  [(set (match_operand:HI 0 "register_operand" "=r,r")
        (match_operator:HI
         5 "extend_operator"
         [(mem:QI (plus:SI (mult:SI (match_operand:SI 1 "register_operand" "r,r")
                                    (match_operand:SI 2 "const_int_operand" "n,n"))
                           (match_operand:SI 3 "register_operand" "r,r")))]))
   (set (match_operand:SI 4 "register_operand" "=3,r")
        (plus:SI (mult:SI (match_dup 1)
                          (match_dup 2))
                 (match_dup 3)))]

  "side_effect_mode_ok(MULT,operands,4,3,1,2,0)"

  "@
   #
   mov%e5.%m5 [%4=%3+%1%T2],%0")

; QI-mode -> SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r")
        (match_operator:SI
         5 "extend_operator"
         [(mem:QI (plus:SI (mult:SI (match_operand:SI 1 "register_operand" "r,r")
                                    (match_operand:SI 2 "const_int_operand" "n,n"))
                           (match_operand:SI 3 "register_operand" "r,r")))]))
   (set (match_operand:SI 4 "register_operand" "=3,r")
        (plus:SI (mult:SI (match_dup 1)
                          (match_dup 2))
                 (match_dup 3)))]

  "side_effect_mode_ok(MULT,operands,4,3,1,2,0)"

  "@
   #
   mov%e5.%m5 [%4=%3+%1%T2],%0")

; HI-mode -> SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r")
        (match_operator:SI
         5 "extend_operator"
         [(mem:HI (plus:SI (mult:SI (match_operand:SI 1 "register_operand" "r,r")
                                    (match_operand:SI 2 "const_int_operand" "n,n"))
                           (match_operand:SI 3 "register_operand" "r,r")))]))
   (set (match_operand:SI 4 "register_operand" "=3,r")
        (plus:SI (mult:SI (match_dup 1)
                          (match_dup 2))
                 (match_dup 3)))]

  "side_effect_mode_ok(MULT,operands,4,3,1,2,0)"

  "@
   #
   mov%e5.%m5 [%4=%3+%1%T2],%0")

;
; [rx=ry+i]
;
; QI-mode -> HI-mode
(define_insn ""
  [(set (match_operand:HI 0 "register_operand" "=r,r,r")
        (match_operator:HI
         4 "extend_operator"
         [(mem:QI (plus:SI (match_operand:SI 1 "bdap_operand" "%r,r,r")
                           (match_operand:SI 2 "bdap_operand" "r>Ri,r,>Ri")))]))
   (set (match_operand:SI 3 "register_operand" "=1,r,r")
        (plus:SI (match_dup 1)
                 (match_dup 2)))]

  "side_effect_mode_ok(PLUS,operands,3,1,2,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[2]) != CONST_INT
             || INTVAL(operands[2]) > 127
             || INTVAL(operands[2]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'J')))
        return \"#\";
     return \"mov%e4.%m4 [%3=%1%S2],%0\";")

; QI-mode -> SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r,r")
        (match_operator:SI
         4 "extend_operator"
         [(mem:QI (plus:SI (match_operand:SI 1 "bdap_operand" "%r,r,r")
                           (match_operand:SI 2 "bdap_operand" "r>Ri,r,>Ri")))]))
   (set (match_operand:SI 3 "register_operand" "=1,r,r")
        (plus:SI (match_dup 1)
                 (match_dup 2)))]

  "side_effect_mode_ok(PLUS,operands,3,1,2,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[2]) != CONST_INT
             || INTVAL(operands[2]) > 127
             || INTVAL(operands[2]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'J')))
        return \"#\";
     return \"mov%e4.%m4 [%3=%1%S2],%0\";")

; HI-mode -> SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r,r")
        (match_operator:SI
         4 "extend_operator"
         [(mem:HI (plus:SI (match_operand:SI 1 "bdap_operand" "%r,r,r")
                           (match_operand:SI 2 "bdap_operand" "r>Ri,r,>Ri")))]))
   (set (match_operand:SI 3 "register_operand" "=1,r,r")
        (plus:SI (match_dup 1)
                 (match_dup 2)))]

  "side_effect_mode_ok(PLUS,operands,3,1,2,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[2]) != CONST_INT
             || INTVAL(operands[2]) > 127
             || INTVAL(operands[2]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'J')))
        return \"#\";
     return \"mov%e4.%m4 [%3=%1%S2],%0\";")

(define_insn "movhi"
  [(set
    (match_operand:HI 0 "general_operand" "=r,r,r,Q>,r,Q>,r,r,r,g,g,r")
    (match_operand:HI 1 "general_operand" "r,Q>,M,M,I,r,L,O,n,M,r,g"))]
;                                    'L' before 'O' to catch 255-31..255
  ""
  "*
{
 switch(which_alternative) {
  case 0: case 1: case 5: case 10: case 11:
    return \"move.w %R1,%R0\";
  case 2: case 3: case 9:
    return \"clear.w %R0\";
  case 4:
    return \"moveq %1,%0\";
  case 7:
    return \"movEq %b1,%0\";
  case 8: case 6:
    if (INTVAL(operands[1]) < 256 && INTVAL(operands[1]) >= -128) {
      if (INTVAL(operands[1]) > 0)
        return \"movu.b %1,%0\";
      return \"movs.b %1,%0\";
    }
    return \"move.w %1,%0\";
  default:
    return \"BOGUS: %1 -> %0\";
  }
}"
  [(set_attr "slottable" "yes,yes,yes,yes,yes,yes,no,yes,no,no,no,no")])

(define_insn "movstricthi"
  [(set
    (strict_low_part
     (match_operand:HI 0 "general_operand" "+r,r,r,Q>,Q>,g,r,g"))
    (match_operand:HI 1 "general_operand" "r,Q>,M,M,r,M,g,r"))]
  ""
  "@
   move.w %1,%0
   move.w %1,%0
   clear.w %0
   clear.w %0
   move.w %1,%0
   clear.w %R0
   move.w %R1,%0
   move.w %1,%R0"
  [(set_attr "slottable" "yes,yes,yes,yes,yes,no,no,no")])

(define_insn "movqi"
  [(set (match_operand:QI 0 "general_operand" "=r,Q>,r,r,Q>,r,g,g,r,r")
        (match_operand:QI 1 "general_operand" "r,r,Q>,M,M,I,M,r,O,g"))]
  ""
  "@
   move.b %1,%0
   move.b %1,%0
   move.b %1,%0
   clear.b %0
   clear.b %0
   moveq %1,%0
   clear.b %R0
   move.b %1,%R0
   moveq %b1,%0
   move.b %R1,%0"
  [(set_attr "slottable" "yes,yes,yes,yes,yes,yes,no,no,yes,no")])

(define_insn "movstrictqi"
  [(set (strict_low_part
         (match_operand:QI 0 "general_operand" "+r,Q>,r,r,Q>,g,g,r"))
        (match_operand:QI 1 "general_operand"   "r,r,Q>,M,M,M,r,g"))]
  ""
  "@
   move.b %1,%0
   move.b %1,%0
   move.b %1,%0
   clear.b %0
   clear.b %0
   clear.b %R0
   move.b %1,%R0
   move.b %R1,%0"
  [(set_attr "slottable" "yes,yes,yes,yes,yes,no,no,no")])

;
; The valid "quick" bit-patterns are, except for 0.0, values REALLY close
; to 0, and some NaN:s (I think; their exponent is all ones).
;  This means I will only include 0.0, which is fairly common...
; It will use clear, so we know ALL types of immediate 0 never change cc.
;
(define_insn "movsf"
  [(set (match_operand:SF 0 "general_operand" "=r,Q>,r,r,Q>,g,g,r")
        (match_operand:SF 1 "general_operand"   "r,r,Q>,G,G,G,r,g"))]
  ""
  "@
   move.d %1,%0
   move.d %1,%0
   move.d %1,%0
   clear.d %0
   clear.d %0
   clear.d %R0
   move.d %1,%R0
   move.d %R1,%0"
  [(set_attr "slottable" "yes,yes,yes,yes,yes,no,no,no")])

;
; Unnecessary to call bcopy for small quantities; expand movstr into
; something usable instead.
;
; The (mem:BLK (match_operand:BLK ...)) seems strange, but results
; from the necessity to both hold a matching (mem:BLK (expr)) match,
; and a generating (match_operand:BLK (expr)).  This is the only way
; to win.
;

(define_expand "movstrqi"
  [(parallel
    [(set (mem:BLK (match_operand:SI 0 "general_operand" ""))
          (mem:BLK (match_operand:SI 1 "general_operand" "")))
     (clobber (match_dup 0))
     (clobber (match_dup 1))
     (clobber (match_scratch:SI 4 "=&r"))
     (use (match_operand 2 "const_int_operand" "n"))
     (use (match_operand 3 "const_int_operand" "n"))])]
  ""
  "{
     if (GET_CODE(operands[2]) != CONST_INT
         || INTVAL(operands[2]) > 4*8
         || GET_CODE(operands[1]) != MEM
         || GET_CODE(operands[0]) != MEM)
       FAIL;

     operands[0] = copy_addr_to_reg(XEXP(operands[0],0));
     operands[1] = copy_addr_to_reg(XEXP(operands[1],0));
}")

(define_expand "movstrhi"
  [(parallel
    [(set (mem:BLK (match_operand:SI 0 "general_operand" ""))
          (mem:BLK (match_operand:SI 1 "general_operand" "")))
     (clobber (match_dup 0))
     (clobber (match_dup 1))
     (clobber (match_scratch:SI 4 "=&r"))
     (use (match_operand 2 "const_int_operand" "n"))
     (use (match_operand 3 "const_int_operand" "n"))])]
  ""
  "{
     if (GET_CODE(operands[2]) != CONST_INT
         || INTVAL(operands[2]) > 4*8
         || GET_CODE(operands[1]) != MEM
         || GET_CODE(operands[0]) != MEM)
       FAIL;

     operands[0] = copy_addr_to_reg(XEXP(operands[0],0));
     operands[1] = copy_addr_to_reg(XEXP(operands[1],0));
}")

(define_expand "movstrsi"
  [(parallel
    [(set (mem:BLK (match_operand:SI 0 "general_operand" ""))
          (mem:BLK (match_operand:SI 1 "general_operand" "")))
     (clobber (match_dup 0))
     (clobber (match_dup 1))
     (clobber (match_scratch:SI 4 "=&r"))
     (use (match_operand 2 "const_int_operand" "n"))
     (use (match_operand 3 "const_int_operand" "n"))])]
  ""
  "{
     if (GET_CODE(operands[2]) != CONST_INT
         || INTVAL(operands[2]) > 4*8
         || GET_CODE(operands[1]) != MEM
         || GET_CODE(operands[0]) != MEM)
       FAIL;

     operands[0] = copy_addr_to_reg(XEXP(operands[0],0));
     operands[1] = copy_addr_to_reg(XEXP(operands[1],0));
}")

; This is the actual insn for the block move.

(define_insn ""
  [(set (mem:BLK (match_operand:SI 0 "register_operand" "r"))
        (mem:BLK (match_operand:SI 1 "register_operand" "r")))
   (clobber (match_dup 0))
   (clobber (match_dup 1))
   (clobber (match_scratch:SI 4 "=&r"))
   (use (match_operand 2 "const_int_operand" "n"))
   (use (match_operand 3 "const_int_operand" "n"))]

  "GET_CODE(operands[2]) == CONST_INT && INTVAL(operands[2]) <= 4*8"

  "* output_block_move(operands); return \"\"; ")

;
;; If there had been a way to say that operands 0 and 1 (and 4) WILL
;; have defined values, then we should definitely had done it.

;
; Extension and truncation insns.
; Those for integer source operand
; are ordered widest source type first.

(define_insn "extendsidi2"
  [(set (match_operand:DI 0 "register_operand" "=r")
        (sign_extend:DI (match_operand:SI 1 "general_operand" "g")))]
  ""
  "move.d %R1,%M0\;bpl 0f\;clear.d %H0\;moveq -1,%H0\\n0:")

(define_insn "extendhidi2"
  [(set (match_operand:DI 0 "register_operand" "=r")
        (sign_extend:DI (match_operand:HI 1 "general_operand" "g")))]
  ""
  "movs.w %R1,%M0\;bpl 0f\;clear.d %H0\;moveq -1,%H0\\n0:")

(define_insn "extendhisi2"
  [(set (match_operand:SI 0 "register_operand" "=r,r,r")
        (sign_extend:SI (match_operand:HI 1 "general_operand" "r,Q>,g")))]
  ""
  "movs.w %R1,%0"
  [(set_attr "slottable" "yes,yes,no")])

(define_insn "extendqidi2"
  [(set (match_operand:DI 0 "register_operand" "=r")
        (sign_extend:DI (match_operand:QI 1 "general_operand" "g")))]
  ""
  "movs.b %R1,%M0\;bpl 0f\;clear.d %H0\;moveq -1,%H0\\n0:")

(define_insn "extendqisi2"
  [(set (match_operand:SI 0 "register_operand" "=r,r,r")
        (sign_extend:SI (match_operand:QI 1 "general_operand" "r,Q>,g")))]
  ""
  "movs.b %R1,%0"
  [(set_attr "slottable" "yes,yes,no")])

; To do a byte->word exension, extend to dword, exept that the top half
; of the register will be clobbered

(define_insn "extendqihi2"
  [(set (match_operand:HI 0 "register_operand" "=r,r,r")
        (sign_extend:HI (match_operand:QI 1 "general_operand" "r,Q>,g")))]
  ""
  "movs.b %R1,%0"
  [(set_attr "slottable" "yes,yes,no")])

;
; Truncate.
;
(define_insn "zero_extendhisi2"
  [(set (match_operand:SI 0 "register_operand" "=r,r,r")
        (zero_extend:SI (match_operand:HI 1 "nonimmediate_operand" "r,Q>,m")))]
  ""
  "movu.w %R1,%0"
  [(set_attr "slottable" "yes,yes,no")])

(define_insn "zero_extendqisi2"
  [(set (match_operand:SI 0 "register_operand" "=r,r,r")
        (zero_extend:SI (match_operand:QI 1 "nonimmediate_operand" "r,Q>,m")))]
  ""
  "movu.b %R1,%0"
  [(set_attr "slottable" "yes,yes,no")])

; Same comment as sign extend byte->word above applies.

(define_insn "zero_extendqihi2"
  [(set (match_operand:HI 0 "register_operand" "=r,r,r")
        (zero_extend:HI (match_operand:QI 1 "nonimmediate_operand" "r,Q>,m")))]
  ""
  "movu.b %R1,%0"
  [(set_attr "slottable" "yes,yes,no")])


;- All kinds of add instructions.
;
; First anonymous patterns to match side-effect addressing modes.
;
; op.S [rx=ry+i],rz; includes add,sub,or,and.
;
; [rx=ry+rz.S]
;
; QI-mode
(define_insn ""
  [(set (match_operand:QI 0 "register_operand" "=r,r")
        (match_operator:QI
         6 "orthogonal_operator"
         [(match_operand:QI 1 "register_operand" "0,0")
          (mem:QI (plus:SI (mult:SI (match_operand:SI 2 "register_operand" "r,r")
                                    (match_operand:SI 3 "const_int_operand" "n,n"))
                           (match_operand:SI 4 "register_operand" "r,r")))]))
   (set (match_operand:SI 5 "register_operand" "=4,r")
        (plus:SI (mult:SI (match_dup 2)
                          (match_dup 3))
                 (match_dup 4)))]

  "side_effect_mode_ok(MULT,operands,5,4,2,3,0)"

  "@
   #
   %x6.%s0 [%5=%4+%2%T3],%0")

; HI-mode
(define_insn ""
  [(set (match_operand:HI 0 "register_operand" "=r,r")
        (match_operator:HI
         6 "orthogonal_operator"
         [(match_operand:HI 1 "register_operand" "0,0")
          (mem:HI (plus:SI (mult:SI (match_operand:SI 2 "register_operand" "r,r")
                                    (match_operand:SI 3 "const_int_operand" "n,n"))
                           (match_operand:SI 4 "register_operand" "r,r")))]))
   (set (match_operand:SI 5 "register_operand" "=4,r")
        (plus:SI (mult:SI (match_dup 2)
                          (match_dup 3))
                 (match_dup 4)))]

  "side_effect_mode_ok(MULT,operands,5,4,2,3,0)"

  "@
   #
   %x6.%s0 [%5=%4+%2%T3],%0")

; SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r")
        (match_operator:SI
         6 "orthogonal_operator"
         [(match_operand:SI 1 "register_operand" "0,0")
          (mem:SI (plus:SI (mult:SI (match_operand:SI 2 "register_operand" "r,r")
                                    (match_operand:SI 3 "const_int_operand" "n,n"))
                           (match_operand:SI 4 "register_operand" "r,r")))]))
   (set (match_operand:SI 5 "register_operand" "=4,r")
        (plus:SI (mult:SI (match_dup 2)
                          (match_dup 3))
                 (match_dup 4)))]

  "side_effect_mode_ok(MULT,operands,5,4,2,3,0)"

  "@
   #
   %x6.%s0 [%5=%4+%2%T3],%0")
;
; %5   [%4=%2+%3],%0   (1 is 0)
;
; QI-mode
(define_insn ""
  [(set (match_operand:QI 0 "register_operand" "=r,r,r")
        (match_operator:QI
         5 "orthogonal_operator"
         [(match_operand:QI 1 "register_operand" "0,0,0")
          (mem:QI (plus:SI (match_operand:SI 2 "bdap_operand" "%r,r,r")
                           (match_operand:SI 3 "bdap_operand" "r>Ri,r,>Ri")))]))
   (set (match_operand:SI 4 "register_operand" "=2,r,r")
        (plus:SI (match_dup 2)
                 (match_dup 3)))]

  "side_effect_mode_ok(PLUS,operands,4,2,3,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[3]) != CONST_INT
             || INTVAL(operands[3]) > 127
             || INTVAL(operands[3]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'J')))
        return \"#\";
     return \"%x5.%s0 [%4=%2%S3],%0\";")

; HI-mode
(define_insn ""
  [(set (match_operand:HI 0 "register_operand" "=r,r,r")
        (match_operator:HI
         5 "orthogonal_operator"
         [(match_operand:HI 1 "register_operand" "0,0,0")
          (mem:HI (plus:SI (match_operand:SI 2 "bdap_operand" "%r,r,r")
                           (match_operand:SI 3 "bdap_operand" "r>Ri,r,>Ri")))]))
   (set (match_operand:SI 4 "register_operand" "=2,r,r")
        (plus:SI (match_dup 2)
                 (match_dup 3)))]

  "side_effect_mode_ok(PLUS,operands,4,2,3,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[3]) != CONST_INT
             || INTVAL(operands[3]) > 127
             || INTVAL(operands[3]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'J')))
        return \"#\";
     return \"%x5.%s0 [%4=%2%S3],%0\";")

; SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r,r")
        (match_operator:SI
         5 "orthogonal_operator"
         [(match_operand:SI 1 "register_operand" "0,0,0")
          (mem:SI (plus:SI (match_operand:SI 2 "bdap_operand" "%r,r,r")
                           (match_operand:SI 3 "bdap_operand" "r>Ri,r,>Ri")))]))
   (set (match_operand:SI 4 "register_operand" "=2,r,r")
        (plus:SI (match_dup 2)
                 (match_dup 3)))]

  "side_effect_mode_ok(PLUS,operands,4,2,3,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[3]) != CONST_INT
             || INTVAL(operands[3]) > 127
             || INTVAL(operands[3]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'J')))
        return \"#\";
     return \"%x5.%s0 [%4=%2%S3],%0\";")
;
; To be able to match commutative operands we may have to have the
; following pattern for add, or & and.  I dont know really, but it
; doesnt break anything.
;
; op.S [rx=ry+i],rz;
;
; [rx=ry+rz.S]
;
; QI-mode
(define_insn ""
  [(set (match_operand:QI 0 "register_operand" "=r,r")
        (match_operator:QI
         6 "commutative_orth_op"
         [(mem:QI (plus:SI (mult:SI (match_operand:SI 2 "register_operand" "r,r")
                                 (match_operand:SI 3 "const_int_operand" "n,n"))
                        (match_operand:SI 4 "register_operand" "r,r")))
          (match_operand:QI 1 "register_operand" "0,0")]))
   (set (match_operand:SI 5 "register_operand" "=4,r")
        (plus:SI (mult:SI (match_dup 2)
                          (match_dup 3))
                 (match_dup 4)))]

  "side_effect_mode_ok(MULT,operands,5,4,2,3,0)"

  "@
   #
   %x6.%s0 [%5=%4+%2%T3],%0")

; HI-mode
(define_insn ""
  [(set (match_operand:HI 0 "register_operand" "=r,r")
        (match_operator:HI
         6 "commutative_orth_op"
         [(mem:HI (plus:SI (mult:SI (match_operand:SI 2 "register_operand" "r,r")
                                    (match_operand:SI 3 "const_int_operand" "n,n"))
                           (match_operand:SI 4 "register_operand" "r,r")))
          (match_operand:HI 1 "register_operand" "0,0")]))
   (set (match_operand:SI 5 "register_operand" "=4,r")
        (plus:SI (mult:SI (match_dup 2)
                          (match_dup 3))
                 (match_dup 4)))]

  "side_effect_mode_ok(MULT,operands,5,4,2,3,0)"

  "@
   #
   %x6.%s0 [%5=%4+%2%T3],%0")

; SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r")
        (match_operator:SI
         6 "commutative_orth_op"
         [(mem:SI (plus:SI (mult:SI (match_operand:SI 2 "register_operand" "r,r")
                                    (match_operand:SI 3 "const_int_operand" "n,n"))
                           (match_operand:SI 4 "register_operand" "r,r")))
          (match_operand:SI 1 "register_operand" "0,0")]))
   (set (match_operand:SI 5 "register_operand" "=4,r")
        (plus:SI (mult:SI (match_dup 2)
                          (match_dup 3))
                 (match_dup 4)))]

  "side_effect_mode_ok(MULT,operands,5,4,2,3,0)"

  "@
   #
   %x6.%s0 [%5=%4+%2%T3],%0")

;
; %5   [%4=%2+%3],%0   (1 is 0)
;
; QI-mode
(define_insn ""
  [(set (match_operand:QI 0 "register_operand" "=r,r,r")
        (match_operator:QI
         5 "commutative_orth_op"
         [(mem:QI (plus:SI (match_operand:SI 2 "bdap_operand" "%r,r,r")
                           (match_operand:SI 3 "bdap_operand" "r>Ri,r,>Ri")))
          (match_operand:QI 1 "register_operand" "0,0,0")]))
   (set (match_operand:SI 4 "register_operand" "=2,r,r")
        (plus:SI (match_dup 2)
                 (match_dup 3)))]

  "side_effect_mode_ok(PLUS,operands,4,2,3,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[3]) != CONST_INT
             || INTVAL(operands[3]) > 127
             || INTVAL(operands[3]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'J')))
        return \"#\";
     return \"%x5.%s0 [%4=%2%S3],%0\";")

; HI-mode
(define_insn ""
  [(set (match_operand:HI 0 "register_operand" "=r,r,r")
        (match_operator:HI
         5 "commutative_orth_op"
         [(mem:HI (plus:SI (match_operand:SI 2 "bdap_operand" "%r,r,r")
                           (match_operand:SI 3 "bdap_operand" "r>Ri,r,>Ri")))
          (match_operand:HI 1 "register_operand" "0,0,0")]))
   (set (match_operand:SI 4 "register_operand" "=2,r,r")
        (plus:SI (match_dup 2)
                 (match_dup 3)))]

  "side_effect_mode_ok(PLUS,operands,4,2,3,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[3]) != CONST_INT
             || INTVAL(operands[3]) > 127
             || INTVAL(operands[3]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'J')))
        return \"#\";
     return \"%x5.%s0 [%4=%2%S3],%0\";")

; SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r,r")
        (match_operator:SI
         5 "commutative_orth_op"
         [(mem:SI (plus:SI (match_operand:SI 2 "bdap_operand" "%r,r,r")
                        (match_operand:SI 3 "bdap_operand" "r>Ri,r,>Ri")))
          (match_operand:SI 1 "register_operand" "0,0,0")]))
   (set (match_operand:SI 4 "register_operand" "=2,r,r")
        (plus:SI (match_dup 2)
                 (match_dup 3)))]

  "side_effect_mode_ok(PLUS,operands,4,2,3,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[3]) != CONST_INT
             || INTVAL(operands[3]) > 127
             || INTVAL(operands[3]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'J')))
        return \"#\";
     return \"%x5.%s0 [%4=%2%S3],%0\";")

;
;
; Normal add operations.
;
; No extra cse for DImode.
;
(define_insn "adddi3"
  [(set (match_operand:DI 0 "register_operand"          "=r,r,r,&r,&r")
        (plus:DI (match_operand:DI 1 "register_operand" "%0,0,0,0,r")
                 (match_operand:DI 2 "general_operand"   "J,N,P,g,!To")))]
  ""                                       ; Inhibit reloading to "To" alternative
  "@
   addq %2,%M0\;ax\;addq 0,%H0
   subq %n2,%M0\;ax\;subq 0,%H0
   add%e2.%z2 %2,%M0\;ax\;addq 0,%H0
   add.d %M2,%M0\;ax\;add.d %H2,%H0
   add.d %M2,%M1,%M0\;ax\;add.d %H2,%H1,%H0")


(define_insn "addsi3"
  [(set (match_operand:SI 0 "register_operand" "=r,r,r,r,r,r,r,r")
        (plus:SI                           ; The last constraints due to
; that after reload, the '%' is not honored, and canonicalization
; doesnt care about same register in destination.  This will happen
; after insn splitting. 
         (match_operand:SI 1 "register_operand" "%0,0,0,0,0,0,r,r")
         (match_operand:SI 2 "general_operand"  "r,Q>,J,N,n,g,!To,0")))]
 ""                                         ; Inhibit reloading to "To" alternative
 "*
{
  switch (which_alternative) {
  case 0: case 1:
    return \"add.d %2,%0\";
  case 2:
    return \"addq %2,%0\";
  case 3:
    return \"subq %n2,%0\";
  case 4: /* Known value, not in -63..63. Check if addu or subu may be used. */
    if (INTVAL(operands[2]) > 0) {
      if (INTVAL(operands[2]) < 256)
        return \"addu.b %2,%0\";
      if (INTVAL(operands[2]) < 65536)
        return \"addu.w %2,%0\";
    } else {
      if (INTVAL(operands[2]) >= -255)
        return \"subu.b %n2,%0\";
      if (INTVAL(operands[2]) >= -65535)
        return \"subu.w %n2,%0\";
    }
    return \"add.d %2,%0\";
  case 6:
    return \"add.d %R2,%1,%0\";
  case 5:
    return \"add.d %R2,%0\";
  case 7:
    return \"add.d %1,%0\";  
  default:
    return \"BOGUS addsi %2+%1 -> %0\";
  }
}"
 [(set_attr "slottable" "yes,yes,yes,yes,no,no,no,yes")])

(define_insn "addhi3"
  [(set (match_operand:HI 0 "register_operand"         "=r,r,r,r,r,r")
        (plus:HI (match_operand:HI 1 "register_operand" "%0,0,0,0,0,r")
                 (match_operand:HI 2 "general_operand"  "r,Q>,J,N,g,!To")))]
                                        ; Inhibit reloading to "To" alternative
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "@
   add.w %2,%0
   add.w %2,%0
   addq %2,%0
   subq %n2,%0
   add.w %R2,%0
   add.w %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,yes,yes,no,no")])

(define_insn "addqi3"
  [(set (match_operand:QI 0 "register_operand"          "=r,r,r,r,r,r,r")
        (plus:QI (match_operand:QI 1 "register_operand" "%0,0,0,0,0,0,r")
                 (match_operand:QI 2 "general_operand" "r,Q>,J,N,O,g,!To")))]
                                        ; Inhibit reloading to "To" alternative
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "@
   add.b %2,%0
   add.b %2,%0
   addq %2,%0
   subq %n2,%0
   subQ -%b2,%0
   add.b %R2,%0
   add.b %R2,%1,%0"   
  [(set_attr "slottable" "yes,yes,yes,yes,yes,no,no")])

;
; Subtract.
;
; Note that because of insn canonicalization these will SELDOM be used
; with a known constant as an operand.
;
;
; No extra cse for DImode.
;
(define_insn "subdi3"
  [(set (match_operand:DI 0 "register_operand"          "=r,r,r,&r,&r")
        (minus:DI (match_operand:DI 1 "register_operand" "0,0,0,0,r")
                 (match_operand:DI 2 "general_operand"   "J,N,P,g,!To")))]
                                        ; Inhibit reloading to "To" alternative
  ""
  "@
   subq %2,%M0\;ax\;subq 0,%H0
   addq %n2,%M0\;ax\;addq 0,%H0
   sub%e2.%z2 %2,%M0\;ax\;subq 0,%H0
   sub.d %M2,%M0\;ax\;sub.d %H2,%H0
   sub.d %M2,%M1,%M0\;ax\;sub.d %H2,%H1,%H0")

(define_insn "subsi3"
  [(set (match_operand:SI 0 "register_operand"          "=r,r,r,r,r,r,r,r")
        (minus:SI (match_operand:SI 1 "register_operand" "0,0,0,0,0,0,0,r")
                  (match_operand:SI 2 "general_operand" "r,Q>,J,N,P,n,g,!To")))]
                                        ; Inhibit reloading to "To" alternative
  ""
; This doesnt do optimal: "addu.w 65535,r0" when %2 is negative.  But
; it shouldnt be.
  "@
   sub.d %2,%0
   sub.d %2,%0
   subq %2,%0
   addq %n2,%0
   sub%e2.%z2 %2,%0
   sub.d %R2,%0
   sub.d %R2,%0
   sub.d %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,yes,yes,no,no,no,no")])

(define_insn "subhi3"
  [(set (match_operand:HI 0 "register_operand"          "=r,r,r,r,r,r")
        (minus:HI (match_operand:HI 1 "register_operand" "0,0,0,0,0,r")
                  (match_operand:HI 2 "general_operand" "r,Q>,J,N,g,!To")))]
                                        ; Inhibit reloading to "To" alternative
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "@
   sub.w %2,%0
   sub.w %2,%0
   subq %2,%0
   addq %n2,%0
   sub.w %R2,%0
   sub.w %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,yes,yes,no,no")])

(define_insn "subqi3"
  [(set (match_operand:QI 0 "register_operand"         "=r,r,r,r,r,r")
        (minus:QI (match_operand:QI 1 "register_operand" "0,0,0,0,0,r")
                  (match_operand:QI 2 "general_operand" "r,Q>,J,N,g,!To")))]
                                        ; Inhibit reloading to "To" alternative
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "@
   sub.b %2,%0
   sub.b %2,%0
   subq %2,%0
   addq %2,%0
   sub.b %R2,%0
   sub.b %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,yes,yes,no,no")])

;
; Cris has some add-with-sign-extend and add-with-zero-extend instructions
; Although these perform sign/zero-extension to SImode, they are 
; equally applicable for the HImode case.
;  Note that these must be located after the normal add/sub patterns,
; so not to get constants into any unspecified operands.
;
; Extend with add/sub and side-effect.
;
; adds/subs/addu/subu also bound, check for zero_extend
;
; adds/subs/addu/subu bound [rx=ry+rz.S]
;
; QI-mode -> HI-mode
(define_insn ""
  [(set (match_operand:HI 0 "register_operand" "=r,r")
        (match_operator:HI
         6 "operand_extend_operator"
         [(match_operand:HI 1 "register_operand" "0,0")
          (match_operator:HI
           7 "extend_operator"
           [(mem:QI (plus:SI (mult:SI (match_operand:SI 2 "register_operand" "r,r")
                                      (match_operand:SI 3 "const_int_operand" "n,n"))
                             (match_operand:SI 4 "register_operand" "r,r")))])]))
   (set (match_operand:SI 5 "register_operand" "=4,r")
        (plus:SI (mult:SI (match_dup 2)
                          (match_dup 3))
                 (match_dup 4)))]

  "(GET_CODE(operands[5]) != UMIN || GET_CODE(operands[7]) == ZERO_EXTEND)
   && side_effect_mode_ok(MULT,operands,5,4,2,3,0)"

  "@
   #
   %x6%e7.%m7 [%5=%4+%2%T3],%0")

; QI-mode -> SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r")
        (match_operator:SI
         6 "operand_extend_operator"
         [(match_operand:SI 1 "register_operand" "0,0")
          (match_operator:SI
           7 "extend_operator"
           [(mem:QI (plus:SI (mult:SI (match_operand:SI 2 "register_operand" "r,r")
                                      (match_operand:SI 3 "const_int_operand" "n,n"))
                             (match_operand:SI 4 "register_operand" "r,r")))])]))
   (set (match_operand:SI 5 "register_operand" "=4,r")
        (plus:SI (mult:SI (match_dup 2)
                          (match_dup 3))
                 (match_dup 4)))]

  "(GET_CODE(operands[5]) != UMIN || GET_CODE(operands[7]) == ZERO_EXTEND)
   && side_effect_mode_ok(MULT,operands,5,4,2,3,0)"

  "@
   #
   %x6%e7.%m7 [%5=%4+%2%T3],%0")

; HI-mode -> SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r")
        (match_operator:SI
         6 "operand_extend_operator"
         [(match_operand:SI 1 "register_operand" "0,0")
          (match_operator:SI
           7 "extend_operator"
           [(mem:HI (plus:SI (mult:SI (match_operand:SI 2 "register_operand" "r,r")
                                      (match_operand:SI 3 "const_int_operand" "n,n"))
                             (match_operand:SI 4 "register_operand" "r,r")))])]))
   (set (match_operand:SI 5 "register_operand" "=4,r")
        (plus:SI (mult:SI (match_dup 2)
                          (match_dup 3))
                 (match_dup 4)))]

  "(GET_CODE(operands[5]) != UMIN || GET_CODE(operands[7]) == ZERO_EXTEND)
   && side_effect_mode_ok(MULT,operands,5,4,2,3,0)"

  "@
   #
   %x6%e7.%m7 [%5=%4+%2%T3],%0")
;
; [rx=ry+i] 
;
; QI-mode -> HI-mode
(define_insn ""
  [(set (match_operand:HI 0 "register_operand" "=r,r,r")
        (match_operator:HI
         5 "operand_extend_operator"
         [(match_operand:HI 1 "register_operand" "0,0,0")
          (match_operator:HI
           6 "extend_operator"
           [(mem:QI (plus:SI (match_operand:SI 2 "bdap_operand" "%r,r,r")
                          (match_operand:SI 3 "bdap_operand" "r>Ri,r,>Ri")
                          ))])]))
   (set (match_operand:SI 4 "register_operand" "=2,r,r")
        (plus:SI (match_dup 2)
                 (match_dup 3)))]

  "(GET_CODE(operands[5]) != UMIN || GET_CODE(operands[6]) == ZERO_EXTEND)
   && side_effect_mode_ok(PLUS,operands,4,2,3,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[3]) != CONST_INT
             || INTVAL(operands[3]) > 127
             || INTVAL(operands[3]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'J')))
        return \"#\";
     return \"%x5%e6.%m6 [%4=%2%S3],%0\";")

; QI-mode -> SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r,r")
        (match_operator:SI
         5 "operand_extend_operator"
         [(match_operand:SI 1 "register_operand" "0,0,0")
          (match_operator:SI
           6 "extend_operator"
           [(mem:QI (plus:SI (match_operand:SI 2 "bdap_operand" "%r,r,r")
                             (match_operand:SI 3 "bdap_operand" "r>Ri,r,>Ri")
                             ))])]))
   (set (match_operand:SI 4 "register_operand" "=2,r,r")
        (plus:SI (match_dup 2)
                 (match_dup 3)))]

  "(GET_CODE(operands[5]) != UMIN || GET_CODE(operands[6]) == ZERO_EXTEND)
   && side_effect_mode_ok(PLUS,operands,4,2,3,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[3]) != CONST_INT
             || INTVAL(operands[3]) > 127
             || INTVAL(operands[3]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'J')))
        return \"#\";
     return \"%x5%e6.%m6 [%4=%2%S3],%0\";")

; HI-mode -> SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r,r")
        (match_operator:SI
         5 "operand_extend_operator"
         [(match_operand:SI 1 "register_operand" "0,0,0")
          (match_operator:SI
           6 "extend_operator"
           [(mem:HI (plus:SI (match_operand:SI 2 "bdap_operand" "%r,r,r")
                             (match_operand:SI 3 "bdap_operand" "r>Ri,r,>Ri")
                             ))])]))
   (set (match_operand:SI 4 "register_operand" "=2,r,r")
        (plus:SI (match_dup 2)
                 (match_dup 3)))]

  "(GET_CODE(operands[5]) != UMIN || GET_CODE(operands[6]) == ZERO_EXTEND)
   && side_effect_mode_ok(PLUS,operands,4,2,3,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[3]) != CONST_INT
             || INTVAL(operands[3]) > 127
             || INTVAL(operands[3]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'J')))
        return \"#\";
     return \"%x5%e6.%m6 [%4=%2%S3],%0\";")

;
; As with op.s we may have to add special pattern to match commuted
; operands to adds/addu +bound
;
;
; adds/addu [rx=ry+rz.S] + bound
;
; QI-mode -> HI-mode
(define_insn ""
  [(set (match_operand:HI 0 "register_operand" "=r,r")
        (match_operator:HI
         7 "plus_or_bound_operator"
         [(match_operator:HI
           6 "extend_operator"
           [(mem:QI (plus:SI (mult:SI (match_operand:SI 2 "register_operand" "r,r")
                                   (match_operand:SI 3 "const_int_operand" "n,n"))
                          (match_operand:SI 4 "register_operand" "r,r")))])
          (match_operand:HI 1 "register_operand" "0,0")]))
   (set (match_operand:SI 5 "register_operand" "=4,r")
        (plus:SI (mult:SI (match_dup 2)
                          (match_dup 3))
                 (match_dup 4)))]

  "(GET_CODE(operands[6]) != UMIN || GET_CODE(operands[6]) == ZERO_EXTEND)
   && side_effect_mode_ok(MULT,operands,5,4,2,3,0)"

  "@
   #
   %x7%e6.%m6 [%5=%4+%2%T3],%0")

; QI-mode -> SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r")
        (match_operator:SI
         7 "plus_or_bound_operator"
         [(match_operator:SI
           6 "extend_operator"
           [(mem:QI (plus:SI (mult:SI (match_operand:SI 2 "register_operand" "r,r")
                                      (match_operand:SI 3 "const_int_operand" "n,n"))
                             (match_operand:SI 4 "register_operand" "r,r")))])
          (match_operand:SI 1 "register_operand" "0,0")]))
   (set (match_operand:SI 5 "register_operand" "=4,r")
        (plus:SI (mult:SI (match_dup 2)
                          (match_dup 3))
                 (match_dup 4)))]

  "(GET_CODE(operands[6]) != UMIN || GET_CODE(operands[6]) == ZERO_EXTEND)
   && side_effect_mode_ok(MULT,operands,5,4,2,3,0)"

  "@
   #
   %x7%e6.%m6 [%5=%4+%2%T3],%0")

; HI-mode -> SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r")
        (match_operator:SI
         7 "plus_or_bound_operator"
         [(match_operator:SI
           6 "extend_operator"
           [(mem:HI (plus:SI (mult:SI (match_operand:SI 2 "register_operand" "r,r")
                                      (match_operand:SI 3 "const_int_operand" "n,n"))
                             (match_operand:SI 4 "register_operand" "r,r")))])
          (match_operand:SI 1 "register_operand" "0,0")]))
   (set (match_operand:SI 5 "register_operand" "=4,r")
        (plus:SI (mult:SI (match_dup 2)
                          (match_dup 3))
                 (match_dup 4)))]

  "(GET_CODE(operands[6]) != UMIN || GET_CODE(operands[6]) == ZERO_EXTEND)
   && side_effect_mode_ok(MULT,operands,5,4,2,3,0)"

  "@
   #
   %x7%e6.%m6 [%5=%4+%2%T3],%0")

;
; [rx=ry+i] 
;
; QI-mode -> HI-mode
(define_insn ""
  [(set (match_operand:HI 0 "register_operand" "=r,r,r")
        (match_operator:HI
         6 "plus_or_bound_operator"
         [(match_operator:HI
           5 "extend_operator"
           [(mem:QI (plus:SI (match_operand:SI 2 "bdap_operand" "%r,r,r")
                             (match_operand:SI 3 "bdap_operand" "r>Ri,r,>Ri")))])
          (match_operand:HI 1 "register_operand" "0,0,0")]))
   (set (match_operand:SI 4 "register_operand" "=2,r,r")
        (plus:SI (match_dup 2)
                 (match_dup 3)))]

  "(GET_CODE(operands[6]) != UMIN || GET_CODE(operands[5]) == ZERO_EXTEND)
   && side_effect_mode_ok(PLUS,operands,4,2,3,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[3]) != CONST_INT
             || INTVAL(operands[3]) > 127
             || INTVAL(operands[3]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'J')))
        return \"#\";
     return \"%x6%e5.%m5 [%4=%2%S3],%0\";")

; QI-mode -> SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r,r")
        (match_operator:SI
         6 "plus_or_bound_operator"
         [(match_operator:SI
           5 "extend_operator"
           [(mem:QI (plus:SI (match_operand:SI 2 "bdap_operand" "%r,r,r")
                             (match_operand:SI 3 "bdap_operand" "r>Ri,r,>Ri")))])
          (match_operand:SI 1 "register_operand" "0,0,0")]))
   (set (match_operand:SI 4 "register_operand" "=2,r,r")
        (plus:SI (match_dup 2)
                 (match_dup 3)))]

  "(GET_CODE(operands[6]) != UMIN || GET_CODE(operands[5]) == ZERO_EXTEND)
   && side_effect_mode_ok(PLUS,operands,4,2,3,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[3]) != CONST_INT
             || INTVAL(operands[3]) > 127
             || INTVAL(operands[3]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'J')))
        return \"#\";
     return \"%x6%e5.%m5 [%4=%2%S3],%0\";")

; HI-mode -> SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r,r")
        (match_operator:SI
         6 "plus_or_bound_operator"
         [(match_operator:SI
           5 "extend_operator"
           [(mem:HI (plus:SI (match_operand:SI 2 "bdap_operand" "%r,r,r")
                             (match_operand:SI 3 "bdap_operand" "r>Ri,r,>Ri")))])
          (match_operand:SI 1 "register_operand" "0,0,0")]))
   (set (match_operand:SI 4 "register_operand" "=2,r,r")
        (plus:SI (match_dup 2)
                 (match_dup 3)))]

  "(GET_CODE(operands[6]) != UMIN || GET_CODE(operands[5]) == ZERO_EXTEND)
   && side_effect_mode_ok(PLUS,operands,4,2,3,-1,0)"

  "* if (which_alternative == 0
         && (GET_CODE(operands[3]) != CONST_INT
             || INTVAL(operands[3]) > 127
             || INTVAL(operands[3]) < -128
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'N')
             || CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'J')))
        return \"#\";
     return \"%x6%e5.%m5 [%4=%2%S3],%0\";")

;
; Extend versions (zero/sign) of normal add/sub.
;
; Because of reloading problems, we have to specify the operand sizes;
; therefore both HI and QI variants.
; QI-mode -> HI-mode
(define_insn ""
  [(set (match_operand:HI 0 "register_operand" "=r,r,r,r")
        (match_operator:HI
         3 "operand_extend_operator"
         [(match_operand:HI 1 "register_operand" "0,0,0,r")
          (match_operator:HI
           4 "extend_operator"
           [(match_operand:QI 2 "nonimmediate_operand" "r,Q>,m,!To")])]))]
                                        ; Inhibit reloading to "To" alternative
;
; All insns that look like (set (..) (plus (...) (reg:SI 8)))
; get problems when reloading r8 (frame pointer) to r14+offs (stack pointer)
;  Avoid that.
;
; Also avoid add.s x,pc totally. The next pattern will match correctly.
;
  "(GET_CODE(operands[3]) != UMIN || GET_CODE(operands[4]) == ZERO_EXTEND)
   && REGNO(operands[0]) != 15
   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
   && (operands[1] != frame_pointer_rtx || GET_CODE(operands[3]) != PLUS)"
  "@
   %x3%e4.%m4 %2,%0
   %x3%e4.%m4 %2,%0
   %x3%e4.%m4 %R2,%0
   %x3%e4.%m4 %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,no,no")])

; QI-mode -> SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r,r,r")
        (match_operator:SI
         3 "operand_extend_operator"
         [(match_operand:SI 1 "register_operand" "0,0,0,r")
          (match_operator:SI
           4 "extend_operator"
           [(match_operand:QI 2 "nonimmediate_operand" "r,Q>,m,!To")])]))]
                                        ; Inhibit reloading to "To" alternative
;
; All insns that look like (set (..) (plus (...) (reg:SI 8)))
; get problems when reloading r8 (frame pointer) to r14+offs (stack pointer)
;  Avoid that.
;
; Also avoid add.s x,pc totally. The next pattern will match correctly.
;
  "(GET_CODE(operands[3]) != UMIN || GET_CODE(operands[4]) == ZERO_EXTEND)
   && REGNO(operands[0]) != 15
   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
   && (operands[1] != frame_pointer_rtx || GET_CODE(operands[3]) != PLUS)"
  "@
   %x3%e4.%m4 %2,%0
   %x3%e4.%m4 %2,%0
   %x3%e4.%m4 %R2,%0
   %x3%e4.%m4 %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,no,no")])

; HI-mode -> SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r,r,r")
        (match_operator:SI
         3 "operand_extend_operator"
         [(match_operand:SI 1 "register_operand" "0,0,0,r")
          (match_operator:SI
           4 "extend_operator"
           [(match_operand:HI 2 "nonimmediate_operand" "r,Q>,m,!To")])]))]
                                        ; Inhibit reloading to "To" alternative
;
; All insns that look like (set (..) (plus (...) (reg:SI 8)))
; get problems when reloading r8 (frame pointer) to r14+offs (stack pointer)
;  Avoid that.
;
; Also avoid add.s x,pc totally. The next pattern will match correctly.
;
  "(GET_CODE(operands[3]) != UMIN || GET_CODE(operands[4]) == ZERO_EXTEND)
   && REGNO(operands[0]) != 15
   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
   && (operands[1] != frame_pointer_rtx || GET_CODE(operands[3]) != PLUS)"
  "@
   %x3%e4.%m4 %2,%0
   %x3%e4.%m4 %2,%0
   %x3%e4.%m4 %R2,%0
   %x3%e4.%m4 %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,no,no")])

;
; As with side-effect pattern, may have to have swapped operands for add.
;
; Still reload problems (see pattern above), have both HI and QI patterns
; QI-mode -> HI-mode
(define_insn ""
  [(set (match_operand:HI 0 "register_operand" "=r,r,r,r")
        (match_operator:HI
         4 "plus_or_bound_operator"
         [(match_operator:HI
           3 "extend_operator"
           [(match_operand:QI 2 "nonimmediate_operand" "r,Q>,m,!To")])
          (match_operand:HI 1 "register_operand" "0,0,0,r")]))]
                                        ; Inhibit reloading to "To" alternative
;
; All insns that look like (set (..) (plus (...) (reg:SI 8)))
; get problems when reloading r8 (frame pointer) to r14+offs (stack pointer)
;  Avoid that.
;
; Also, reject "adds [pc+i],pc" (resulting from constant switch():es)
; because the "i"-offset calculation will be screwed up.
;  This is somekind of a kludge; constant "switch"-statements should
; be folded away, but is painful to do right (for the moment). Do the
; right thing later.  (Right thing is to change "casesi" NOT to kludge
; away with "reg 15"-fixes, for the jump.)
;

  "(GET_CODE(operands[3]) != UMIN || GET_CODE(operands[4]) == ZERO_EXTEND)
   && (REGNO(operands[0]) != 15
       || (GET_CODE(operands[2]) == MEM
           && GET_CODE(XEXP(operands[2],0)) == PLUS
           && GET_CODE(XEXP(XEXP(operands[2],0),1)) != CONST_INT
           && GET_CODE(XEXP(XEXP(operands[2],0),0)) != CONST_INT))
   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
   && operands[1] != frame_pointer_rtx"
  "@
   %x4%e3.%m3 %2,%0
   %x4%e3.%m3 %2,%0
   %x4%e3.%m3 %R2,%0
   %x4%e3.%m3 %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,no,no")])

; QI-mode -> SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r,r,r")
        (match_operator:SI
         4 "plus_or_bound_operator"
         [(match_operator:SI
           3 "extend_operator"
           [(match_operand:QI 2 "nonimmediate_operand" "r,Q>,m,!To")])
          (match_operand:SI 1 "register_operand" "0,0,0,r")]))]
                                        ; Inhibit reloading to "To" alternative
;
; All insns that look like (set (..) (plus (...) (reg:SI 8)))
; get problems when reloading r8 (frame pointer) to r14+offs (stack pointer)
;  Avoid that.
;
; Also, reject "adds [pc+i],pc" (resulting from constant switch():es)
; because the "i"-offset calculation will be screwed up.
;  This is somekind of a kludge; constant "switch"-statements should
; be folded away, but is painful to do right (for the moment). Do the
; right thing later.  (Right thing is to change "casesi" NOT to kludge
; away with "reg 15"-fixes, for the jump.)
;

  "(GET_CODE(operands[3]) != UMIN || GET_CODE(operands[4]) == ZERO_EXTEND)
   && (REGNO(operands[0]) != 15
       || (GET_CODE(operands[2]) == MEM
           && GET_CODE(XEXP(operands[2],0)) == PLUS
           && GET_CODE(XEXP(XEXP(operands[2],0),1)) != CONST_INT
           && GET_CODE(XEXP(XEXP(operands[2],0),0)) != CONST_INT))
   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
   && operands[1] != frame_pointer_rtx"
  "@
   %x4%e3.%m3 %2,%0
   %x4%e3.%m3 %2,%0
   %x4%e3.%m3 %R2,%0
   %x4%e3.%m3 %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,no,no")])

; HI-mode -> SI-mode
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r,r,r,r")
        (match_operator:SI
         4 "plus_or_bound_operator"
         [(match_operator:SI
           3 "extend_operator"
           [(match_operand:HI 2 "nonimmediate_operand" "r,Q>,m,!To")])
          (match_operand:SI 1 "register_operand" "0,0,0,r")]))]
                                        ; Inhibit reloading to "To" alternative
;
; All insns that look like (set (..) (plus (...) (reg:SI 8)))
; get problems when reloading r8 (frame pointer) to r14+offs (stack pointer)
;  Avoid that.
;
; Also, reject "adds [pc+i],pc" (resulting from constant switch():es)
; because the "i"-offset calculation will be screwed up.
;  This is somekind of a kludge; constant "switch"-statements should
; be folded away, but is painful to do right (for the moment). Do the
; right thing later.  (Right thing is to change "casesi" NOT to kludge
; away with "reg 15"-fixes, for the jump.)
;

  "(GET_CODE(operands[3]) != UMIN || GET_CODE(operands[4]) == ZERO_EXTEND)
   && (REGNO(operands[0]) != 15
       || (GET_CODE(operands[2]) == MEM
           && GET_CODE(XEXP(operands[2],0)) == PLUS
           && GET_CODE(XEXP(XEXP(operands[2],0),1)) != CONST_INT
           && GET_CODE(XEXP(XEXP(operands[2],0),0)) != CONST_INT))
   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
   && operands[1] != frame_pointer_rtx"
  "@
   %x4%e3.%m3 %2,%0
   %x4%e3.%m3 %2,%0
   %x4%e3.%m3 %R2,%0
   %x4%e3.%m3 %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,no,no")])

;
;- Multiply instructions.
;

;
; This is a cr*ck: the "reload" procedure allows a last resort when it
; dont know what to do with the things it needs to reload.
;  It knows what to do to reload  a+b into c.  If thats not what it is
; asked to do, it checks to reload d into c, where d in a "simple"
; operand; a reg or mem.
;  After that it "gives up" and tries (set c d), regardless of what
; complexity d has.  That would be ok if it wasnt for that the
; constraints MUST MATCH, OR GCC WILL ABORT! i.e. there is no reload
; of any registers involved. You could as well omit the constraints
; (but I keep them for a last semi-destructive test).
;  As a last resort, before the "giving up" Gcc checks for a pattern
; "reload_load_address".  This means we are just lucky enough to clean
; up the mess in this file, instead of rewriting gcc "core" code.
;  This probably mean that the "mult" pattern below is no longer
; needed.
(define_expand "reload_load_address"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (match_operand 1 "" ""))] ; Must match strange creatures

  "reload_in_progress && !reload_completed"

; Match all binary expressions. Add is taken care of in reload, so
; dont worry about this being non-optimal for most code.

  "{
   if (GET_RTX_CLASS(GET_CODE(operands[1])) != 'c'
       && GET_RTX_CLASS(GET_CODE(operands[1])) != '2') {
      debug_rtx(operands[1]);     
      fatal(\"Dont know how to reload the above into a register\");
   }
   /* Move mem or pseudoreg into dest, then do operation; 2-address */   
   if (GET_CODE(XEXP(operands[1],0)) == MEM
       || (GET_CODE(XEXP(operands[1],0)) == REG
           && REGNO(XEXP(operands[1],0)) >= FIRST_PSEUDO_REGISTER)
       || GET_RTX_CLASS(GET_CODE(operands[1])) == '2') {
     emit_move_insn(operands[0],XEXP(operands[1],0));
     emit_insn(gen_rtx(SET,VOIDmode,
                       operands[0],
                       gen_rtx(GET_CODE(operands[1]),GET_MODE(operands[0]),
                               operands[0],XEXP(operands[1],1))));
   } else {
     emit_move_insn(operands[0],XEXP(operands[1],1));
     emit_insn(gen_rtx(SET,VOIDmode,
                       operands[0],
                       gen_rtx(GET_CODE(operands[1]),GET_MODE(operands[0]),
                               operands[0],XEXP(operands[1],0))));
   }
   DONE;
}")

;
; Sometimes powers of 2 (which are normally canonicalized to a
; left-shift) appear here, as a result of address reloading.
; As a special, for values 3 and 5, we can match with an addi, so add those.
;
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r")
        (mult:SI
         (match_operand:SI 1 "register_operand" "%0")
         (match_operand:SI 2 "const_int_operand" "n")))]
;
; Due to problems related to changing frame-pointer into
; stack-pointer+offset, (the insn cant be re-recognized) it is best to
; leave this alternative out.
;
  "operands[0] != frame_pointer_rtx
   && operands[1] != frame_pointer_rtx
   && GET_CODE(operands[2]) == CONST_INT
   && (INTVAL(operands[2]) == 2
       || INTVAL(operands[2]) == 4 || INTVAL(operands[2]) == 3 
       || INTVAL(operands[2]) == 5)"
  "*
{
  if (INTVAL(operands[2]) == 2)
    return \"lslq 1,%0\";
  else if (INTVAL(operands[2]) == 4)
    return \"lslq 2,%0\";
  else if (INTVAL(operands[2]) == 3)
    return \"addi %0.w,%0\";
  else if (INTVAL(operands[2]) == 5)
    return \"addi %0.d,%0\";
  return \"BAD: adr_mulsi: %0=%1*%2\";
}"
[(set_attr "slottable" "yes")])

;
; Define the addi insn as it normally is used.
;
(define_insn ""
  [(set (match_operand:SI 0 "register_operand" "=r")
        (plus:SI
         (mult:SI (match_operand:SI 2 "register_operand" "r")
                  (match_operand:SI 3 "const_int_operand" "n"))
         (match_operand:SI 1 "register_operand" "0")))]
;
; Due to problems related to changing frame-pointer into
; stack-pointer+offset, (the insn cant be re-recognized) it is best to
; leave this alternative out.
;
  "operands[0] != frame_pointer_rtx
   && operands[1] != frame_pointer_rtx
   && GET_CODE(operands[3]) == CONST_INT
   && (INTVAL(operands[3]) == 1
       || INTVAL(operands[3]) == 2 || INTVAL(operands[3]) == 4)"
  "addi %2%T3,%0"
  [(set_attr "slottable" "yes")])

;
; The mstep instruction.
;
(define_insn "mstep_shift"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (if_then_else
         (lt (cc0) (const_int 0))
         (plus:SI (ashift:SI (match_operand:SI 1 "register_operand" "0")
                             (const_int 1))
                  (match_operand:SI 2 "register_operand" "r"))
         (ashift:SI (match_operand:SI 3 "register_operand" "0")
                    (const_int 1))))]
  ""
  "mstep %2,%0"
  [(set_attr "slottable" "yes")])

;
; When illegitimate addresses are legitimized, sometimes gcc forgets
; to canonicalize the multiplies.
;
(define_insn "mstep_mul"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (if_then_else
         (lt (cc0) (const_int 0))
         (plus:SI (mult:SI (match_operand:SI 1 "register_operand" "0")
                           (const_int 2))
                  (match_operand:SI 2 "register_operand" "r"))
         (mult:SI (match_operand:SI 3 "register_operand" "0")
                  (const_int 2))))]
  "operands[0] != frame_pointer_rtx
   && operands[1] != frame_pointer_rtx
   && operands[2] != frame_pointer_rtx
   && operands[3] != frame_pointer_rtx"
  "mstep %2,%0"
  [(set_attr "slottable" "yes")])

;
; "Real" multiply
;
; Say we can save the zero-extend operations, since the operands
; really are zero-extended
;
(define_expand "umulhisi3"
;  [(set (match_operand:SI 0 "register_operand" "=r")
;       (mult:SI (zero_extend:SI (match_operand:HI 1 "register_operand" "%0"))
;                (zero_extend:SI (match_operand:HI 2 "register_operand" "r"))))]
  [(set (match_dup 3)
        (zero_extend:SI (match_operand:HI 2 "nonimmediate_operand" "rm")))
   (set (match_dup 4)
        (subreg:SI (match_operand:HI 1 "general_operand" "g") 0))
   (set (match_dup 5) (ashift:SI (match_dup 4)
                                 (const_int 16)))
   (set (match_dup 6) (match_dup 5))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (match_operand:SI 0 "nonimmediate_operand" "=rm")
        (match_dup 6))]
  "TARGET_MUL"
  "{ int i; for (i = 3; i <= 6; i++) operands[i] = gen_reg_rtx(SImode);}")


;(define_expand "mulhisi3"
;  [(set (match_operand:SI 0 "register_operand" "=r")
;       (mult:SI (sign_extend:SI (match_operand:HI 1 "register_operand" "%0"))
;                (sign_extend:SI (match_operand:HI 2 "register_operand" "r"))))]
;  "TARGET_MUL"
;  "")

(define_expand "umulqihi3"
;  [(set (match_operand:HI 0 "register_operand" "=r")
;       (mult:HI (zero_extend:HI (match_operand:QI 1 "register_operand" "%0"))
;                (zero_extend:HI (match_operand:QI 2 "register_operand" "r"))))]
  [(set (match_dup 3)
        (zero_extend:SI (match_operand:QI 2 "nonimmediate_operand" "rm")))
   (set (match_dup 4)
        (subreg:SI (match_operand:QI 1 "general_operand" "g") 0))
   (set (match_dup 5) (ashift:SI (match_dup 4)
                                 (const_int 24)))
   (set (match_dup 6) (match_dup 5))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (match_operand:HI 0 "nonimmediate_operand" "=rm")
        (subreg:HI (match_dup 6) 0))]
  "TARGET_MUL"
  "{ int i; for (i = 3; i <= 6; i++) operands[i] = gen_reg_rtx(SImode);}")

;(define_expand "mulqihi3"
;  [(set (match_operand:HI 0 "register_operand" "=r")
;       (mult:HI (sign_extend:HI (match_operand:QI 1 "register_operand" "%0"))
;                (sign_extend:HI (match_operand:QI 2 "register_operand" "r"))))]
;  "TARGET_MUL"
;  "")

(define_expand "mulsi3"
  [(set (match_operand:SI 0 "general_operand" "=r")
        (mult:SI (match_operand:SI 1 "general_operand" "%0")
                 (match_operand:SI 2 "general_operand" "r")))]
  "TARGET_MUL"
;
; Seems unnecessary to do this here, when GCC could expand it as well,
; or better. See optabs.c for an explanation.
;
  "
{
  rtx ops[8];
  int i;
  for (i = 0; i < 3; i++)
    ops[i] = gen_reg_rtx(SImode);
  for (i = 3; i < 6; i++)
    ops[i] = gen_reg_rtx(HImode);
  for (i = 6; i < 8; i++)
    ops[i] = gen_reg_rtx(SImode);
  emit_insn(gen_umulhisi3(ops[0],
                         gen_rtx(SUBREG,HImode,operands[1],0),
                         gen_rtx(SUBREG,HImode,operands[2],0)));
  emit_insn(gen_lshrsi3(ops[1],
                        operands[1],
                        gen_rtx(CONST_INT, VOIDmode, 16)));
  emit_insn(gen_lshrsi3(ops[2],
                        operands[2],
                        gen_rtx(CONST_INT, VOIDmode, 16)));
  emit_insn(gen_mulhi3(ops[3],
                       gen_rtx(SUBREG,HImode,ops[1],0),
                       gen_rtx(SUBREG,HImode,operands[2],0)));
  emit_insn(gen_mulhi3(ops[4],
                       gen_rtx(SUBREG,HImode,ops[2],0),
                       gen_rtx(SUBREG,HImode,operands[1],0)));
  emit_insn(gen_addhi3(ops[5],ops[3],ops[4]));
  emit_insn(gen_ashlsi3(ops[6],
                        gen_rtx(SUBREG,SImode,ops[5],0),
                        gen_rtx(CONST_INT,VOIDmode,16)));
  emit_insn(gen_addsi3(ops[7],ops[0],ops[6]));
  emit_insn(gen_movsi(operands[0],ops[7]));
  DONE;
}")

(define_expand "mulhi3"
;  [(set (match_operand:HI 0 "register_operand" "=r")
;       (mult:HI (match_operand:HI 1 "register_operand" "%0")
;                (match_operand:HI 2 "register_operand" "r")))]
  [(set (match_dup 3)
        (zero_extend:SI (match_operand:HI 2 "nonimmediate_operand" "rm")))
   (set (match_dup 4)
        (subreg:SI (match_operand:HI 1 "general_operand" "g") 0))
   (set (match_dup 5) (ashift:SI (match_dup 4)
                                 (const_int 16)))
   (set (match_dup 6) (match_dup 5))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (match_operand:HI 0 "nonimmediate_operand" "=rm")
        (subreg:HI (match_dup 6) 0))]
  "TARGET_MUL"
  "{ int i; for (i = 3; i <= 6; i++) operands[i] = gen_reg_rtx(SImode);}")

(define_expand "mulqi3"
;  [(set (match_operand:QI 0 "register_operand" "=r")
;       (mult:QI (match_operand:QI 1 "register_operand" "%0")
;                (match_operand:QI 2 "register_operand" "r")))]
  [(set (match_dup 3)
        (zero_extend:SI (match_operand:QI 2 "nonimmediate_operand" "rm")))
   (set (match_dup 4)
        (subreg:SI (match_operand:QI 1 "general_operand" "g") 0))
   (set (match_dup 5) (ashift:SI (match_dup 4)
                                 (const_int 24)))
   (set (match_dup 6) (match_dup 5))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (cc0) (match_dup 6))
   (set (match_dup 6) (if_then_else
                       (lt (cc0) (const_int 0))
                       (plus:SI (ashift:SI (match_dup 6)
                                           (const_int 1))
                                (match_dup 3))
                       (ashift:SI (match_dup 6)
                                  (const_int 1))))
   (set (match_operand:QI 0 "nonimmediate_operand" "=rm")
        (subreg:QI (match_dup 6) 0))]
  "TARGET_MUL"
  "{ int i; for (i = 3; i <= 6; i++) operands[i] = gen_reg_rtx(SImode);}")

;
;- Divide and modulus instructions. Cris has a step instruction.
;
; First, define the dstep insn.
(define_insn "dstep_shift"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (if_then_else
         (geu (ashift:SI (match_operand:SI 1 "register_operand" "0")
                         (const_int 1))
              (match_operand:SI 2 "register_operand" "r"))
         (minus:SI (ashift:SI (match_operand:SI 3 "register_operand" "0")
                        (const_int 1))
                   (match_operand:SI 4 "register_operand" "2"))
         (ashift:SI (match_operand:SI 5 "register_operand" "0")
                        (const_int 1))))]
  ""
  "dstep %2,%0"
  [(set_attr "slottable" "yes")])

;
; And with mult istf. ashift         
;
(define_insn "dstep_mul"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (if_then_else
         (geu (mult:SI (match_operand:SI 1 "register_operand" "0")
                       (const_int 2))
              (match_operand:SI 2 "register_operand" "r"))
         (minus:SI (mult:SI (match_operand:SI 3 "register_operand" "0")
                            (const_int 2))
                   (match_operand:SI 4 "register_operand" "2"))
         (mult:SI (match_operand:SI 5 "register_operand" "0")
                  (const_int 2))))]

  "operands[0] != frame_pointer_rtx
   && operands[1] != frame_pointer_rtx
   && operands[2] != frame_pointer_rtx
   && operands[3] != frame_pointer_rtx"
  "dstep %2,%0"
  [(set_attr "slottable" "yes")])

;(define_expand "udivmodsi4"
; [(set (match_operand:SI 0 "register_operand" "=r")
;   (udiv:SI (match_operand:SI 1 "register_operand" "r")
;    (match_operand:SI 2 "register_operand" "0")))
;  (set (match_operand:SI 3 "register_operand" "=1")
;   (umod:SI (match_dup 1) (match_dup 2)))]
;
; "TARGET_DIV"
; "")

;
; Signed divmod is define_expanded into the code below
; (approximately).
;
;move %G1,%4
;abs %4,%5
;move %G2,%6
;abs %6,%7
;move %6,%8
;move %5,%9
;move %7,%10
;:udiv %10 = %9 / %10
;:umod %9 = %9 % %10
;xor %4,%8
;if (%8 > 0) goto %11
;neg %10,%10
;%11:
;if (%4 > 0) goto %12
;neg %9,%9
;%12:
;move %10,%0
;;move %9,%3
;(define_expand "divmodsi4"
;  [(set (match_dup 4) (match_operand:SI 1 "general_operand" "g"))
;   (set (match_dup 5) (abs:SI (match_dup 4)))
;   (set (match_dup 9) (match_dup 5))
;   (set (match_dup 6) (match_operand:SI 2 "general_operand" "g"))
;   (set (match_dup 7) (abs:SI (match_dup 6)))
;   (set (match_dup 8) (match_dup 6))
;   (set (match_dup 10) (match_dup 7))
;   (parallel [(set (match_dup 10) (udiv:SI (match_dup 9) (match_dup 10)))
;             (set (match_dup 9) (umod:SI (match_dup 9) (match_dup 10)))])
;   (set (match_dup 11) (xor:SI (match_dup 8) (match_dup 4)))
;   (set (cc0) (match_dup 11))
;   (set (pc) (if_then_else
;             (ge (cc0)
;                 (const_int 0))
;             (label_ref (match_dup 12))
;             (pc)))
;   (set (match_dup 10) (neg:SI (match_dup 10)))
;   (match_dup 12)
;   (set (cc0) (match_dup 4))
;   (set (pc) (if_then_else
;             (ge (cc0)
;                 (const_int 0))
;             (label_ref (match_dup 13))
;             (pc)))
;   (set (match_dup 9) (neg:SI (match_dup 9)))
;   (match_dup 13)
;   (set (match_operand:SI 0 "general_operand" "g") (match_dup 10))
;   (set (match_operand:SI 3 "general_operand" "g") (match_dup 9))]
;  "TARGET_DIV"
;  "{
;     int i;
;
;     for (i = 4; i < 12; i++)
;       operands[i] = gen_reg_rtx(SImode);
;     /* Generate an unique label. */
;     operands[12] = gen_label_rtx();
;     operands[13] = gen_label_rtx();
;   }")
;
;(define_expand "udivmodhi4"
;  [(set (match_operand:HI 0 "register_operand" "=r")
;       (udiv:HI (match_operand:HI 1 "register_operand" "r")
;               (match_operand:HI 2 "register_operand" "0")))
;   (set (match_operand:HI 3 "register_operand" "=1")
;       (umod:HI (match_dup 1) (match_dup 2)))]
;  "TARGET_DIV"
;  "")
;
;(define_expand "divmodhi4"
;  [(set (match_dup 4) (sign_extend:SI (match_operand:HI 1 "general_operand" "g")))
;   (set (match_dup 5) (abs:SI (match_dup 4)))
;   (set (match_dup 6) (sign_extend:SI (match_operand:HI 2 "general_operand" "g")))
;   (set (match_dup 7) (abs:SI (match_dup 6)))
;   (set (match_dup 8) (match_dup 6))
;   (set (match_dup 9) (match_dup 5))
;   (set (match_dup 10) (match_dup 7))
;   (parallel [(set (match_dup 10) (udiv:SI (match_dup 9) (match_dup 10)))
;             (set (match_dup 9) (umod:SI (match_dup 9) (match_dup 10)))])
;   (set (match_dup 11) (xor:SI (match_dup 8) (match_dup 4))) ; Do tying here.
;   (set (cc0) (match_dup 11))
;   (set (pc) (if_then_else
;             (ge (cc0)
;                 (const_int 0))
;             (label_ref (match_dup 12))
;             (pc)))
;   (set (match_dup 10) (neg:SI (match_dup 10)))
;   (match_dup 12)
;   (set (cc0) (match_dup 4))
;   (set (pc) (if_then_else
;             (ge (cc0)
;                 (const_int 0))
;             (label_ref (match_dup 13))
;             (pc)))
;   (set (match_dup 9) (neg:SI (match_dup 9)))
;   (match_dup 13)
;   (set (match_operand:HI 0 "general_operand" "g") (subreg:HI (match_dup 10) 0))
;   (set (match_operand:HI 3 "general_operand" "g") (subreg:HI (match_dup 9) 0))]
;  "TARGET_DIV"
;  "{
;     int i;
;
;     for (i = 4; i < 12; i++)
;       operands[i] = gen_reg_rtx(SImode);
;     /* Generate an unique label. */
;     operands[12] = gen_label_rtx();
;     operands[13] = gen_label_rtx();
;   }")
;
;(define_expand "udivmodqi4"
;  [(set (match_operand:QI 0 "register_operand" "=r")
;       (udiv:QI (match_operand:QI 1 "general_operand" "r")
;               (match_operand:QI 2 "general_operand" "0")))
;   (set (match_operand:QI 3 "register_operand" "=1")
;       (umod:QI (match_dup 1) (match_dup 2)))]
;  "TARGET_DIV"
;  "")
;
;(define_expand "divmodqi4"
;  [(set (match_dup 4) (sign_extend:SI (match_operand:QI 1 "general_operand" "g")))
;   (set (match_dup 5) (abs:SI (match_dup 4)))
;   (set (match_dup 6) (sign_extend:SI (match_operand:QI 2 "general_operand" "g")))
;   (set (match_dup 7) (abs:SI (match_dup 6)))
;   (set (match_dup 8) (match_dup 6))
;   (set (match_dup 9) (match_dup 5))
;   (set (match_dup 10) (match_dup 7))
;   (parallel [(set (match_dup 10) (udiv:SI (match_dup 9) (match_dup 10)))
;             (set (match_dup 9) (umod:SI (match_dup 9) (match_dup 10)))])
;   (set (match_dup 11) (xor:SI (match_dup 8) (match_dup 4))) ; Do tying here.
;   (set (cc0) (match_dup 11))
;   (set (pc) (if_then_else
;             (ge (cc0)
;                 (const_int 0))
;             (label_ref (match_dup 12))
;             (pc)))
;   (set (match_dup 10) (neg:SI (match_dup 10)))
;   (match_dup 12)
;   (set (cc0) (match_dup 4))
;   (set (pc) (if_then_else
;             (ge (cc0)
;                 (const_int 0))
;             (label_ref (match_dup 13))
;             (pc)))
;   (set (match_dup 9) (neg:SI (match_dup 9)))
;   (match_dup 13)
;   (set (match_operand:QI 0 "general_operand" "g") (subreg:QI (match_dup 10) 0))
;   (set (match_operand:QI 3 "general_operand" "g") (subreg:QI (match_dup 9) 0))]
;  "TARGET_DIV"
;  "{
;     int i;
;
;     for (i = 4; i < 12; i++)
;       operands[i] = gen_reg_rtx(SImode);
;     /* Generate an unique label. */
;     operands[12] = gen_label_rtx();
;     operands[13] = gen_label_rtx();
;   }")
;
;
;
; Logical operators.
;
; Bitwise and
;

; There is no use in defining "anddi3", because gcc can expand this by
;  itself, and make good code without us interfering.
;

;
; Gcc isnt smart enough to ALWAYS change a and 255|65535 into zero_extend
; thus we make a expand of this and match 65535|255 separately (we are
; out of constant constraint letters). Note that operand #1 is
; general_operand in the expand.
;
(define_expand "andsi3"
; Is "nonimmediate_operand" allright, or must we have
; "register_operand" ? (That would break the use of anonymous
; to-memory "and" patterns below).  Trying 2.7.2
  [(set (match_operand:SI 0 "nonimmediate_operand"         "")
        (and:SI (match_operand:SI 1 "nonimmediate_operand" "")
                (match_operand:SI 2 "general_operand"    "")))]
  ""  
;
; Also, make sure one of the first operands is a register, or that the
; third operand is 255, -256, 65535, -65536, else the pattern will not
; be recognized later.  Try to remember to update the numbers above
; when we discover new nifty and-related optimizations.
;  Note: this is NOT the same as asking for a register_operand for
; operand[0], since it is only when the first operand is
; non-register and the third is a non-optimizable integer that this
; will happen.
;
  
  "{
     if (GET_CODE(operands[0]) != REG
         && (GET_CODE(operands[2]) != CONST_INT
             || (INTVAL(operands[2]) != -256
                 && INTVAL(operands[2]) != -65536)))
     {
       /* Instead of 'and.d X,Y => Z' where neither X nor Y is a
          register, we make 'and.d X,Y => r'; 'move.d r => Z', and hope that
          the reload phase will make something usefult out of it. */
       rtx reg = gen_reg_rtx(SImode);
       emit_insn(gen_rtx(SET,SImode,reg,
                         gen_rtx(AND,SImode,operands[1],operands[2])));
       emit_move_insn(operands[0],reg);
       DONE;
     }
   }")

;
; Don't try to be wise and clear upper word/byte when (and 255) or (and 65535),
; that makes too hard constraints for the poor reloader to handle,
; expecially during -O0.

(define_insn ""
  [(set (match_operand:SI 0 "register_operand"             "=r,r,r")
        (and:SI (match_operand:SI 1 "nonimmediate_operand" "%r,Q>,m")
                (match_operand:SI 2 "const_int_operand"    "n,n,n")))]
  "INTVAL(operands[2]) == 255 || INTVAL(operands[2]) == 65535"  
  "movu.%z2 %R1,%0"
  [(set_attr "slottable" "yes,yes,no")])

(define_insn ""
  [(set (match_operand:SI 0 "nonimmediate_operand"            "=r,r,Q>,Q>,m,m")
        (and:SI (match_operand:SI 1 "nonimmediate_operand" "%0,0,0,0,0,0")
                (match_operand:SI 2 "const_int_operand"    "P,n,P,n,P,n")))]
  "INTVAL(operands[2]) == -65536 || INTVAL(operands[2]) == -256"  
  "* CC_STATUS_INIT; /* A kludge.  Really fix in notice_update_cc() */
   { static char *insns[] = { \"cLear.b %0\", \"cLear.w %0\", \"cLear.b %0\",
     \"cLear.w %0\", \"cLear.b %0\", \"cLear.w %0\"};
    return insns[which_alternative];}"
  [(set_attr "slottable" "yes,yes,yes,yes,no,no")])

;
; This pattern has to take care of what wasnt matched in the insns
; above. Therefore it has to match anything matched by the _expand
; pattern. Unfortunately this means that the constraints has to take
; care of reloading to registers for the destination.
;
(define_insn ""
  [(set (match_operand:SI 0 "register_operand"    "=r,r,r,r,r")
        (and:SI (match_operand:SI 1 "nonimmediate_operand" "%0,0,0,0,r")
                (match_operand:SI 2 "general_operand" "I,r,Q>,g,!To")))]
                                        ; Inhibit reloading to "To" alternative
  ""
  "@
   andq %2,%0
   and.d %2,%0
   and.d %2,%0
   and.d %R2,%0
   and.d %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,yes,no,no")])

;
; For both QI and HI we may use the quick patterns.  This results in
; worthless condition codes, but that is known by notice_update_cc().
;

; See note for andsi3
(define_expand "andhi3"
  [(set (match_operand:HI 0 "nonimmediate_operand"         "")
        (and:HI (match_operand:HI 1 "nonimmediate_operand" "")
                (match_operand:HI 2 "general_operand"  "")))]
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
;
; Also, make sure the first operands is a register, or that the
; third operand is -256 or 65280, else the pattern will not
; be recognized later.  Try to remember to update the numbers above
; when we discover new nifty and-related optimizations.
;
  "{
     if (GET_CODE(operands[0]) != REG
         && (GET_CODE(operands[2]) != CONST_INT
             || (INTVAL(operands[2]) != -256
                 && INTVAL(operands[2]) != 65280)))
     {
       /* Instead of 'and.d X,Y => Z' where neither X nor Y is a
          register, we make 'and.d X,Y => r'; 'move.d r => Z', and hope that
          the reload phase will make something usefult out of it. */
       rtx reg = gen_reg_rtx(HImode);
       emit_insn(gen_rtx(SET,HImode,reg,
                         gen_rtx(AND,HImode,operands[1],operands[2])));
       emit_move_insn(operands[0],reg);
       DONE;
     }
   }")

(define_insn ""
  [(set (match_operand:HI 0 "register_operand"         "=r,r,r")
        (and:HI (match_operand:HI 1 "nonimmediate_operand" "r,Q>,m")
                (const_int 255)))]
  ""
  "mOvu.b %R1,%0"
  [(set_attr "slottable" "yes,yes,no")])

(define_insn ""
  [(set (match_operand:HI 0 "nonimmediate_operand"             "=r,Q>,m")
        (and:SI (match_operand:HI 1 "nonimmediate_operand" "0,0,0")
                (const_int -256)))] ; 0xff00
  ""
; A kludge.  Really fix in notice_update_cc()
  "* CC_STATUS_INIT; return \"cLear.b %0\";"
  [(set_attr "slottable" "yes,yes,no")])

(define_insn ""
  [(set (match_operand:HI 0 "nonimmediate_operand"             "=r,Q>,m")
        (and:HI (match_operand:HI 1 "nonimmediate_operand" "0,0,0")
                (const_int 65280)))] ; 0xff00
  ""
; A kludge.  Really fix in notice_update_cc()
  "* CC_STATUS_INIT; return \"cLear.b %0\";"
  [(set_attr "slottable" "yes,yes,no")])

(define_insn ""
  [(set (match_operand:HI 0 "register_operand"         "=r,r,r,r,r,r,r")
        (and:HI (match_operand:HI 1 "general_operand" "%0,0,0,0,0,0,r")
                (match_operand:HI 2 "general_operand"  "I,r,Q>,L,O,g,!To")))]
;                                                'L' before 'O' to
;                                                catch 255-31..255
                                        ; Inhibit reloading to "To" alternative
  ""
  "@
   andq %2,%0
   and.w %2,%0
   and.w %2,%0
   and.w %2,%0
   anDq %b2,%0
   and.w %R2,%0
   and.w %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,yes,no,yes,no,no")])

(define_insn ""
  [(set (strict_low_part
         (match_operand:HI 0 "register_operand"         "=r,r,r,r,r,r"))
        (and:HI (match_operand:HI 1 "register_operand" "%0,0,0,0,0,r")
                (match_operand:HI 2 "general_operand"  "r,Q>,L,O,g,!To")))]
;                                                'L' before 'O' to
;                                                catch 255-31..255
                                        ; Inhibit reloading to "To" alternative
  ""
  "@
   and.w %2,%0
   and.w %2,%0
   and.w %2,%0
   anDq %b2,%0
   and.w %R2,%0
   and.w %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,no,yes,no,no")])

(define_insn "andqi3"
  [(set (match_operand:QI 0 "register_operand"         "=r,r,r,r,r,r")
        (and:QI (match_operand:QI 1 "register_operand" "%0,0,0,0,0,r")
                (match_operand:QI 2 "general_operand"  "I,r,Q>,O,g,!To")))]
                                        ; Inhibit reloading to "To" alternative
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "@
   andq %2,%0
   and.b %2,%0
   and.b %2,%0
   andQ %b2,%0
   and.b %R2,%0
   and.b %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,yes,yes,no,no")])

(define_insn ""
  [(set (strict_low_part
         (match_operand:QI 0 "register_operand"        "=r,r,r,r,r"))
        (and:QI (match_operand:QI 1 "register_operand" "%0,0,0,0,r")
                (match_operand:QI 2 "general_operand"   "r,Q>,O,g,!To")))]
                                        ; Inhibit reloading to "To" alternative
  ""
  "@
   and.b %2,%0
   and.b %2,%0
   andQ %b2,%0
   and.b %R2,%0
   and.b %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,yes,no,no")])

;
; Bitwise or.
;

;
; Same comment as anddi3 applies here.
;

(define_insn "iorsi3"
  [(set (match_operand:SI 0 "register_operand" "=r,r,r,r,r,r")
        (ior:SI (match_operand:SI 1 "register_operand" "%0,0,0,0,0,r")
                (match_operand:SI 2 "general_operand" "I,r,Q>,n,g,!To")))]
                                        ; Inhibit reloading to "To" alternative
  ""
  "@
   orq %2,%0
   or.d %2,%0
   or.d %2,%0
   oR.%s2 %2,%0
   or.d %R2,%0
   or.d %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,yes,no,no,no")])

(define_insn "iorhi3"
  [(set (match_operand:HI 0 "register_operand" "=r,r,r,r,r,r,r")
        (ior:HI (match_operand:HI 1 "register_operand" "%0,0,0,0,0,0,r")
                (match_operand:HI 2 "general_operand" "I,r,Q>,L,O,g,!To")))]
;                                      'L' before 'O' to catch 255-31..255
                                        ; Inhibit reloading to "To" alternative
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "@
   orq %2,%0
   or.w %2,%0
   or.w %2,%0
   or.w %2,%0
   oRq %b2,%0
   or.w %R2,%0
   or.w %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,yes,no,yes,no,no")])

(define_insn "iorqi3"
  [(set (match_operand:QI 0 "register_operand" "=r,r,r,r,r,r")
        (ior:QI (match_operand:QI 1 "register_operand" "%0,0,0,0,0,r")
                (match_operand:QI 2 "general_operand" "I,r,Q>,O,g,!To")))]
                                        ; Inhibit reloading to "To" alternative
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "@
   orq %2,%0
   or.b %2,%0
   or.b %2,%0
   orQ %b2,%0
   or.b %R2,%0
   or.b %R2,%1,%0"
  [(set_attr "slottable" "yes,yes,yes,yes,no,no")])

;
;- xor instructions.
;

; See comment about "anddi3" for xordi3

(define_insn "xorsi3"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (xor:SI (match_operand:SI 1 "register_operand" "%0")
                (match_operand:SI 2 "register_operand" "r")))]
  ""
  "xor %2,%0"
  [(set_attr "slottable" "yes")])

(define_insn "xorhi3"
  [(set (match_operand:HI 0 "register_operand" "=r")
        (xor:HI (match_operand:HI 1 "register_operand" "%0")
                (match_operand:HI 2 "register_operand" "r")))]
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "xor %2,%0"
  [(set_attr "slottable" "yes")])

(define_insn "xorqi3"
  [(set (match_operand:QI 0 "register_operand" "=r")
        (xor:QI (match_operand:QI 1 "register_operand" "%0")
                (match_operand:QI 2 "register_operand" "r")))]
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "xor %2,%0"
  [(set_attr "slottable" "yes")])

;
; Negate insns.
;

;
; Better than a library call:
;
(define_expand "negsf2"
  [(set (match_dup 2)
        (const_int -2147483648)) ; 0x80000000
   (parallel [(set (match_operand:SF 0 "register_operand" "=r")
                   (neg:SF (match_operand:SF 1 "register_operand" "0")))
              (use (match_dup 2))])]
  ""
  "operands[2] = gen_reg_rtx(SImode);")

(define_insn ""
  [(set (match_operand:SF 0 "register_operand" "=r")
        (neg:SF (match_operand:SF 1 "register_operand" "0")))
   (use (match_operand:SI 2 "register_operand" "r"))]
  ""
  "xor %2,%0"
  [(set_attr "slottable" "yes")])

;
; The assembler-part is incorrect.
;
;(define_insn "negdi2"
;  [(set (match_operand:DI 0 "register_operand" "=r")
;       (neg:DI (match_operand:DI 1 "register_operand" "r")))]
;  ""
;  "neg.d %M1,%M0\;bne 0f\;not %H1,%H0\;addq 1,%M1\\n0:")

(define_insn "negsi2"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (neg:SI (match_operand:SI 1 "register_operand" "r")))]
  ""
  "neg.d %1,%0"
  [(set_attr "slottable" "yes")])

(define_insn "neghi2"
  [(set (match_operand:HI 0 "register_operand" "=r")
        (neg:HI (match_operand:HI 1 "register_operand" "r")))]
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "neg.w %1,%0"
  [(set_attr "slottable" "yes")])

(define_insn "negqi2"
  [(set (match_operand:QI 0 "register_operand" "=r")
        (neg:QI (match_operand:QI 1 "register_operand" "r")))]
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "neg.b %1,%0"
  [(set_attr "slottable" "yes")])

;
; One-complements
;

; See comment on anddi3

(define_insn "one_cmplsi2"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (not:SI (match_operand:SI 1 "register_operand" "0")))]
  ""
  "not %0"
  [(set_attr "slottable" "yes")])

(define_insn "one_cmplhi2"
  [(set (match_operand:HI 0 "register_operand" "=r")
        (not:HI (match_operand:HI 1 "register_operand" "0")))]
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "not %0"
  [(set_attr "slottable" "yes")])

(define_insn "one_cmplqi2"
  [(set (match_operand:QI 0 "register_operand" "=r")
        (not:QI (match_operand:QI 1 "register_operand" "0")))]
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "not %0"
  [(set_attr "slottable" "yes")])

;
; Arithmetic shift right
;

(define_insn "ashrsi3"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (ashiftrt:SI (match_operand:SI 1 "register_operand" "0")
                     (match_operand:SI 2 "nonmemory_operand" "nr")))]
  ""
  "*{ if (REG_S_P(operands[2]))
        return \"asr.d %2,%0\";
      if (GET_CODE(operands[2]) == CONST_INT && INTVAL(operands[2]) > 31)
        return \"moveq -1,%0\";
      else return \"asrq %2,%0\"; }"
  [(set_attr "slottable" "yes")])

;
; Since gcc gets lost, and forgets to zero-extend the source (or mask
; the destination) when it changes shifts of lower modes into SImode,
; its better to make these expands and anonymous patterns instead of
; the more correct define_insns.  This occurs when gcc thinks that is
; is better to widen to SImode and use immediate shift count.
;

(define_expand "ashrhi3"
  [(set (match_dup 3)
        (sign_extend:SI (match_operand:HI 1 "nonimmediate_operand" "rm")))
   (set (match_dup 4)
        (zero_extend:SI (match_operand:HI 2 "nonimmediate_operand" "rm")))
   (set (match_dup 5) (ashiftrt:SI (match_dup 3) (match_dup 4)))
   (set (match_operand:HI 0 "general_operand" "=g")
        (subreg:HI (match_dup 5) 0))]
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "{ int i; for (i = 3; i < 6; i++) operands[i] = gen_reg_rtx(SImode);}")

(define_insn ""
  [(set (match_operand:HI 0 "register_operand" "=r")
        (ashiftrt:HI (match_operand:HI 1 "register_operand" "0")
                     (match_operand:HI 2 "register_operand" "r")))]
  ""
  "asr.w %2,%0"
  [(set_attr "slottable" "yes")])


(define_insn ""
  [(set (strict_low_part (match_operand:HI 0 "register_operand" "+r"))
        (ashiftrt:HI (match_dup 0)
                     (match_operand:HI 1 "register_operand" "r")))]
  ""
  "asr.w %1,%0"
  [(set_attr "slottable" "yes")])

;

(define_expand "ashrqi3"
  [(set (match_dup 3)
        (sign_extend:SI (match_operand:QI 1 "nonimmediate_operand" "g")))
   (set (match_dup 4)
        (zero_extend:SI (match_operand:QI 2 "nonimmediate_operand" "g")))
   (set (match_dup 5) (ashiftrt:SI (match_dup 3) (match_dup 4)))
   (set (match_operand:QI 0 "general_operand" "=g")
        (subreg:QI (match_dup 5) 0))]
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "{ int i; for (i = 3; i < 6; i++) operands[i] = gen_reg_rtx(SImode);}")

(define_insn ""
  [(set (match_operand:QI 0 "register_operand" "=r")
        (ashiftrt:QI (match_operand:QI 1 "register_operand" "0")
                     (match_operand:QI 2 "register_operand" "r")))]
  ""
  "asr.b %2,%0"
  [(set_attr "slottable" "yes")])

(define_insn ""
  [(set (strict_low_part (match_operand:QI 0 "register_operand" "+r"))
        (ashiftrt:QI (match_dup 0)
                     (match_operand:QI 1 "register_operand" "r")))]
  ""
  "asr.b %1,%0"
  [(set_attr "slottable" "yes")])


;
; Logical shift right
;

(define_insn "lshrsi3"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (lshiftrt:SI (match_operand:SI 1 "register_operand" "0")
                     (match_operand:SI 2 "nonmemory_operand" "nr")))]
  ""
  "*{ if (REG_S_P(operands[2]))
        return \"lsr.d %2,%0\";
      if (GET_CODE(operands[2]) == CONST_INT && INTVAL(operands[2]) > 31)
        return \"moveq 0,%0\";
      else return \"lsrq %2,%0\"; }"
  [(set_attr "slottable" "yes")])

(define_expand "lshrhi3"
  [(set (match_dup 3)
        (zero_extend:SI (match_operand:HI 1 "nonimmediate_operand" "g")))
   (set (match_dup 4)
        (zero_extend:SI (match_operand:HI 2 "nonimmediate_operand" "g")))
   (set (match_dup 5) (lshiftrt:SI (match_dup 3) (match_dup 4)))
   (set (match_operand:HI 0 "general_operand" "=g")
        (subreg:HI (match_dup 5) 0))]
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "{ int i; for (i = 3; i < 6; i++) operands[i] = gen_reg_rtx(SImode);}")

(define_insn ""
  [(set (match_operand:HI 0 "register_operand" "=r")
        (lshiftrt:HI (match_operand:HI 1 "register_operand" "0")
                     (match_operand:HI 2 "register_operand" "r")))]
  ""
  "lsr.w %2,%0"
  [(set_attr "slottable" "yes")])


(define_insn ""
  [(set (strict_low_part (match_operand:HI 0 "register_operand" "+r"))
        (lshiftrt:HI (match_dup 0)
                     (match_operand:HI 1 "register_operand" "r")))]
  ""
  "lsr.w %1,%0"
  [(set_attr "slottable" "yes")])

;

(define_expand "lshrqi3"
  [(set (match_dup 3)
        (zero_extend:SI (match_operand:QI 1 "nonimmediate_operand" "g")))
   (set (match_dup 4)
        (zero_extend:SI (match_operand:QI 2 "nonimmediate_operand" "g")))
   (set (match_dup 5) (lshiftrt:SI (match_dup 3) (match_dup 4)))
   (set (match_operand:QI 0 "general_operand" "=g")
        (subreg:QI (match_dup 5) 0))]
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "{ int i; for (i = 3; i < 6; i++) operands[i] = gen_reg_rtx(SImode);}")

(define_insn ""
  [(set (match_operand:QI 0 "register_operand" "=r")
        (lshiftrt:QI (match_operand:QI 1 "register_operand" "0")
                     (match_operand:QI 2 "register_operand" "r")))]
  ""
  "lsr.b %2,%0"
  [(set_attr "slottable" "yes")])

(define_insn ""
  [(set (strict_low_part (match_operand:QI 0 "register_operand" "+r"))
        (lshiftrt:QI (match_dup 0)
                     (match_operand:QI 1 "register_operand" "r")))]
  ""
  "lsr.b %1,%0"
  [(set_attr "slottable" "yes")])


;
; Arithmetic/logical shift left
;

(define_insn "ashlsi3"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (ashift:SI (match_operand:SI 1 "register_operand" "0")
                   (match_operand:SI 2 "nonmemory_operand" "nr")))]
  ""
  "*{ if (REG_S_P(operands[2]))
        return \"lsl.d %2,%0\";
      if (GET_CODE(operands[2]) == CONST_INT && INTVAL(operands[2]) > 31)
        return \"moveq 0,%0\";
      else return \"lslq %2,%0\"; }"
  [(set_attr "slottable" "yes")])

;
; For narrower modes than SI, we may use lslq (but cc is useless), so
; we dont have to reload shift-count into a reg.
;

(define_insn "ashlhi3"
  [(set (match_operand:HI 0 "register_operand" "=r")
        (ashift:HI (match_operand:HI 1 "register_operand" "0")
                   (match_operand:HI 2 "nonmemory_operand" "nr")))]

  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "*{ return GET_CODE(operands[2]) == CONST_INT
         && INTVAL(operands[2]) > 15 ? \"moveq 0,%0\"
         : CONSTANT_P(operands[2]) ? \"lslq %2,%0\" : \"lsl.w %2,%0\"; }"
  [(set_attr "slottable" "yes")])

(define_insn ""
  [(set (strict_low_part (match_operand:HI 0 "register_operand" "+r"))
        (ashift:HI (match_dup 0)
                   (match_operand:HI 1 "register_operand" "r")))]
  ""
  "lsl.w %1,%0"
  [(set_attr "slottable" "yes")])

(define_insn "ashlqi3"
  [(set (match_operand:QI 0 "register_operand" "=r")
        (ashift:QI (match_operand:QI 1 "register_operand" "0")
                     (match_operand:QI 2 "nonmemory_operand" "nr")))]
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "*{ return GET_CODE(operands[2]) == CONST_INT
         && INTVAL(operands[2]) > 7 ? \"moveq 0,%0\"
         : CONSTANT_P(operands[2]) ? \"lslq %2,%0\" : \"lsl.b %2,%0\"; }"
  [(set_attr "slottable" "yes")])

(define_insn ""
  [(set (strict_low_part (match_operand:QI 0 "register_operand" "+r"))
        (ashift:QI (match_dup 0)
                     (match_operand:QI 1 "register_operand" "r")))]
  ""
  "lsl.b %1,%0"
  [(set_attr "slottable" "yes")])

;
; Various strange insns that gcc likes
;
;
; Note: abs only works in si (dword) mode, so we cannot define abshi2
; or absqi2.

;
; Fortunately, it is simple to construct an abssf
;
(define_insn "abssf2"
  [(set (match_operand:SF 0 "register_operand" "=r")
        (abs:SF (match_operand:SF 1 "register_operand" "0")))]
  ""
  "lslq 1,%0\;lsrq 1,%0")

(define_insn "abssi2"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (abs:SI (match_operand:SI 1 "register_operand" "r")))]
  ""
  "abs %1,%0"
  [(set_attr "slottable" "yes")])

(define_expand "abshi2"
  [(set (match_dup 2)
        (sign_extend:SI (match_operand:HI 1 "general_operand" "g")))
   (set (match_dup 3) (abs:SI (match_dup 2)))
   (set (match_operand:HI 0 "register_operand" "=r")
        (subreg:HI (match_dup 3) 0))]
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "operands[2] = gen_reg_rtx(SImode); operands[3] = gen_reg_rtx(SImode);")

(define_expand "absqi2"
  [(set (match_dup 2)
        (sign_extend:SI (match_operand:QI 1 "general_operand" "g")))
   (set (match_dup 3) (abs:SI (match_dup 2)))
   (set (match_operand:QI 0 "register_operand" "=r")
        (subreg:QI (match_dup 3) 0))]
  "TARGET_SMALL_ARIT" ; maybe we should sign/zero_extend the operand instead?
  "operands[2] = gen_reg_rtx(SImode); operands[3] = gen_reg_rtx(SImode);")

;
; Bound-insn.  Arithmetically defined to be the same as an unsigned
; minimum, which is an operation supported by gcc.  Used in casesi.
;
; May be useful too in insn combination; therefore the generalized pattern.
;
(define_insn "uminsi3"
  [(set (match_operand:SI 0 "register_operand"           "=r,r,r,r")
        (umin:SI  (match_operand:SI 1 "register_operand" "%0,0,0,r")
                  (match_operand:SI 2 "nonmemory_operand" "r,Q>,g,!To")))]
                                        ; Inhibit reloading to "To" alternative
  ""
  "*
{
  if (GET_CODE(operands[2]) == CONST_INT) {
    if (INTVAL(operands[2]) < 256)
      return \"bound.b %2,%0\";
    if (INTVAL(operands[2]) < 65536)
      return \"bound.w %2,%0\";
    return \"bound.d %2,%0\";
  } else if (which_alternative == 3)
      return \"bound.d %2,%1,%0\";
    else
      return \"bound.d %2,%0\";
}"
 [(set_attr "slottable" "yes,yes,no,no")])

;
; Jump and branch insns.
;
(define_insn "jump"
  [(set (pc)
        (label_ref (match_operand 0 "" "")))]
  ""
  "ba %l0%#"
  [(set_attr "slottable" "branch")])

;
; It seems we have to pretend a move from r15 to pc in order to
; convince gcc that a jump is done. (and match ordinary arithmetic insns)
;  There used to be a tm.h macro "PC_REGNUM", but that isnt used by
; gcc any more (v2.1).  We have to tell gcc explicitly when we jump.
; Pity.  Remove such dummy jumps with this insn.
;  Another use for this is to put out the "use" marks, so the
; jump-table or the default code wont be removed as dead code.
;  The default-case mark is used to output that label last in the
; table using ASM_OUTPUT_CASE_END.  If you change the order of the
; insns in define_expand casesi, you have to change ASM_OUTPUT_CASE_END too. 
;
(define_insn "dummy_jump"
  [(set (pc) (reg:SI 15))
   (use (label_ref (match_operand:SI 0 "" "")))
   (use (label_ref (match_operand:SI 1 "" "")))]
  ""
  "")

;
; It seems more appropriate to tell gcc that "r15 comes from pc" than
; to make gcc confused by the seemingly out-of-the-air use of r15.
;
(define_insn "dummy_jump2"
  [(set (reg:SI 15) (pc))]
  ""
  "")

(define_insn "indirect_jump"
  [(set (pc) (match_operand:SI 0 "general_operand" "g"))]
  ""
  "jump %R0")

;
; Return insn - use whenever the epilogue is very simple;
;  if it is only a single ret or jump [sp+] (no saved registers or allocated
; stack space)
;  In this pattern, although named, it is legal to check for the
; context in the test, not only on compiler switches.
;
;
(define_insn "return"
  [(return)]
  "simple_epilogue()"
  "*
{
  int i;

  /* Must be able to return a pointer to this */
  static char rd[80];

  *rd = 0;

  for (i = 8; i >= 0; i--)
    if (regs_ever_live[i])
       break;
  if (i >= 0) 
    sprintf(rd,\"movem [sp+],r%d\",i);

  if (regs_ever_live[SRP_REGNUM]) {
    if (*rd)
      output_asm_insn(rd,operands);
    return \"jump [sp+]\";
  }
  if (*rd) {
    output_asm_insn(\"reT\",operands);
    return rd;
  }
  return \"ret%#\";
}"
  [(set (attr "slottable" )
        (if_then_else
         (ne (symbol_ref "regs_ever_live[SRP_REGNUM]") (const_int 0))
         (const_string "no")         ; If jump then not slottable.
         (if_then_else
          (ne (symbol_ref "regs_ever_live[0]") (const_int 0))
          (const_string "no") ; ret+movem [sp+],rx
          (const_string "branch"))))]) ; If ret then delayed.

;
; Conditional branches.
;

(define_insn "beq"
  [(set (pc)
        (if_then_else (eq (cc0)
                          (const_int 0))
                      (label_ref (match_operand 0 "" ""))
                      (pc)))]
  ""
  "beq %l0%#"
  [(set_attr "slottable" "branch")])

(define_insn "bne"
  [(set (pc)
        (if_then_else (ne (cc0)
                          (const_int 0))
                      (label_ref (match_operand 0 "" ""))
                      (pc)))]
  ""
  "bne %l0%#"
  [(set_attr "slottable" "branch")])

(define_insn "bgt"
  [(set (pc)
        (if_then_else (gt (cc0)
                          (const_int 0))
                      (label_ref (match_operand 0 "" ""))
                      (pc)))]
  ""
  "*{ return (cc_prev_status.flags & CC_NO_OVERFLOW)
             ?  0 : \"bgt %l0%#\";}"
  [(set_attr "slottable" "branch")])

(define_insn "bgtu"
  [(set (pc)
        (if_then_else (gtu (cc0)
                           (const_int 0))
                      (label_ref (match_operand 0 "" ""))
                      (pc)))]
  ""
  "bhi %l0%#"
  [(set_attr "slottable" "branch")])

(define_insn "blt"
  [(set (pc)
        (if_then_else (lt (cc0)
                          (const_int 0))
                      (label_ref (match_operand 0 "" ""))
                      (pc)))]
  ""
  "*{ return (cc_prev_status.flags & CC_NO_OVERFLOW)
             ?  \"bmi %l0%#\" : \"blt %l0%#\";}"
  [(set_attr "slottable" "branch")])

(define_insn "bltu"
  [(set (pc)
        (if_then_else (ltu (cc0)
                           (const_int 0))
                      (label_ref (match_operand 0 "" ""))
                      (pc)))]
  ""
  "blo %l0%#"
  [(set_attr "slottable" "branch")])

(define_insn "bge"
  [(set (pc)
        (if_then_else (ge (cc0)
                          (const_int 0))
                      (label_ref (match_operand 0 "" ""))
                      (pc)))]
  ""
  "*{ return (cc_prev_status.flags & CC_NO_OVERFLOW)
             ?  \"bpl %l0%#\" : \"bge %l0%#\";}"
  [(set_attr "slottable" "branch")])

(define_insn "bgeu"
  [(set (pc)
        (if_then_else (geu (cc0)
                           (const_int 0))
                      (label_ref (match_operand 0 "" ""))
                      (pc)))]
  ""
  "bhs %l0%#"
  [(set_attr "slottable" "branch")])

(define_insn "ble"
  [(set (pc)
        (if_then_else (le (cc0)
                          (const_int 0))
                      (label_ref (match_operand 0 "" ""))
                      (pc)))]
  ""
  "*{ return (cc_prev_status.flags & CC_NO_OVERFLOW)
             ?  0 : \"ble %l0%#\";}"
  [(set_attr "slottable" "branch")])

(define_insn "bleu"
  [(set (pc)
        (if_then_else (leu (cc0)
                           (const_int 0))
                      (label_ref (match_operand 0 "" ""))
                      (pc)))]
  ""
  "bls %l0%#"
  [(set_attr "slottable" "branch")])

;
; Reversed anonymous patterns to the ones above.
;

(define_insn ""
  [(set (pc)
        (if_then_else (eq (cc0)
                          (const_int 0))
                      (pc)
                      (label_ref (match_operand 0 "" ""))))]
  ""
  "bne %l0%#"
  [(set_attr "slottable" "branch")])

(define_insn ""
  [(set (pc)
        (if_then_else (ne (cc0)
                          (const_int 0))
                      (pc)
                      (label_ref (match_operand 0 "" ""))))]
  ""
  "beq %l0%#"
  [(set_attr "slottable" "branch")])

(define_insn ""
  [(set (pc)
        (if_then_else (gt (cc0)
                          (const_int 0))
                      (pc)
                      (label_ref (match_operand 0 "" ""))))]
  ""
  "*{ return (cc_prev_status.flags & CC_NO_OVERFLOW)
             ?  0 : \"ble %l0%#\";}"
  [(set_attr "slottable" "branch")])

(define_insn ""
  [(set (pc)
        (if_then_else (gtu (cc0)
                           (const_int 0))
                      (pc)
                      (label_ref (match_operand 0 "" ""))))]
  ""
  "bls %l0%#"
  [(set_attr "slottable" "branch")])

(define_insn ""
  [(set (pc)
        (if_then_else (lt (cc0)
                          (const_int 0))
                      (pc)
                      (label_ref (match_operand 0 "" ""))))]
  ""
  "*{ return (cc_prev_status.flags & CC_NO_OVERFLOW)
             ?  \"bpl %l0%#\" : \"bge %l0%#\";}"
  [(set_attr "slottable" "branch")])

(define_insn ""
  [(set (pc)
        (if_then_else (ltu (cc0)
                           (const_int 0))
                      (pc)
                      (label_ref (match_operand 0 "" ""))))]
  ""
  "bhs %l0%#"
  [(set_attr "slottable" "branch")])

(define_insn ""
  [(set (pc)
        (if_then_else (ge (cc0)
                          (const_int 0))
                      (pc)
                      (label_ref (match_operand 0 "" ""))))]
  ""
  "*{ return (cc_prev_status.flags & CC_NO_OVERFLOW)
             ?  \"bmi %l0%#\" : \"blt %l0%#\";}"
  [(set_attr "slottable" "branch")])

(define_insn ""
  [(set (pc)
        (if_then_else (geu (cc0)
                           (const_int 0))
                      (pc)
                      (label_ref (match_operand 0 "" ""))))]
  ""
  "blo %l0%#"
  [(set_attr "slottable" "branch")])

(define_insn ""
  [(set (pc)
        (if_then_else (le (cc0)
                          (const_int 0))
                      (pc)
                      (label_ref (match_operand 0 "" ""))))]
  ""
  "*{ return (cc_prev_status.flags & CC_NO_OVERFLOW)
             ?  0 : \"bgt %l0%#\";}"
  [(set_attr "slottable" "branch")])

(define_insn ""
  [(set (pc)
        (if_then_else (leu (cc0)
                           (const_int 0))
                      (pc)
                      (label_ref (match_operand 0 "" ""))))]
  ""
  "bhi %l0%#"
  [(set_attr "slottable" "branch")])

;
; SET on condition : Scc
;

(define_insn "sgeu"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (geu (cc0) (const_int 0)))]
  ""
  "shs %0"
  [(set_attr "slottable" "yes")])

(define_insn "sltu"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (ltu (cc0) (const_int 0)))]
  ""
  "slo %0"
  [(set_attr "slottable" "yes")])

(define_insn "seq"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (eq (cc0) (const_int 0)))]
  ""
  "seq %0"
  [(set_attr "slottable" "yes")])

(define_insn "sge"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (ge (cc0) (const_int 0)))]
  ""
  "*{ return (cc_prev_status.flags & CC_NO_OVERFLOW)
             ?  \"spl %0\" : \"sge %0\";}"
  [(set_attr "slottable" "yes")])

(define_insn "sgt"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (gt (cc0) (const_int 0)))]
  ""
  "*{ return (cc_prev_status.flags & CC_NO_OVERFLOW)
             ?  0 : \"sgt %0\";}"
  [(set_attr "slottable" "yes")])

(define_insn "sgtu"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (gtu (cc0) (const_int 0)))]
  ""
  "shi %0"
  [(set_attr "slottable" "yes")])

(define_insn "sle"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (le (cc0) (const_int 0)))]
  ""
  "*{ return (cc_prev_status.flags & CC_NO_OVERFLOW)
             ?  0 : \"sle %0\";}"
  [(set_attr "slottable" "yes")])

(define_insn "sleu"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (leu (cc0) (const_int 0)))]
  ""
  "sls %0"
  [(set_attr "slottable" "yes")])

(define_insn "slt"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (lt (cc0) (const_int 0)))]
  ""
  "*{ return (cc_prev_status.flags & CC_NO_OVERFLOW)
             ?  \"smi %0\" : \"slt %0\";}"
  [(set_attr "slottable" "yes")])

(define_insn "sne"
  [(set (match_operand:SI 0 "register_operand" "=r")
        (ne (cc0) (const_int 0)))]
  ""
  "sne %0"
  [(set_attr "slottable" "yes")])

;
; CALL insns
;
;
; We need to make these patterns "expand", since the real operand is
; hidden in a (mem:QI ) inside operand[0] (call_value: operand[1]),
; and cannot be checked if it were a "normal" pattern.
;  I believe someone goofed when it comes to this mem-kind hiding,
; since it totally stops validation of operands on machines where the
; call-insn does NOT "automatically" strip the (mem) from its operands.
;  Note that "call" and "call_value" are ALWAYS called with a
; mem-operand for operand 0 and 1 respective.  What happens in cse and
; instruction combination is another issue.
;
(define_expand "call"
  [(parallel [(call (match_operand:QI 0 "memory_operand" "")
	            (match_operand 1 "general_operand" ""))
	      (clobber (reg:SI 16))])] ;; 16 is the srp (can't use symbolic name)
  ""
  "")

;
; Accept ANYTHING as operand 1.  Accept operands for operand 0 in
; order of preference (Q includes r, but r is shorter, faster)
;
(define_insn ""
  [(call (mem:QI (match_operand:SI 0 "general_operand" "r,Q>,g"))
         (match_operand 1 "" ""))
   (clobber (reg:SI 16))] ;; 16 is the srp (can't use symbolic name)
  ""
  "jsr %0")

(define_expand "call_value"
  [(parallel [(set (match_operand 0 "" "")
	           (call (match_operand:QI 1 "memory_operand" "")
		         (match_operand 2 "" "")))
	      (clobber (reg:SI 16))])] ;; 16 is the srp (can't use symbolic name)
  ""
  "")

;
; Accept ANYTHING as operand 2.  The validity other than "general" of
; operand 0 will be checked elsewhere.  Accept operands for operand 1 in
; order of preference (Q includes r, but r is shorter, faster)
;
(define_insn ""
  [(set (match_operand 0 "general_operand" "=g,g,g")
        (call (mem:QI (match_operand:SI 1 "general_operand" "r,Q>,g"))
              (match_operand 2 "" "")))
   (clobber (reg:SI 16))]
  ""
  "Jsr %1")

;
; Used in debugging.  No usage for the direct pattern; unfilled
; delayed-branches are taken care of in other ways.
;
(define_insn "nop"
  [(const_int 0)]
  ""
  "")

;( define_insn "tablejump"
;  [(set (pc) (match_operand:SI 0 "general_operand" "g"))
;   (use (label_ref (match_operand 1 "" "")))]
;  ""
;  "Jump %0")

;
; No, let's use casesi instead. Will save a few cycles.
; Change back if "bound" insn is removed.
;
; NOTE: if you change the "parallell" (or add anything after it) in
; this expansion, you must change the macro ASM_OUTPUT_CASE_END
; accordingly, to add the default case at the end of the jump-table.
;
(define_expand "casesi"
  [(set (match_dup 5) (match_operand:SI 0 "general_operand" ""))
   (set (match_dup 6)
        (minus:SI (match_dup 5)
                  (match_operand:SI 1 "const_int_operand" "n")))
   (set (match_dup 7)
        (umin:SI (match_dup 6)
                 (match_operand:SI 2 "const_int_operand" "n")))
   (set (reg:SI 15) (pc))
   (set (reg:SI 15)
        (plus:SI (sign_extend:SI (mem:HI
                                  (plus:SI (mult:SI (match_dup 7)
                                                    (const_int 2))
                                           (reg:SI 15))))
                 (reg:SI 15)))
   (parallel
    [(set (pc) (reg:SI 15))
     (use (label_ref (match_operand 3 "" "")))
     (use (label_ref (match_operand 4 "" "")))])]
  ""
  "{ operands[2] = plus_constant(operands[2],1);
     operands[5] = gen_reg_rtx(SImode);
     operands[6] = gen_reg_rtx(SImode);
     operands[7] = gen_reg_rtx(SImode); }")

;
; Splitting patterns.  Some of them have modes unspecified.  This
; should always be ok.  (If you don't agree, then sparc.md has it as
; well). 
;
; op [rx],ry,rz
; Note the absence of index on rx. This happens at end of routines
; when op:ing together some values to end up in rz.

;
; Recognize  bound [rx],ry,rz; [rx] is zero_extended
; and add/sub [rx],ry,rz zero or sign extend on [r]
;  Change into move ry,rz; op [rx],rz
;  Lose if rz=ry or rx=rz
(define_split
  [(set (match_operand 0 "register_operand" "=r")
        (match_operator
         4 "operand_extend_operator"
         [(match_operand 1 "register_operand" "r")
          (match_operator
           3 "extend_operator"
           [(match_operand 2 "memory_operand" "m")])]))]
  "REGNO(operands[1]) != REGNO(operands[0])
   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
   && (GET_CODE(XEXP(operands[2],0)) == REG)
   && REGNO(XEXP(operands[2],0)) != REGNO(operands[0])"
  [(set (match_dup 0)
        (match_dup 1))
   (set (match_dup 0)
        (match_op_dup
         4 [(match_dup 0)
            (match_op_dup 3 [(match_dup 2)])]))]
  "")

;
; As above, but change op [rz],ry,rz into ext [rz],rz; op ry,rz
; plus or bound only.
;
(define_split
  [(set (match_operand 0 "register_operand" "=r")
        (match_operator
         4 "plus_or_bound_operator"
         [(match_operand 1 "register_operand" "r")
          (match_operator
           3 "extend_operator"
           [(match_operand 2 "memory_operand" "m")])]))]
  "REGNO(operands[1]) != REGNO(operands[0])
   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
   && (GET_CODE(XEXP(operands[2],0)) == REG)
   && REGNO(XEXP(operands[2],0)) == REGNO(operands[0])"
  [(set (match_dup 0)
        (match_op_dup 3 [(match_dup 2)]))
   (set (match_dup 0)
        (match_op_dup
         4 [(match_dup 0)
            (match_dup 1)]))]
  "")
;
; As above, but swapped operands for add and bound
;
(define_split
  [(set (match_operand 0 "register_operand" "=r")
        (match_operator
         4 "plus_or_bound_operator"
         [(match_operator
           3 "extend_operator"
           [(match_operand 2 "memory_operand" "m")])
          (match_operand 1 "register_operand" "r")]))]
  "REGNO(operands[1]) != REGNO(operands[0])
   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
   && (GET_CODE(XEXP(operands[2],0)) == REG)
   && REGNO(XEXP(operands[2],0)) != REGNO(operands[0])"
  [(set (match_dup 0)
        (match_dup 1))
   (set (match_dup 0)
        (match_op_dup
         4 [(match_dup 0)
            (match_op_dup 3 [(match_dup 2)])]))]
  "")
;
; As above, but change op [rz],ry,rz into ext [rz],rz; op ry,rz
; plus or bound only, of course.
;
(define_split
  [(set (match_operand 0 "register_operand" "=r")
        (match_operator
         4 "plus_or_bound_operator"
         [(match_operator
           3 "extend_operator"
           [(match_operand 2 "memory_operand" "m")])
          (match_operand 1 "register_operand" "r")]))]
  "REGNO(operands[1]) != REGNO(operands[0])
   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
   && (GET_CODE(XEXP(operands[2],0)) == REG)
   && REGNO(XEXP(operands[2],0)) == REGNO(operands[0])"
  [(set (match_dup 0)
        (match_op_dup 3 [(match_dup 2)]))
   (set (match_dup 0)
        (match_op_dup
         4 [(match_dup 0)
            (match_dup 1)]))]
  "")

;
;
; Recognize as above, but the mem operand is not extended.
;
; op [rx],ry,rz changed into move ry,rz; op [rx],rz,rz
; lose if ry=rz or rx=rz
;
(define_split
  [(set (match_operand 0 "register_operand" "=r")
        (match_operator
         3 "orthogonal_operator"
         [(match_operand 1 "register_operand" "r")
          (match_operand 2 "memory_operand" "m")]))]
  "REGNO(operands[1]) != REGNO(operands[0])
   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
   && (GET_CODE(XEXP(operands[2],0)) == REG)
   && REGNO(XEXP(operands[2],0)) != REGNO(operands[0])"
  [(set (match_dup 0)
        (match_dup 1))
   (set (match_dup 0)
        (match_op_dup
         3 [(match_dup 0)
            (match_dup 2)]))]
  "")                 
;
; Above, but take care of op [rz],ry,rz into move [rz],rz; op ry,rz,rz
; Only commutative operators.
;
(define_split
  [(set (match_operand 0 "register_operand" "=r")
        (match_operator
         3 "commutative_orth_op"
         [(match_operand 1 "register_operand" "r")
          (match_operand 2 "memory_operand" "m")]))]
  "REGNO(operands[1]) != REGNO(operands[0])
   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
   && (GET_CODE(XEXP(operands[2],0)) == REG)
   && REGNO(XEXP(operands[2],0)) == REGNO(operands[0])"
  [(set (match_dup 0)
        (match_dup 2))
   (set (match_dup 0)
        (match_op_dup
         3 [(match_dup 0)
            (match_dup 1)]))]
  "")                 

;
; As above, but commutative operators.
;
(define_split
  [(set (match_operand 0 "register_operand" "=r")
        (match_operator
         3 "commutative_orth_op"
         [(match_operand 2 "memory_operand" "m")
          (match_operand 1 "register_operand" "r")]))]
  "REGNO(operands[1]) != REGNO(operands[0])
   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
   && (GET_CODE(XEXP(operands[2],0)) == REG)"
  [(set (match_dup 0)
        (match_dup 1))
   (set (match_dup 0)
        (match_op_dup
         3 [(match_dup 0)
            (match_dup 2)]))]
  "")

;
; Again as above for rx=rz
;
(define_split
  [(set (match_operand 0 "register_operand" "=r")
        (match_operator
         3 "commutative_orth_op"
         [(match_operand 2 "memory_operand" "m")
          (match_operand 1 "register_operand" "r")]))]
  "REGNO(operands[1]) != REGNO(operands[0])
   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
   && (GET_CODE(XEXP(operands[2],0)) == REG)
   && REGNO(XEXP(operands[2],0)) == REGNO(operands[0])"
  [(set (match_dup 0)
        (match_dup 2))
   (set (match_dup 0)
        (match_op_dup
         3 [(match_dup 0)
            (match_dup 1)]))]
  "")                 

;
; Splits for all cases where (possibly after reload and local/global
; register allocation) rx and ry in [rx=ry+i] are equal in side-effect insns.
;
(define_split
  [(parallel
    [(set (match_operand 0 "register_operand" "=r,r")
          (mem (plus:SI (mult:SI (match_operand:SI 1 "register_operand" "r,r")
                                 (match_operand:SI 2 "const_int_operand" "n,n"))
                      (match_operand:SI 3 "register_operand" "r,r"))))
     (set (match_operand:SI 4 "register_operand" "=3,r")
          (plus:SI (mult:SI (match_dup 1)
                            (match_dup 2))
                   (match_dup 3)))])]
  "REGNO(operands[3]) == REGNO(operands[4])"
  [(set (match_dup 4) (plus:SI (mult:SI (match_dup 1) (match_dup 2))
                               (match_dup 3)))
   (set (match_dup 0) (match_dup 5))]
  "operands[5] = gen_rtx(MEM,GET_MODE(operands[0]),operands[3]);")

(define_split
  [(parallel
    [(set (match_operand 0 "register_operand" "=r,r,r")
          (mem (plus:SI (match_operand:SI 1 "bdap_operand" "%r,r,r")
                        (match_operand:SI 2 "bdap_operand" "r>Ri,r,>Ri"))))
     (set (match_operand:SI 3 "register_operand" "=1,r,r")
          (plus:SI (match_dup 1)
                   (match_dup 2)))])]
  "(rtx_equal_p(operands[3],operands[1])
    || rtx_equal_p(operands[3],operands[2]))"
  [(set (match_dup 3) (plus:SI (match_dup 1) (match_dup 2)))
   (set (match_dup 0) (match_dup 4))]
  "operands[4] = gen_rtx(MEM,GET_MODE(operands[0]),operands[3]);")

(define_split
  [(parallel
    [(set (mem (plus:SI (mult:SI (match_operand:SI 0 "register_operand" "r,r")
                                 (match_operand:SI 1 "const_int_operand" "n,n"))
                        (match_operand:SI 2 "register_operand" "r,r")))
          (match_operand 3 "register_operand" "r,r"))
     (set (match_operand:SI 4 "register_operand" "=2,r")
          (plus:SI (mult:SI (match_dup 0)
                            (match_dup 1))
                   (match_dup 2)))])]
  "REGNO(operands[4]) == REGNO(operands[2])"
  [(set (match_dup 4) (plus:SI (mult:SI (match_dup 0) (match_dup 1))
                               (match_dup 2)))
   (set (match_dup 5) (match_dup 3))]
  "operands[5] = gen_rtx(MEM,GET_MODE(operands[3]),operands[4]);")

(define_split
  [(parallel
    [(set (mem (plus:SI (match_operand:SI 0 "bdap_operand" "%r,r,r")
                        (match_operand:SI 1 "bdap_operand" "r>Ri,r,>Ri")))
          (match_operand 2 "register_operand" "r,r,r"))
     (set (match_operand:SI 3 "register_operand" "=0,r,r")
          (plus:SI (match_dup 0)
                  (match_dup 1)))])]
  "(rtx_equal_p(operands[3],operands[0])
    || rtx_equal_p(operands[3],operands[1]))"
  [(set (match_dup 3) (plus:SI (match_dup 0) (match_dup 1)))
   (set (match_dup 5) (match_dup 2))]
  "operands[5] = gen_rtx(MEM,GET_MODE(operands[2]),operands[3]);")

(define_split
  [(parallel
    [(set (mem:SI (plus:SI (mult:SI (match_operand:SI 0 "register_operand" "r,r")
                                    (match_operand:SI 1 "const_int_operand" "n,n"))
                           (match_operand:SI 2 "register_operand" "r,r")))
          (const_int 0))
     (set (match_operand:SI 3 "register_operand" "=2,r")
          (plus:SI (mult:SI (match_dup 0)
                            (match_dup 1))
                   (match_dup 2)))])]
  "REGNO(operands[3]) == REGNO(operands[2])"
  [(set (match_dup 3) (plus:SI (mult:SI (match_dup 0) (match_dup 1))
                               (match_dup 2)))
   (set (mem:SI (match_dup 3)) (const_int 0))]
  "")

(define_split
  [(parallel
    [(set (mem:HI (plus:SI (mult:SI (match_operand:SI 0 "register_operand" "r,r")
                                    (match_operand:SI 1 "const_int_operand" "n,n"))
                           (match_operand:SI 2 "register_operand" "r,r")))
          (const_int 0))
     (set (match_operand:SI 3 "register_operand" "=2,r")
          (plus:SI (mult:SI (match_dup 0)
                            (match_dup 1))
                   (match_dup 2)))])]
  "REGNO(operands[3]) == REGNO(operands[2])"
  [(set (match_dup 3) (plus:SI (mult:SI (match_dup 0) (match_dup 1))
                               (match_dup 2)))
   (set (mem:HI (match_dup 3)) (const_int 0))]
  "")

(define_split
  [(parallel
    [(set (mem:QI (plus:SI (mult:SI (match_operand:SI 0 "register_operand" "r,r")
                                    (match_operand:SI 1 "const_int_operand" "n,n"))
                           (match_operand:SI 2 "register_operand" "r,r")))
          (const_int 0))
     (set (match_operand:SI 3 "register_operand" "=2,r")
          (plus:SI (mult:SI (match_dup 0)
                            (match_dup 1))
                   (match_dup 2)))])]
  "REGNO(operands[3]) == REGNO(operands[2])"
  [(set (match_dup 3) (plus:SI (mult:SI (match_dup 0) (match_dup 1))
                               (match_dup 2)))
   (set (mem:QI (match_dup 3)) (const_int 0))]
  "")

(define_split
  [(parallel
    [(set (mem:SI (plus:SI (match_operand:SI 0 "bdap_operand" "%r,r,r")
                           (match_operand:SI 1 "bdap_operand" "r>Ri,r,>Ri")))
          (const_int 0))
     (set (match_operand:SI 2 "register_operand" "=0,r,r")
          (plus:SI (match_dup 0)
                   (match_dup 1)))])]
  "(rtx_equal_p(operands[0],operands[2])
    || rtx_equal_p(operands[2],operands[1]))"
    [(set (match_dup 2) (plus:SI (match_dup 0) (match_dup 1)))
     (set (mem:SI (match_dup 2)) (const_int 0))]
    "")

(define_split
  [(parallel
    [(set (mem:HI (plus:SI (match_operand:SI 0 "bdap_operand" "%r,r,r")
                           (match_operand:SI 1 "bdap_operand" "r>Ri,r,>Ri")))
          (const_int 0))
     (set (match_operand:SI 2 "register_operand" "=0,r,r")
          (plus:SI (match_dup 0)
                   (match_dup 1)))])]
    "(rtx_equal_p(operands[0],operands[2])
     || rtx_equal_p(operands[2],operands[1]))"
    [(set (match_dup 2) (plus:SI (match_dup 0) (match_dup 1)))
     (set (mem:HI (match_dup 2)) (const_int 0))]
    "")

(define_split
  [(parallel
    [(set (mem:QI (plus:SI (match_operand:SI 0 "bdap_operand" "%r,r,r")
                           (match_operand:SI 1 "bdap_operand" "r>Ri,r,>Ri")))
          (const_int 0))
     (set (match_operand:SI 2 "register_operand" "=0,r,r")
          (plus:SI (match_dup 0)
                   (match_dup 1)))])]
    "(rtx_equal_p(operands[0],operands[2])
      || rtx_equal_p(operands[2],operands[1]))"
    [(set (match_dup 2) (plus:SI (match_dup 0) (match_dup 1)))
     (set (mem:QI (match_dup 2)) (const_int 0))]
    "")

(define_split
  [(parallel
    [(set (match_operand 0 "register_operand" "=r,r")
          (match_operator
           5 "extend_operator"
           [(mem (plus:SI (mult:SI (match_operand:SI 1 "register_operand" "r,r")
                                   (match_operand:SI 2 "const_int_operand" "n,n"))
                          (match_operand:SI 3 "register_operand" "r,r")))]))
     (set (match_operand:SI 4 "register_operand" "=3,r")
          (plus:SI (mult:SI (match_dup 1)
                            (match_dup 2))
                   (match_dup 3)))])]
  "REGNO(operands[3]) == REGNO(operands[4])"
  [(set (match_dup 4) (plus:SI (mult:SI (match_dup 1) (match_dup 2))
                               (match_dup 3)))
   (set (match_dup 0) (match_op_dup 5 [(match_dup 6)]))]
  "operands[6] = gen_rtx(MEM,GET_MODE(XEXP(operands[5],0)),operands[4]);")

(define_split
  [(parallel
    [(set (match_operand 0 "register_operand" "=r,r,r")
          (match_operator
           4 "extend_operator"
           [(mem (plus:SI (match_operand:SI 1 "bdap_operand" "%r,r,r")
                          (match_operand:SI 2 "bdap_operand" "r>Ri,r,>Ri")))]))
     (set (match_operand:SI 3 "register_operand" "=1,r,r")
          (plus:SI (match_dup 1)
                   (match_dup 2)))])]
  "(rtx_equal_p(operands[1],operands[3])
    || rtx_equal_p(operands[2],operands[3]))"
  [(set (match_dup 3) (plus:SI (match_dup 1) (match_dup 2)))
   (set (match_dup 0) (match_op_dup 4 [(match_dup 5)]))]
  "operands[5] = gen_rtx(MEM,GET_MODE(XEXP(operands[4],0)),operands[3]);")

(define_split
  [(parallel
    [(set (match_operand 0 "register_operand" "=r,r,r")
          (match_operator
           5 "orthogonal_operator"
           [(match_operand 1 "register_operand" "0,0,0")
            (mem (plus:SI (match_operand:SI 2 "bdap_operand" "%r,r,r")
                          (match_operand:SI 3 "bdap_operand" "r>Ri,r,>Ri")))]))
     (set (match_operand:SI 4 "register_operand" "=2,r,r")
          (plus:SI (match_dup 2)
                   (match_dup 3)))])]
  "(rtx_equal_p(operands[4],operands[2])
   || rtx_equal_p(operands[4],operands[3]))"
  [(set (match_dup 4) (plus:SI (match_dup 2) (match_dup 3)))
   (set (match_dup 0) (match_op_dup 5 [(match_dup 1) (match_dup 6)]))]
  "operands[6] = gen_rtx(MEM,GET_MODE(operands[0]),operands[4]);")

(define_split
  [(parallel
    [(set (match_operand 0 "register_operand" "=r,r")
          (match_operator
           6 "orthogonal_operator"
           [(match_operand 1 "register_operand" "0,0")
            (mem (plus:SI (mult:SI (match_operand:SI 2 "register_operand" "r,r")
                                   (match_operand:SI 3 "const_int_operand" "n,n"))
                          (match_operand:SI 4 "register_operand" "r,r")))]))
     (set (match_operand:SI 5 "register_operand" "=4,r")
          (plus:SI (mult:SI (match_dup 2)
                            (match_dup 3))
                  (match_dup 4)))])]
  "REGNO(operands[5]) == REGNO(operands[4])"
  [(set (match_dup 5) (plus:SI (mult:SI (match_dup 2) (match_dup 3))
                               (match_dup 4)))
   (set (match_dup 0) (match_op_dup 6 [(match_dup 1) (match_dup 7)]))]
  "operands[7] = gen_rtx(MEM,GET_MODE(operands[0]),operands[5]);")

(define_split
  [(parallel
    [(set (match_operand 0 "register_operand" "=r,r")
          (match_operator
           6 "commutative_orth_op"
           [(mem (plus:SI (mult:SI (match_operand:SI 2 "register_operand" "r,r")
                                  (match_operand:SI 3 "const_int_operand" "n,n"))
                          (match_operand:SI 4 "register_operand" "r,r")))
            (match_operand 1 "register_operand" "0,0")]))
     (set (match_operand:SI 5 "register_operand" "=4,r")
          (plus:SI (mult:SI (match_dup 2)
                            (match_dup 3))
                   (match_dup 4)))])]
  "REGNO(operands[5]) == REGNO(operands[4])"
  [(set (match_dup 5) (plus:SI (mult:SI (match_dup 2) (match_dup 3))
                               (match_dup 4)))
   (set (match_dup 0) (match_op_dup 6 [(match_dup 7) (match_dup 1)]))]
  "operands[7] = gen_rtx(MEM,GET_MODE(operands[0]),operands[5]);")

(define_split
  [(parallel
    [(set (match_operand 0 "register_operand" "=r,r,r")
          (match_operator
           5 "commutative_orth_op"
           [(mem (plus:SI (match_operand:SI 2 "bdap_operand" "%r,r,r")
                          (match_operand:SI 3 "bdap_operand" "r>Ri,r,>Ri")))
            (match_operand 1 "register_operand" "0,0,0")]))
     (set (match_operand:SI 4 "register_operand" "=2,r,r")
          (plus:SI (match_dup 2)
                   (match_dup 3)))])]    
  "(rtx_equal_p(operands[4],operands[2])
    || rtx_equal_p(operands[4],operands[3]))"
  [(set (match_dup 4) (plus:SI (match_dup 2) (match_dup 3)))
   (set (match_dup 0) (match_op_dup 5 [(match_dup 6) (match_dup 1)]))]
  "operands[6] = gen_rtx(MEM,GET_MODE(operands[0]),operands[4]);")

(define_split
  [(parallel
    [(set (match_operand 0 "register_operand" "=r,r")
          (match_operator
           6 "operand_extend_operator"
           [(match_operand 1 "register_operand" "0,0")
            (match_operator
             7 "extend_operator"
             [(mem (plus:SI (mult:SI (match_operand:SI 2 "register_operand" "r,r")
                                     (match_operand:SI 3 "const_int_operand" "n,n"))
                            (match_operand:SI 4 "register_operand" "r,r")))])]))
     (set (match_operand:SI 5 "register_operand" "=4,r")
          (plus:SI (mult:SI (match_dup 2)
                            (match_dup 3))
                   (match_dup 4)))])]
  "REGNO(operands[5]) == REGNO(operands[4])"
  [(set (match_dup 5) (plus:SI (mult:SI (match_dup 2) (match_dup 3))
                               (match_dup 4)))
   (set (match_dup 0) (match_op_dup 6 [(match_dup 1) (match_dup 8)]))]
  "operands[8] = gen_rtx(GET_CODE(operands[7]),GET_MODE(operands[7]),
                         gen_rtx(MEM,GET_MODE(XEXP(operands[7],0)),
                                 operands[5]));")
(define_split
  [(parallel
    [(set (match_operand 0 "register_operand" "=r,r,r")
          (match_operator
           5 "operand_extend_operator"
           [(match_operand 1 "register_operand" "0,0,0")
            (match_operator
             6 "extend_operator"
             [(mem (plus:SI (match_operand:SI 2 "bdap_operand" "%r,r,r")
                            (match_operand:SI 3 "bdap_operand" "r>Ri,r,>Ri")
                            ))])]))
     (set (match_operand:SI 4 "register_operand" "=2,r,r")
          (plus:SI (match_dup 2)
                   (match_dup 3)))])]
  "(rtx_equal_p(operands[4],operands[2])
   || rtx_equal_p(operands[4],operands[3]))"
  [(set (match_dup 4) (plus:SI (match_dup 2) (match_dup 3)))
   (set (match_dup 0) (match_op_dup 5 [(match_dup 1) (match_dup 7)]))]
  "operands[7] = gen_rtx(GET_CODE(operands[6]),GET_MODE(operands[6]),
                         gen_rtx(MEM,GET_MODE(XEXP(operands[6],0)),
                                 operands[4]));")

(define_split
  [(parallel
    [(set (match_operand 0 "register_operand" "=r,r")
          (match_operator
           7 "plus_or_bound_operator"
           [(match_operator
             6 "extend_operator"
             [(mem (plus:SI (mult:SI (match_operand:SI 2 "register_operand" "r,r")
                                     (match_operand:SI 3 "const_int_operand" "n,n"))
                            (match_operand:SI 4 "register_operand" "r,r")))])
            (match_operand 1 "register_operand" "0,0")]))
     (set (match_operand:SI 5 "register_operand" "=4,r")
          (plus:SI (mult:SI (match_dup 2)
                            (match_dup 3))
                   (match_dup 4)))])]
  "REGNO(operands[5]) == REGNO(operands[4])"
  [(set (match_dup 5) (plus:SI (mult:SI (match_dup 2) (match_dup 3))
                               (match_dup 4)))
   (set (match_dup 0) (match_op_dup 6 [(match_dup 8) (match_dup 1)]))]
  "operands[8] = gen_rtx(GET_CODE(operands[6]),GET_MODE(operands[6]),
                         gen_rtx(MEM,GET_MODE(XEXP(operands[6],0)),
                                     operands[5]));")

(define_split
  [(parallel
    [(set (match_operand 0 "register_operand" "=r,r,r")
          (match_operator
           6 "plus_or_bound_operator"
           [(match_operator
             5 "extend_operator"
            [(mem (plus:SI (match_operand:SI 2 "bdap_operand" "%r,r,r")
                           (match_operand:SI 3 "bdap_operand" "r>Ri,r,>Ri")))])
            (match_operand 1 "register_operand" "0,0,0")]))
     (set (match_operand:SI 4 "register_operand" "=2,r,r")
          (plus:SI (match_dup 2)
                   (match_dup 3)))])]
  "(rtx_equal_p(operands[4],operands[2])
   || rtx_equal_p(operands[4],operands[3]))"
  [(set (match_dup 4) (plus:SI (match_dup 2) (match_dup 3)))
   (set (match_dup 0) (match_op_dup 6 [(match_dup 7) (match_dup 1)]))]
  "operands[7] = gen_rtx(GET_CODE(operands[5]),GET_MODE(operands[5]),
                         gen_rtx(MEM,GET_MODE(XEXP(operands[5],0)),
                                     operands[4]));")
;
;(define_split
;  [(parallel
;    [(set (match_operand 0 "register_operand" "=r")
;         (match_operand 1 "memory_operand" "m"))
;     (set (match_operand:SI 2 "register_operand" "=r")
;         (and:SI (subreg:SI (match_dup 0) 0)
;                 (match_operand 3 "const_int_operand" "n")))])]
;  "REGNO(operands[0]) == REGNO(operands[2])"
;  [(set (match_dup 2) (zero_extend:SI (match_dup 1)))
;   (set (strict_low_part (match_dup 0))
;       (and:QI (match_dup 0) (match_dup 3)))]
;  "")
;
; When splitting from a define_peephole, strange things happen. We
; probably shouldnt do it, but this makes life more interesting:
;  The result from the define_peephole of the and/zero_extend should
; look like the one immediately above, but will
; match the following (and hopefully nothing else):
;
(define_split
  [(parallel[ 
       (match_operand 0 "register_operand" "=r")
       (match_operand 1 "nonimmediate_operand" "rm")
       (match_operand 2 "register_operand" "=r")
       (match_operand 3 "const_int_operand" "n")])]
  "REGNO(operands[0]) == REGNO(operands[2])"
  [(set (match_dup 2) (match_dup 5))
   (set (match_dup 6)
        (match_dup 4))]
  "{ enum machine_mode mode = INTVAL(operands[3]) > 255 ? HImode : QImode;
     rtx newreg = gen_rtx(REG,mode,REGNO(operands[0]));
     operands[6] = gen_rtx(STRICT_LOW_PART,mode,newreg);
     operands[4] = gen_rtx(AND,mode,newreg,operands[3]);
     operands[5] = gen_rtx(ZERO_EXTEND, GET_MODE(operands[2]),
                           gen_rtx(GET_CODE(operands[1]),mode,
                                   XEXP(operands[1],0)));}")

;
; Lots of opportunities to split up addressing prefixes, so we can
; fill a delay slot.  Never split if we lose something, though.
;
;   move [indirect_ref],rx  indirect ref = {const, [r+], [r]}
; costs as much as
;   move indirect_ref,rx \; move [rx],rx
; Just dont allow indirect_ref = r
(define_split
  [(set (match_operand 0 "register_operand" "=r")
        (match_operand 1 "indirect_operand" "m"))]
  "GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
   && (GET_CODE(XEXP(operands[1],0)) == MEM
       || CONSTANT_P(XEXP(operands[1],0)))"
  [(set (match_dup 2) (match_dup 4))
   (set (match_dup 0) (match_dup 3))]
  "operands[2] = gen_rtx(REG,Pmode,REGNO(operands[0]));
   operands[3] = gen_rtx(MEM,GET_MODE(operands[0]),operands[2]);
   operands[4] = XEXP(operands[1],0);")

; movs, movu as above
(define_split
  [(set (match_operand 0 "register_operand" "=r")
        (match_operator
         4 "extend_operator"
         [(match_operand 1 "indirect_operand" "m")]))]
  "GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
   && (GET_CODE(XEXP(operands[1],0)) == MEM
       || CONSTANT_P(XEXP(operands[1],0)))"
  [(set (match_dup 2) (match_dup 5))
   (set (match_dup 0) (match_op_dup 4 [(match_dup 3)]))]
  "operands[2] = gen_rtx(REG,Pmode,REGNO(operands[0]));
   operands[3] = gen_rtx(MEM,GET_MODE(XEXP(operands[4],0)),operands[2]);
   operands[5] = XEXP(operands[1],0);")

;;
;; SOMETIME IN THE FUTURE when we can trust the reg_dead notes, we may
;; uncomment this. I dont trust PRESERVE_DEATH_INFO_REGNO_P
;
;   move [ry+index],rx
; costs as much as
;   add index,ry \; move [ry],rx, if ry is dead, or ry=rx
; Index can be many things. bdap_biap matches them all. Just remember
; to check that one of the plus-operands is the same as rx.
;
;(define_split
;  [(set (match_operand 0 "register_operand" "=r")
;       (mem (plus:SI (match_operand:SI 1 "bdap_biap_operand" "rR>")
;                     (match_operand:SI 2 "bdap_biap_operand" "rR>i"))))]
;; Stuff the insn into operands[4], as we cannot get to it otherwise.
;  "(BASE_P(operands[1]) && dead_or_set_p(operands[4]=insn,operands[1])
;   || BASE_P(operands[2]) && dead_or_set_p(operands[4]=insn,operands[2]))
;   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD"
;  [(set (match_dup 3) (plus:SI (match_dup 6) (match_dup 4)))
;   (set (match_dup 0) (match_dup 5))]
;  "if (BASE_P(operands[1]) && dead_or_set_p(operands[4],operands[1])) {
;      operands[3] = operands[1];
;      operands[6] = operands[1];
;      operands[4] = operands[2];
;   } else {
;      operands[3] = operands[2];
;      if (GET_CODE(operands[1]) == MULT) {
;        operands[6] = operands[1];
;        operands[4] = operands[2];
;      } else {
;        operands[6] = operands[2];
;        operands[4] = operands[1];
;      }
;   }
;   operands[5] = gen_rtx(MEM,GET_MODE(operands[0]),operands[3]);")

;; See note above. 
; movs movu as above
;(define_split
;  [(set (match_operand 0 "register_operand" "=r")
;       (match_operator
;        7 "extend_operator"
;        [(mem (plus:SI (match_operand:SI 1 "bdap_biap_operand" "rR>")
;                       (match_operand:SI 2 "bdap_biap_operand" "rR>i")))]))]
;; Stuff the insn into operands[4], as we cannot get to it otherwise.
;  "(BASE_P(operands[1]) && dead_or_set_p(operands[4]=insn,operands[1])
;   || BASE_P(operands[2]) && dead_or_set_p(operands[4]=insn,operands[2]))
;   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD"
;  [(set (match_dup 3) (plus:SI (match_dup 6) (match_dup 4)))
;   (set (match_dup 0) (match_op_dup 7 [(match_dup 5)]))]
;  "if (BASE_P(operands[1]) && dead_or_set_p(operands[4],operands[1])) {
;      operands[3] = operands[1];
;      operands[6] = operands[1];
;      operands[4] = operands[2];
;   } else {
;      operands[3] = operands[2];
;      if (GET_CODE(operands[1]) == MULT) {
;        operands[6] = operands[1];
;        operands[4] = operands[2];
;      } else {
;        operands[6] = operands[2];
;        operands[4] = operands[1];
;      }
;   }
;   operands[5] = gen_rtx(MEM,GET_MODE(XEXP(operands[7],0)),operands[3]);")

;; See note above.
;
; move rx,[ry+index] when ry is dead, except when
; index is [-128..-64,64..127]
;
;(define_split
;  [(set (mem:SI (plus:SI (match_operand:SI 1 "bdap_biap_operand" "rR>")
;                     (match_operand:SI 2 "bdap_biap_operand" "rR>i")))
;       (match_operand:SI 0 "nonmemory_operand" "=rM"))]
;; Stuff the insn into operands[4], as we cannot get to it otherwise.
;  "((BASE_P(operands[1]) && dead_or_set_p(operands[4]=insn,operands[1])
;    && (GET_CODE(operands[2]) != CONST_INT
;        || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'J')
;        || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'N')
;        || INTVAL(operands[2]) > 127
;        || INTVAL(operands[2]) < -128))
;    || BASE_P(operands[2]) && dead_or_set_p(operands[4]=insn,operands[2]))
;   && (BASE_P(operands[0]) || operands[0] == const0_rtx)"
;  [(set (match_dup 3) (plus:SI (match_dup 6) (match_dup 4)))
;   (set (mem:SI (match_dup 3)) (match_dup 0))]
;  "if (BASE_P(operands[1]) && dead_or_set_p(operands[4],operands[1])) {
;      operands[3] = operands[1];
;      operands[6] = operands[1];
;      operands[4] = operands[2];
;   } else {
;      operands[3] = operands[2];
;      if (GET_CODE(operands[1]) == MULT) {
;        operands[6] = operands[1];
;        operands[4] = operands[2];
;      } else {
;        operands[6] = operands[2];
;        operands[4] = operands[1];
;      }
;   }")

;; See note above. 
;(define_split
;  [(set (mem:HI (plus:SI (match_operand:SI 1 "bdap_biap_operand" "rR>")
;                     (match_operand:SI 2 "bdap_biap_operand" "rR>i")))
;       (match_operand:HI 0 "nonmemory_operand" "=rM"))]
;; Stuff the insn into operands[4], as we cannot get to it otherwise.
;  "((BASE_P(operands[1]) && dead_or_set_p(operands[4]=insn,operands[1])
;    && (GET_CODE(operands[2]) != CONST_INT
;        || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'J')
;        || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'N')
;        || INTVAL(operands[2]) > 127
;        || INTVAL(operands[2]) < -128))
;    || BASE_P(operands[2]) && dead_or_set_p(operands[4]=insn,operands[2]))
;   && (BASE_P(operands[0]) || operands[0] == const0_rtx)"
;  [(set (match_dup 3) (plus:SI (match_dup 6) (match_dup 4)))
;   (set (mem:HI (match_dup 3)) (match_dup 0))]
;  "if (BASE_P(operands[1]) && dead_or_set_p(operands[4],operands[1])) {
;      operands[3] = operands[1];
;      operands[6] = operands[1];
;      operands[4] = operands[2];
;   } else {
;      operands[3] = operands[2];
;      if (GET_CODE(operands[1]) == MULT) {
;        operands[6] = operands[1];
;        operands[4] = operands[2];
;      } else {
;        operands[6] = operands[2];
;        operands[4] = operands[1];
;      }
;   }")

;; See note above
;(define_split
;  [(set (mem:QI (plus:SI (match_operand:SI 1 "bdap_biap_operand" "rR>")
;                     (match_operand:SI 2 "bdap_biap_operand" "rR>i")))
;       (match_operand:QI 0 "nonmemory_operand" "=rM"))]
;; Stuff the insn into operands[4], as we cannot get to it otherwise.
;  "((BASE_P(operands[1]) && dead_or_set_p(operands[4]=insn,operands[1])
;    && (GET_CODE(operands[2]) != CONST_INT
;        || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'J')
;        || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'N')
;        || INTVAL(operands[2]) > 127
;        || INTVAL(operands[2]) < -128))
;    || BASE_P(operands[2]) && dead_or_set_p(operands[4]=insn,operands[2]))
;   && (BASE_P(operands[0]) || operands[0] == const0_rtx)"
;  [(set (match_dup 3) (plus:SI (match_dup 6) (match_dup 4)))
;   (set (mem:QI (match_dup 3)) (match_dup 0))]
;  "if (BASE_P(operands[1]) && dead_or_set_p(operands[4],operands[1])) {
;      operands[3] = operands[1];
;      operands[6] = operands[1];
;      operands[4] = operands[2];
;   } else {
;      operands[3] = operands[2];
;      if (GET_CODE(operands[1]) == MULT) {
;        operands[6] = operands[1];
;        operands[4] = operands[2];
;      } else {
;        operands[6] = operands[2];
;        operands[4] = operands[1];
;      }
;   }")

;; See note above
;
; op.s [ry+i],rx, when ry != rx, and ry is dead and i not [-128..-64,64..127]   
;
;(define_split
;  [(set (match_operand 0 "register_operand" "=r")
;       (match_operator
;        3 "orthogonal_operator"
;        [(match_dup 0)
;         (mem (plus:SI (match_operand:SI 1 "bdap_biap_operand" "r>R")
;                       (match_operand:SI 2 "bdap_biap_operand" "r>Ri")))]))]
;; Stuff the insn into operands[4], as we cannot get to it otherwise.
;  "!reg_overlap_mentioned_p(operands[0],operands[1])
;   && !reg_overlap_mentioned_p(operands[0],operands[2])
;   && ((BASE_P(operands[1]) && dead_or_set_p(operands[4]=insn,operands[1])
;        && (GET_CODE(operands[2]) != CONST_INT
;            || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'J')
;            || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'N')
;            || INTVAL(operands[2]) > 127
;            || INTVAL(operands[2]) < -128))
;       || BASE_P(operands[2]) && dead_or_set_p(operands[4]=insn,operands[2]))
;   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
;   && REGNO(operands[0]) != 15"
;  [(set (match_dup 4) (plus:SI (match_dup 7) (match_dup 5)))
;   (set (match_dup 0) (match_op_dup 3 [(match_dup 0) (match_dup 6)]))]
;  "if (BASE_P(operands[1]) && dead_or_set_p(operands[4],operands[1])) {
;      operands[4] = operands[1];
;      operands[5] = operands[2];
;      operands[7] = operands[1];
;   } else {
;      operands[4] = operands[2];
;      if (GET_CODE(operands[1]) == MULT) {
;        operands[7] = operands[1];
;        operands[5] = operands[2];
;      } else {
;        operands[7] = operands[2];
;        operands[5] = operands[1];
;      }
;   }
;   operands[6] = gen_rtx(MEM,GET_MODE(operands[0]),operands[4]);")

;; See note above.
;
; comm_op.s [ry+i],rx, when ry != rx, and ry is dead and i not [-128..-64,64..127]      
;
;(define_split
;  [(set (match_operand 0 "register_operand" "=r")
;       (match_operator
;        3 "commutative_orth_op"
;        [(mem (plus:SI (match_operand:SI 1 "bdap_biap_operand" "r>R")
;                       (match_operand:SI 2 "bdap_biap_operand" "r>Ri")))
;         (match_dup 0)]))]
;  "!reg_overlap_mentioned_p(operands[0],operands[1])
;   && !reg_overlap_mentioned_p(operands[0],operands[2])
;   && ((BASE_P(operands[1]) && dead_or_set_p(operands[4]=insn,operands[1])
;        && (GET_CODE(operands[2]) != CONST_INT
;            || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'J')
;            || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'N')
;            || INTVAL(operands[2]) > 127
;            || INTVAL(operands[2]) < -128))
;       || BASE_P(operands[2]) && dead_or_set_p(operands[4]=insn,operands[2]))
;   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
;   && REGNO(operands[0]) != 15"
;  [(set (match_dup 4) (plus:SI (match_dup 7) (match_dup 5)))
;   (set (match_dup 0) (match_op_dup 3 [(match_dup 0) (match_dup 6)]))]
;  "if (BASE_P(operands[1]) && dead_or_set_p(operands[4],operands[1])) {
;      operands[4] = operands[1];
;      operands[5] = operands[2];
;      operands[7] = operands[1];
;   } else {
;      operands[4] = operands[2];
;      if (GET_CODE(operands[1]) == MULT) {
;        operands[7] = operands[1];
;        operands[5] = operands[2];
;      } else {
;        operands[7] = operands[2];
;        operands[5] = operands[1];
;      }
;   }
;   operands[6] = gen_rtx(MEM,GET_MODE(operands[0]),operands[4]);")

;; See note above
;
; op[ext].s [ry+i],rx, when ry != rx, and ry is dead and i not [-128..-64,64..127]      
;
;(define_split
;  [(set (match_operand 0 "register_operand" "=r")
;       (match_operator
;        3 "operand_extend_operator"
;        [(match_dup 0)
;         (match_operator
;          7 "extend_operator"
;          [(mem (plus:SI (match_operand:SI 1 "bdap_biap_operand" "r>R")
;                         (match_operand:SI 2 "bdap_biap_operand" "r>Ri")))])]))]
;  "!reg_overlap_mentioned_p(operands[0],operands[1])
;   && !reg_overlap_mentioned_p(operands[0],operands[2])
;   && ((BASE_P(operands[1]) && dead_or_set_p(operands[4]=insn,operands[1])
;        && (GET_CODE(operands[2]) != CONST_INT
;            || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'J')
;            || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'N')
;            || INTVAL(operands[2]) > 127
;            || INTVAL(operands[2]) < -128))
;       || BASE_P(operands[2]) && dead_or_set_p(operands[4]=insn,operands[2]))
;   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
;   && REGNO(operands[0]) != 15"
;  [(set (match_dup 4) (plus:SI (match_dup 8) (match_dup 5)))
;   (set (match_dup 0)
;       (match_op_dup
;        3 [(match_dup 0) (match_op_dup 7 [(match_dup 6)])]))]
;  "if (BASE_P(operands[1]) && dead_or_set_p(operands[4],operands[1])) {
;      operands[4] = operands[1];
;      operands[5] = operands[2];
;      operands[8] = operands[1];
;   } else {
;      operands[4] = operands[2];
;      if (GET_CODE(operands[1]) == MULT) {
;        operands[8] = operands[1];
;        operands[5] = operands[2];
;      } else {
;        operands[8] = operands[2];
;        operands[5] = operands[1];
;      }
;   }
;   operands[6] = gen_rtx(MEM,GET_MODE(XEXP(operands[7],0)),operands[4]);")

;; See note above
;
; comm_op[ext].s [ry+i],rx, when ry != rx, and ry is dead and not [-128..-64,64..127]   
;
;(define_split
;  [(set (match_operand 0 "register_operand" "=r")
;       (match_operator
;        3 "plus_or_bound_operator"
;        [(match_operator
;          7 "extend_operator"
;          [(mem (plus:SI (match_operand:SI 1 "bdap_biap_operand" "r>R")
;                         (match_operand:SI 2 "bdap_biap_operand" "r>Ri")))])
;         (match_dup 0)]))]
;  "!reg_overlap_mentioned_p(operands[0],operands[1])
;   && !reg_overlap_mentioned_p(operands[0],operands[2])
;   && ((BASE_P(operands[1]) && dead_or_set_p(operands[4]=insn,operands[1])
;        && (GET_CODE(operands[2]) != CONST_INT
;            || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'J')
;            || CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'N')
;            || INTVAL(operands[2]) > 127
;            || INTVAL(operands[2]) < -128))
;       || BASE_P(operands[2]) && dead_or_set_p(operands[4]=insn,operands[2]))
;   && GET_MODE_SIZE(GET_MODE(operands[0])) <= UNITS_PER_WORD
;   && REGNO(operands[0]) != 15"
;  [(set (match_dup 4) (plus:SI (match_dup 8) (match_dup 5)))
;   (set (match_dup 0)
;       (match_op_dup
;        3 [(match_dup 0) (match_op_dup 7 [(match_dup 6)])]))]
;  "if (BASE_P(operands[1]) && dead_or_set_p(operands[4],operands[1])) {
;      operands[4] = operands[1];
;      operands[5] = operands[2];
;      operands[8] = operands[1];
;   } else {
;      operands[4] = operands[2];
;      if (GET_CODE(operands[1]) == MULT) {
;        operands[8] = operands[1];
;        operands[5] = operands[2];
;      } else {
;        operands[8] = operands[2];
;        operands[5] = operands[1];
;      }
;   }
;   operands[6] = gen_rtx(MEM,GET_MODE(XEXP(operands[7],0)),operands[4]);")

;
; Various peephole optimization.
; Only put sequences of ordinary insns that can be combined into one insn.
;
; WATCH OUT when you exchange one instruction for another, the
; condition codes setting must be the same, or you have to CC_INIT (or
; whatever you think is needed) in the pattern, before you emit the
; assembly text.  This is best done here and not in
; notice_update_cc(), to reduce the number of places to modify.
;
; Dont add patterns that you dont know will be matched.
;
; I cannot tell Gcc (2.1) how to reload an instruction that looks like
; and.b mem_byte,const,reg_32
; where reg is the destination of the "three-address" code optimally.
;  It should be:
; movu.b mem_byte,reg_32
; and.b const,reg_32
;  but is turns into:
; move.b mem_byte,reg_32
; and.d const,reg_32
;  Fix it.
;
(define_peephole
  [(set (match_operand 0 "register_operand" "=r")
        (match_operand 1 "nonimmediate_operand" "rm"))
   (set (match_operand:SI 2 "register_operand" "=r")
        (and:SI (subreg:SI (match_dup 0) 0)
                (match_operand 3 "const_int_operand" "n")))]
; Just bail out when volatile mems are involved.  It's not worth the fuss.
  "REGNO(operands[0]) == REGNO(operands[2])
   && !CONST_OK_FOR_LETTER_P(INTVAL(operands[3]),'I')
   && !(GET_CODE(operands[1]) == MEM && MEM_VOLATILE_P(operands[1]))"
  "#") ; Match a split.

;
; We have trouble with and:s that could be transformed into
; zero_extend:s and reloading them.  Maybe something is broken in gcc?
; Or it could just be that bitfield insn expanding is a bit
; unoptimized when not having extzv insns.
;
(define_peephole
  [(set (match_operand 0 "register_operand" "=r")
        (ashiftrt:SI (match_dup 0)
                     (match_operand:SI 1 "const_int_operand" "n")))
   (set (match_dup 0)
        (and:SI (match_dup 0)
                (match_operand 2 "const_int_operand" "n")))]
  "INTVAL(operands[2]) > 31 && INTVAL(operands[2]) < 255
   && INTVAL(operands[1]) > 23"
; The n flag may be set, because this will be a "byte" and.
  "*cc_status.flags |= CC_NOT_NEGATIVE; return \"lsrq %1,%0\;and.b %2,%0\";")

(define_peephole
  [(set (match_operand 0 "register_operand" "=r")
        (ashiftrt:SI (match_dup 0)
                     (match_operand:SI 1 "const_int_operand" "n")))
   (set (match_dup 0)
        (and:SI (match_dup 0)
                (match_operand 2 "const_int_operand" "n")))]
  "INTVAL(operands[2]) > 31 && INTVAL(operands[2]) < 65535
   && INTVAL(operands[2]) != 255
   && INTVAL(operands[1]) > 15"
; The n flag may be set, because this will be a "word" and.
  "*cc_status.flags |= CC_NOT_NEGATIVE; return \"lsrq %1,%0\;and.w %2,%0\";")

(define_peephole
  [(set (match_operand 0 "register_operand" "=r")
        (lshiftrt:SI (match_dup 0)
                     (match_operand:SI 1 "const_int_operand" "n")))
   (set (match_dup 0)
        (and:SI (match_dup 0)
                (match_operand 2 "const_int_operand" "n")))]
  "INTVAL(operands[2]) > 31 && INTVAL(operands[2]) < 255
   && INTVAL(operands[1]) > 23"
; The n flag may be set, because this will be a "word" and.
  "*cc_status.flags |= CC_NOT_NEGATIVE; return \"lsrq %1,%0\;and.b %2,%0\";")

(define_peephole
  [(set (match_operand 0 "register_operand" "=r")
        (lshiftrt:SI (match_dup 0)
                     (match_operand:SI 1 "const_int_operand" "n")))
   (set (match_dup 0)
        (and:SI (match_dup 0)
                (match_operand 2 "const_int_operand" "n")))]
  "INTVAL(operands[2]) > 31 && INTVAL(operands[2]) < 65535
   && INTVAL(operands[2]) != 255
   && INTVAL(operands[1]) > 15"
; The n flag may be set, because this will be a "word" and.
  "*cc_status.flags |= CC_NOT_NEGATIVE; return \"lsrq %1,%0\;and.w %2,%0\";")

(define_peephole
  [(set (match_operand:SI 0 "register_operand" "=r")
        (match_operand:SI 1 "nonimmediate_operand" "rm"))
   (set (match_operand 2 "register_operand" "=0")
        (and:SI (match_dup 0)
                (match_operand 3 "const_int_operand" "n")))]
; Just bail out when volatile mems are involved.  It's not worth the fuss.
  "REGNO(operands[2]) == REGNO(operands[0])
   && INTVAL(operands[3]) <= 65535 && INTVAL(operands[3]) >= 0
   && !(GET_CODE(operands[1]) == MEM && MEM_VOLATILE_P(operands[1]))"
  "#")

(define_peephole
  [(set (match_operand:HI 0 "register_operand" "=r")
        (match_operand:HI 1 "nonimmediate_operand" "rm"))
   (set (match_operand 2 "register_operand" "=0")
        (and:HI (match_dup 0)
                (match_operand 3 "const_int_operand" "n")))]
; Just bail out when volatile mems are involved.  It's not worth the fuss.
  "INTVAL(operands[3]) <= 255 && INTVAL(operands[3]) >= 32
   && REGNO(operands[2]) == REGNO(operands[0])
   && !(GET_CODE(operands[1]) == MEM && MEM_VOLATILE_P(operands[1]))"
  "#")


;(define_peephole
;  [(set (match_operand 0 "register_operand" "=r,r,r")
;       (and (match_operand 1 "nonimmediate_operand" "r,>Q,m" )
;            (match_operand 2 "const_int_operand" "n,n,n")))
;   (set (match_operand 3 "register_operand" "=r,r,r")
;       (match_dup 0))]
;   "dead_or_set_p(insn,operands[0])
;    && (INTVAL(operands[2]) == 255 || INTVAL(operands[2]) == 65535)"
;   "moVu.%z2 %1,%3"
;   [(set_attr "slottable" "yes,yes,no")])
;
;(define_peephole
;  [(set (match_operand 0 "register_operand" "=r,r,r")
;       (zero_extend (match_operand 1 "nonimmediate_operand" "r,>Q,m")))
;   (set (match_operand 3 "register_operand" "=r,r,r")
;       (match_dup 0))]
;   "dead_or_set_p(insn,operands[0])"
;   "movU.%z2 %1,%3"
;   [(set_attr "slottable" "yes,yes,no")])
   
;
; Make add.d n,rx move [rx],ry into move [rx=rx+n],ry
; This will reduce the size of the assembler code for n = [-128..127],
; and speed up accordingly.
(define_peephole
  [(set (match_operand:SI 0 "register_operand" "=r")
        (plus:SI (match_operand:SI 1 "register_operand" "0")
                 (match_operand:SI 2 "const_int_operand" "n")))
   (set (match_operand 3 "register_operand" "=r")
        (mem (match_dup 0)))]
  "GET_MODE(operands[3]) != DImode
    && REGNO(operands[3]) != REGNO(operands[0])
    && (BASE_P(operands[1]) || BASE_P(operands[2]))
    && !CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'J')
    && !CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'N')
    && (INTVAL(operands[2]) >= -128 && INTVAL(operands[2]) < 128)"
  "move.%s3 [%0=%1%S2],%3") 
;
; Vice versa.
;
(define_peephole
  [(set (match_operand:SI 0 "register_operand" "=r")
        (plus:SI (match_operand:SI 1 "register_operand" "0")
                 (match_operand:SI 2 "const_int_operand" "n")))
   (set (mem (match_dup 0))
        (match_operand 3 "register_operand" "=r"))]

  "GET_MODE(operands[3]) != DImode
    && REGNO(operands[3]) != REGNO(operands[0])
    && (BASE_P(operands[1]) || BASE_P(operands[2]))
    && !CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'J')
    && !CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'N')
    && (INTVAL(operands[2]) >= -128 && INTVAL(operands[2]) < 128)"
; cc is really from previous insn, not the "add"
  "*CC_STATUS_INIT; return \"move.%s3 %3,[%0=%1%S2]\";") 
;
; make add.d n,rx op.d [rx],ry into op.d [rx=rx+n],ry
;  Saves when n = [-128..127], harmless otherwise
; We don't care about other valid combinations for sideeffect modes, they
; probably wont save the time they take typing in, at this stage.
;
(define_peephole
  [(set (match_operand:SI 0 "register_operand" "=r")
        (plus:SI (match_operand:SI 1 "register_operand" "0")
                 (match_operand:SI 2 "const_int_operand" "n")))
   (set (match_operand 3 "register_operand" "=r")
        (match_operator 4 "orthogonal_operator"
                           [(match_dup 3)
                            (mem (match_dup 0))]))]

  "GET_MODE(operands[3]) != DImode
    && REGNO(operands[0]) != REGNO(operands[3])
    && !CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'J')
    && !CONST_OK_FOR_LETTER_P(INTVAL(operands[2]),'N')
    && (INTVAL(operands[2]) >= -128 && INTVAL(operands[2]) < 128)"
  "%x4.%s0 [%3=%1%S2],%0")

;
; Sometimes, for some reason the pattern
;  move x,rx
;  add y,rx
;  move [rx],rz
; will occur.  Solve this, and likewise for operand.
;
(define_peephole
  [(set (match_operand:SI 0 "register_operand" "=r,r,r,r")
        (match_operand:SI 1 "bdap_biap_operand" "r,>Ri,r,>Ri"))
   (set (match_dup 0)
        (plus:SI (match_operand:SI 2 "bdap_biap_operand" "0,0,r>Ri,r")
                 (match_operand:SI 3 "bdap_biap_operand" "r>Ri,r,0,0")))
   (set (match_operand 4 "register_operand" "=r,r,r,r")
        (mem (match_dup 0)))]
  "(rtx_equal_p(operands[2],operands[0]) || rtx_equal_p(operands[3],operands[0]))
    && side_effect_mode_ok(PLUS,operands,0,
                           (REG_S_P(operands[1]) ? 1
                            : rtx_equal_p(operands[2],operands[0]) ? 3 : 2),
                           (!REG_S_P(operands[1]) ? 1 :
                            rtx_equal_p(operands[2],operands[0]) ? 3 : 2),-1,4)"

  "@
   move.%s4 [%0=%1%S3],%4
   move.%s4 [%0=%3%S1],%4
   move.%s4 [%0=%1%S2],%4
   move.%s4 [%0=%2%S1],%4") 
;
; Above but to memory.
;
(define_peephole
  [(set (match_operand:SI 0 "register_operand" "=r,r,r,r")
        (match_operand:SI 1 "bdap_biap_operand" "r,>Ri,r,>Ri"))
   (set (match_dup 0)
        (plus:SI (match_operand:SI 2 "bdap_biap_operand" "0,0,r>Ri,r")
                 (match_operand:SI 3 "bdap_biap_operand" "r>Ri,r,0,0")))
   (set (mem (match_dup 0))
        (match_operand 4 "register_operand" "=r,r,r,r"))]
  "(rtx_equal_p(operands[2],operands[0]) || rtx_equal_p(operands[3],operands[0]))
    && side_effect_mode_ok(PLUS,operands,0,
                           (REG_S_P(operands[1]) ? 1
                            : rtx_equal_p(operands[2],operands[0]) ? 3 : 2),
                           (!REG_S_P(operands[1]) ? 1 :
                            rtx_equal_p(operands[2],operands[0]) ? 3 : 2),-1,4)"
  "*CC_STATUS_INIT; { static char *insns[] = { \"move.%s4 %4,[%0=%1%S3]\",
   \"move.%s4 %4,[%0=%3%S1]\", \"move.%s4 %4,[%0=%1%S2]\", \"move.%s4 %4,[%0=%2%S1]\"};
   return insns[which_alternative];}") 

;
; Op as above.
;
(define_peephole
  [(set (match_operand:SI 0 "register_operand" "=r,r,r,r")
        (match_operand:SI 1 "bdap_biap_operand" "r,>Ri,r,>Ri"))
   (set (match_dup 0)
        (plus:SI (match_operand:SI 2 "bdap_biap_operand" "0,0,r>Ri,r")
                 (match_operand:SI 3 "bdap_biap_operand" "r>Ri,r,0,0")))
   (set (match_operand 4 "register_operand" "=r,r,r,r")
        (match_operator 5 "orthogonal_operator"
                        [(match_dup 3)
                         (mem (match_dup 0))]))]
  "(rtx_equal_p(operands[2],operands[0]) || rtx_equal_p(operands[3],operands[0]))
    && side_effect_mode_ok(PLUS,operands,0,
                           (REG_S_P(operands[1]) ? 1
                            : rtx_equal_p(operands[2],operands[0]) ? 3 : 2),
                           (!REG_S_P(operands[1]) ? 1 :
                            rtx_equal_p(operands[2],operands[0]) ? 3 : 2),-1,4)"
  "@
   %x5.%s4 [%0=%1%S3],%4
   %x5.%s4 [%0=%3%S1],%4
   %x5.%s4 [%0=%1%S2],%4
   %x5.%s4 [%0=%2%S1],%4")
;
(define_peephole
  [(set (match_operand:SI 0 "register_operand" "=r,r,r,r")
        (match_operand:SI 1 "bdap_biap_operand" "r,>Ri,r,>Ri"))
   (set (match_dup 0)
        (plus:SI (match_operand:SI 2 "bdap_biap_operand" "0,0,r>Ri,r")
                 (match_operand:SI 3 "bdap_biap_operand" "r>Ri,r,0,0")))
   (set (match_operand 4 "register_operand" "=r,r,r,r")
        (match_operator 5 "commutative_orth_op"
                        [(mem (match_dup 0))
                         (match_dup 3)]))]
  "(rtx_equal_p(operands[2],operands[0]) || rtx_equal_p(operands[3],operands[0]))
    && side_effect_mode_ok(PLUS,operands,0,
                           (REG_S_P(operands[1]) ? 1
                            : rtx_equal_p(operands[2],operands[0]) ? 3 : 2),
                           (!REG_S_P(operands[1]) ? 1 :
                            rtx_equal_p(operands[2],operands[0]) ? 3 : 2),-1,4)"
  "@
   %x5.%s4 [%0=%1%S3],%4
   %x5.%s4 [%0=%3%S1],%4
   %x5.%s4 [%0=%1%S2],%4
   %x5.%s4 [%0=%2%S1],%4")

;
; Another braindamage: move rx,ry \; move [ry],ry
;
(define_peephole
  [(set (match_operand:SI 0 "register_operand" "=r")
        (match_operand:SI 1 "register_operand" "r"))
   (set (match_operand 2 "register_operand" "=r")
        (mem (match_dup 0)))]
  "REGNO(operands[0]) == REGNO(operands[2])
   && GET_MODE_SIZE(GET_MODE(operands[2])) <= UNITS_PER_WORD"
  "move.%s2 [%1],%0"
  [(set_attr "slottable" "yes")])

(define_peephole
  [(set (match_operand:SI 0 "register_operand" "=r")
        (match_operand:SI 1 "register_operand" "r"))
   (set (match_operand 2 "register_operand" "=r")
        (match_operator 3 "extend_operator" [(mem (match_dup 0))]))]
  "REGNO(operands[0]) == REGNO(operands[2])
   && GET_MODE_SIZE(GET_MODE(operands[2])) <= UNITS_PER_WORD"
  "mov%e3.%m3 [%1],%0"
  [(set_attr "slottable" "yes")])

;;- Local variables:
;;- mode:emacs-lisp
;;- comment-start: ";;- "
;;- eval: (set-syntax-table (copy-sequence (syntax-table)))
;;- eval: (modify-syntax-entry ?[ "(]")
;;- eval: (modify-syntax-entry ?] ")[")
;;- eval: (modify-syntax-entry ?{ "(}")
;;- eval: (modify-syntax-entry ?} "){")
;;- End:
