/*!**************************************************************************
*!
*!  FILE NAME   :  XA_SUM
*!
*!  DESCRIPTION :  This program takes as input two binary files used for
*!                 prom programming of the XA project, one containing low
*!                 bytes and the other containing high bytes. A checksum is
*!                 calculated and put at the end of the two files.
*!
*!  Tested using TC and CC.
*!
*!------------------------------------------------------------------------
*!  HISTORY
*!
*!  DATE    NAME            CHANGES
*!  ----    ----            -------
*!  900830  Thomas H        Initial version.
*!  910201  Ricard W        Modified for compilation by CC. Changed command
*!	                        line.
*!  910212  Ricard W        16-bit (word) checksum calculation.
*!  960603  H-P Nilsson     Could not find xasum1, so make this accept
*!                          when only one file is input.
*!                           Make it compilable without compiler.h
*!                          Entered it as-is in the src/tools
*!                          directory, since I could not find it
*!                          *anywhere near* the repository.
*!
*!**************************************************************************/
/* @(#) xasum.c 1.1 07/03/96 */

/**************************  INCLUDE FILES  ********************************/
#include <stdio.h>
#include <string.h>
#include <ctype.h>

/* This is ok on all current platforms (960703) at Axis, so why make
   it more difficult? */
typedef unsigned char byte;
typedef unsigned long udword;
typedef long dword;
typedef unsigned short uword;
typedef short word;
#define TRUE 1
#define FALSE 0

#ifdef __STDC__

#define PROTOTYPES
#define NO_RET void
#define NO_PARAMS void
#define CONST const

#else

#define NO_RET
#define NO_PARAMS
#define CONST

#endif

#ifdef __TURBOC__
#include <stdlib.h>
#endif

/************************** DEFINITIONS ********************************/

/************************ FUNCTION PROTOTYPES ***************************/

#ifdef PROTOTYPES

NO_RET error(byte*);
word   get_eprom_sum(byte*, byte*);
NO_RET get_arguments(uword, byte**, byte*, byte*, uword*);
NO_RET help(NO_PARAMS);
NO_RET put_check_sum(word, uword, byte*, byte*);
NO_RET warning(byte*);

#else

NO_RET error();
word   get_eprom_sum();
NO_RET get_arguments();
NO_RET help();
NO_RET put_check_sum();
NO_RET warning();

#endif

/***************************    VARIABLES    ********************************/

CONST byte helptext[][80] =
{
  "\n*** Usage : \n",
  "XASUM [options] lofile [hifile] prom_size\n",
  "lofile      : Name of the file containing low bytes of the proms.\n",
  "hifile      : Name of the file containing high bytes of the proms.\n",
  "prom_size   : Size of each bus half, rounded up to fill the proms used.\n",
  "              Must be given in kilobytes.\n",
  "All options should be preceded with '-'. Uppercase and lowercase\n",
  "are treated equally. Allowed options are:\n",
  "-?, -H      : Don't execute, print help\n",
  "Description :\n",
  "  XASUM calculates a checksum for the proms of the XA project, and put\n",
  "  this sum last in the given files. The sum is calculated as 0 minus the\n",
  "  total sum of all words of both files. The result, a signed word, is split\n",
  "  in two bytes, one for the file of low bytes and the other for the file\n",
  "  of high bytes.\n",
  "  Before the checksum is written into the files, these are expanded to\n",
  "  contain <prom_size> kilobytes each.\n",
  "Example :\n",
  "  XASUM prom.lo [prom.hi] 256\n",
  "    Calculates the checksum from the two files prom.lo and prom.hi. Each\n",
  "    file is expanded to 256 kB, and the low byte of the\n",
  "    checksum is placed last in prom.lo and the high byte in prom.hi.\n",
  "\n",
};

/*#************************************************************************
*#
*#  FUNCTION NAME : error
*#
*#  PARAMETERS    : Text message to print.
*#
*#  RETURNS       : Nothing.
*#
*#  SIDE EFFECTS  : Exits the program.
*#
*#  DESCRIPTION   : Prints error message and help text. Then exits program.
*#
*#************************************************************************/
NO_RET error(message)
byte  *message;
{
  printf("*** Error : %s\n",message);
  printf("*** Enter XASUM -H for description\7\n");
  exit(1);
}

/*#************************************************************************
*#
*#  FUNCTION NAME : warning
*#
*#  PARAMETERS    : Text message to print.
*#
*#  RETURNS       : Nothing.
*#
*#  DESCRIPTION   : Prints warning message.
*#
*#************************************************************************/
NO_RET  warning(message)
byte  *message;
{
  printf("*** Warning : %s\n",message);
}

/*#************************************************************************
*#
*#  FUNCTION NAME : help
*#
*#  PARAMETERS    : Nothing.
*#
*#  RETURNS       : Nothing.
*#
*#  DESCRIPTION   : Prints help text.
*#
*#************************************************************************/
NO_RET help(NO_PARAMS)
{
  word count;
  word no_of_lines;

  no_of_lines = sizeof(helptext) / sizeof(helptext[0]);
  for (count = 0; count < no_of_lines; count++)
  {
    printf("%s",helptext[count]);
  }
}

/*#**********************************************************************
*#
*#  FUNCTION NAME : get_arguments
*#
*#  PARAMETERS    : argc       : Number of arguments.
*#                  argv       : List of arguments.
*#                  loname     : Name of binary file of low bytes.
*#                  hiname     : Name of binary file of high bytes.
*#                  eprom_size : Size of eprom to build, in kB.
*#
*#  RETURNS       : Nothing.
*#
*#  DESCRIPTION   : Checks that the program has been called in a proper
*#                  way. If any error occurs, an error message is shown
*#                  and the program terminates.
*#
*#************************************************************************/
NO_RET get_arguments(argc, argv, loname, hiname, eprom_size)
uword  argc;
byte  *argv[];
byte  *loname;
byte  *hiname;
uword *eprom_size;
{
  uword ix;
  uword argcnt = 0;
  byte  error_text[40];

  for (ix = 1; ix < argc; ix++)
  {
    switch (argv[ix][0]) /* had to remove toupper(), which only works if */
 			 /* its argument is a lower case character (cc). */
    {
      case '?': help();
                exit(0);
                break;

      case '-': if (strlen(argv[ix]) < 2)
                {
                  sprintf(error_text,"Illegal option '%s'",argv[ix]);
                  error(error_text);
                }
                switch (argv[ix][1])
                {
                  case '?':
 		  case 'h':
                  case 'H': help();
                            exit(0);
                            break;
                  default : sprintf(error_text,"Illegal option '%s'",argv[ix]);
                            error(error_text);
                            break;
                }
                break;
      default : switch (argcnt)
                {
                  case 0  : strcpy(loname, argv[ix]);
                            argcnt++;
                            break;
                  case 1  : strcpy(hiname, argv[ix]);
                            argcnt++;
                            break;
                  case 2  : *eprom_size = atoi(argv[ix]);
                            argcnt++;
                            break;
                  default : error("Too many arguments.");
                            break;
                }
                break;
    }
  }

  if (argcnt == 2)
  {
    *eprom_size = atoi(hiname);
    strcpy(hiname,loname);
  }

  if (argcnt < 2)
    error("Missing argument(s).");
}

/*#**********************************************************************
*#
*#  FUNCTION NAME : get_eprom_sum
*#
*#  PARAMETERS    : loname     : Name of binary file of low bytes.
*#                  hiname     : Name of binary file of high bytes.
*#
*#  RETURNS       : Sum of all bytes in the files.
*#
*#  DESCRIPTION   : This function calculates the sum of all bytes in the
*#                  files used for prom progarmming of the XA project.
*#
*#************************************************************************/
word get_eprom_sum(loname, hiname)
byte  *loname;
byte  *hiname;
{
  FILE *lofile;
  FILE *hifile;
  word  sum = 0;
  word  ch;
  udword hi_count = 0;
  udword lo_count = 0;

  lofile = fopen(loname, "rb");
  if (strcmp(loname,hiname) != 0)
  {
    hifile = fopen(hiname, "rb");
  }
  else
  {
    hifile = lofile;
  }

  if ((hifile != NULL) && (lofile != NULL))
  {
    ch = fgetc(lofile);
    while (ch != EOF)
    {
      lo_count++;
      sum = sum + ch;
      ch = fgetc(lofile);
    }
    printf("Number of bytes in lofile: %ld\n", lo_count);

    ch = fgetc(hifile);
    while (ch != EOF)
    {
      hi_count++;
      sum = sum + (ch << 8);
      ch = fgetc(hifile);
    }
    printf("Number of bytes in hifile: %ld\n", hi_count);

    fclose(lofile);

    if (hifile != lofile)
    {
      fclose(hifile);
    }

    if (lo_count != hi_count)
      error("Alignment error; hi- and lofiles were not of the same length.");
  }
  else
    error("Nonexisting file(s)");

  return(sum);
}

/*#**********************************************************************
*#
*#  FUNCTION NAME : put_check_sum
*#
*#  PARAMETERS    : eprom_sum  : Sum of all bytes in the files.
*#                  eprom_size : Size of one eprom, in kB.
*#                  loname     : Name of binary file of low bytes.
*#                  hiname     : Name of binary file of high bytes.
*#
*#  RETURNS       : Nothing.
*#
*#  DESCRIPTION   : This function puts a checksum last in the two binary
*#                  files.
*#
*#************************************************************************/
NO_RET put_check_sum(eprom_sum, eprom_size, loname, hiname)
word   eprom_sum;
uword  eprom_size;
byte  *loname;
byte  *hiname;
{
  FILE  *lofile;
  FILE  *hifile;
  word   sum;
  word   check_sum;
  udword count;
  udword current;
  udword end_address;

  sum = eprom_sum;

  lofile = fopen(loname, "rb");
  if (lofile != NULL)
  {
    fseek(lofile, 0L, SEEK_END);
    current = ftell(lofile);
    fclose(lofile);
  }

  lofile = fopen(loname, "ab");
  hifile = fopen(hiname, "ab");
  if ((hifile != NULL) && (lofile != NULL))
  {
    end_address = (long) eprom_size * 1024;  /* no more 2* */
    for (count = current; count < (end_address - 1); count++)
    {
      fputc(0xFF, lofile);
      fputc(0xFF, hifile);
      sum--;  /* add FFFF = -1 */
    }
    check_sum = 0 - sum;

    printf("Check sum: 0x%04x\n", (uword) check_sum);

    fputc(((check_sum & 0xFF00) >> 8), hifile);
    fputc((byte) (check_sum & 0x00FF), lofile);
    fclose(lofile);
    fclose(hifile);
  }
  else
    error("Nonexisting file(s)");
}

/*#**********************************************************************
*#
*#  FUNCTION NAME : main
*#
*#  PARAMETERS    : argc: Number of arguments
*#                  argv: The arguments
*#
*#  RETURNS       : Possible error code. 0 if everything is OK.
*#
*#  DESCRIPTION   : See help text.
*#
*#************************************************************************/
uword main(argc, argv)
uword argc;
byte  *argv[];
{
  byte hiname[1024];
  byte loname[1024];
  word eprom_sum = 0;
  uword eprom_size = 0;

  get_arguments(argc, argv, loname, hiname, &eprom_size);
  eprom_sum = get_eprom_sum(loname, hiname);
  put_check_sum(eprom_sum, eprom_size, loname, hiname);
  return  0;
}
