/*
 * cris-burn.c
 *
 * 930804   RW   Built from HP's cstrip.c. 
 *               Generate PROM files from CRIS a.out file.
 *               Partially compatible with GNX nburn, regarding
 *               options -w -b -o and starting address.
 * 981807   HP   Found it, updated to accept "both" magic numbers.  Will be
 *               distributed but is deprecated for the 
 */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>

/************ MACRO SECTION ***********/

#define DBG(statement)  (statement)

#define BUFFER_SIZE 32768

/************* VARIABLES **************/

char inbuf[BUFFER_SIZE], outbuf[BUFFER_SIZE];
int width, bank; /* bus width and prom bank no */

/************* FUNCTIONS ***************/

unsigned long rd(FILE *f) 
{
  int c1,c2,c3,c4;

  c1 = getc(f);
  c2 = getc(f);
  c3 = getc(f);
  c4 = getc(f);

  return c1+c2*256+c3*65536+c4*65536*256;
}

int usage(void)
{
  fprintf(stderr,"Usage: cburn [options] -o <outfile> <infile>\n");
  exit(-1);
}

int split(int size, char *inbuf, char *outbuf)
{
  /* printf("split: %d ", size); */
  while (size > 0)
  {
    *outbuf = *(inbuf+bank);
    outbuf++;
    inbuf += width;
    size -= width;
  }
}

int main(int argc, char **argv)
{
  FILE *infile, *outfile;
  unsigned long tl, size;
  int arg, i;
  char *infilename, *outfilename;

  /* set up defaults */

  width = 2;
  bank = 0;
  outfilename = NULL;
  infilename = NULL;

  /* scan options and arguments */

  arg = 1;

  while (argv[arg] != NULL)
  {
    if (argv[arg][0] == '-') /* option */
    {
      switch (argv[arg][1])
      {
        case 'w': 
          width = atoi(&argv[arg][2]);
          DBG(printf("Bus width %d\n", width));
          break;
        case 'b':
          bank = atoi(&argv[arg][2]);
          DBG(printf("Bank %d\n", bank));
          break;
        case 'o':
          outfilename = argv[++arg];
          DBG(printf("Will write output to %s\n", outfilename));
          break;
        case 'h':
        case 'H':
        case '?':
          printf("cris-burn [options] infile: Generate prom files from CRIS a.out file\n");
          printf("options : -b<bank no>        set bank number\n");
          printf("          -w<bus width>      set bus width (1 or 2)\n");
          printf("          -o<out file>       output file (mandatory)\n");
          printf(" -i -V -x -n options are ignored, but do not cause termination\n");
          printf("(compatibility with GNX nburn)\n");
          printf("Note output is in binary, not intel hex or anything like that\n");
          exit(-1);
          break;
#ifdef IGNORE_CERTAIN_NBURN_OPTIONS
        case 'V':
          DBG(printf("-V option ignored\n"));
          break;
        case 'x':
          DBG(printf("-x option (address) ignored\n"));
          break;
        case 'l':
          DBG(printf("-l option (prom size) ignored\n"));
          break;
        case 'i':
          DBG(printf("-i option (intel hex output format) ignored\n"));
          break;
        case 'n':
          DBG(printf("-n option ignored\n"));
          break;
#endif
        default:
          printf("Unknown option : %s\n", argv[arg]);
          usage();
          break;
      }
    }
    else
    {
      infilename = argv[arg];
      DBG(printf("Will read from %s\n", infilename));
    }
    arg++;
  }

  /* Check argument and option validity */

  if (width != 1 && width != 2)
  {
    fprintf(stderr, "Illegal bus width %d! (terminating)\n", width);
    exit(-1);
  }
  if (bank > width - 1)
  {
    fprintf(stderr, "Illegal bank no %d (must be less than bus width)! (terminating)\n", bank);
    exit(-1);
  }

  if (infilename == NULL || outfilename == NULL)
    usage();

  /* start reading infile */
        
  if ((infile = fopen(infilename,"rb")) == NULL)
  {
    fprintf(stderr,"Cannot read ");
    perror(infilename);
    exit(-1);
  }

  if ((outfile = fopen(outfilename, "wb")) == NULL)
  {
    fprintf(stderr, "Cannot write ");
    perror(outfilename);
    exit(-1);
  }

  tl = rd(infile);
  if (tl != 0x0000010b && tl != 0x01ff010b) {
    fprintf(stderr,"Bad magic in %s: %x!= 0x0000010b\n",infilename,tl);
    exit(-1);
  }

  /* read file sizes */

  size = rd(infile);		/* text */
  tl = size;
  size = rd(infile);		/* data */
  tl += size;
  rd(infile);			/* bss */
  rd(infile);			/* symbol */
  rd(infile);			/* start */
  rd(infile);			/* text-reloc */
  rd(infile);			/* data-reloc */

  /* read data and generate output. Start with complete buffers. */

  for (i = tl; i >= BUFFER_SIZE; i -= BUFFER_SIZE)
  {
    fread(inbuf, BUFFER_SIZE, 1, infile);
    split(BUFFER_SIZE, inbuf, outbuf);
    fwrite(outbuf, BUFFER_SIZE / width, 1, outfile);
  }

  /* fill inbuf with zeroes, in case input file is not evenly divisble
   * by bus width. Then the additional bytes will be filled with zeroes
   * in the prom files.
   */
  /* 961220 HP  Just _had_ to fix that blatant misordered-memset-operand bug */
  memset(inbuf, 0, BUFFER_SIZE);

  /* handle the remaining part. */
  if (i > 0)
  {
    if (i % width != 0)
    {
      fprintf(stderr, "Error in infile: segment sizes should be a multiple of width\n");
    }
    fread(inbuf, i, 1, infile);
    split(i, inbuf, outbuf);
    i += i % width; /* try to fix non-integral segment size (filled out with 0) */
    fwrite(outbuf, i / width, 1, outfile);
  }
  if (errno)
  {
    perror("Failed");
    fclose(outfile);
    unlink(outfilename);
    exit(-1);
  }

  fclose(infile);
  fclose(outfile);

  return 0;
}

/************** END OF FILE cris-burn.c *****************/
