/* Print cris instructions for objdump
   Copyright (C) 1986, 1987, 1989, 1991 Free Software Foundation, Inc.


This file is part of the binutils.

The binutils are free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

The binutils are distributed in the hope that they will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with the binutils; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

/*
 * Modified from m68k-pinsn.c ver
 * $Id: m68k-pinsn.c,v 1.7 1991/12/08 01:14:04 sac Exp $
   $Log: m68k-pinsn.c,v $
 * Revision 1.7  1991/12/08  01:14:04  sac
 *
 * 1993-04-13 HP Created
 * 1993-09-16 HP Changed constraints; special register match gets 3,
 *               not 0.  See constraint()
 * 1994-10-13 HP Got tired of not being able to trace switch-sequences.
 *               cris-pinsn will now (following def of TRACE_CASE) try
 *               to trace a switch statement, printing out correct
 *               interpretation of offsets after "sub x,r%d/adds.w [pc+r%d.w],pc"
 *               statements. 
 * 1997-11-17 HP First note about Svinto flag disassembly changes.
 */
#include "dis-asm.h"
#include "bfd.h"
#include "sysdep.h"
#include <stdio.h>
#include "opcode/cris-opc.h"

static char prefix_string[80] = "";
static char was_prefix = 0;

#ifndef TRACE_CASE
#define TRACE_CASE 1
#endif /* TRACE_CASE */

/* The following variables belong to TRACE_CASE too, but would make the
   code look (even more) like a total mess if you put ifdefs around
   their use. */ 
long last_immediate = 0; /* Candidate for next case_offset. */

#if TRACE_CASE
/* Trace the following sequence:
   sub* %d1,r%d
   bound* %d2,r%d
   adds.w [pc+r%d.w],pc

   Which is the assembly form of a switch-statement in C.
    The first subtraction is optional, if there is none, then d1 will
   be zero.
    d1 is the value of the first case,
    d2 is the number of cases (including default),

   This results in case offsets printed on the form:
   (case N: ORG)
   where N is an estimation on the corresponding 'case' operand in C,
   and ORG is where execution of that case continues after the
   sequence presented above.
    The old style of output was to print the
   offsets as instructions, which made it hard to follow
   "case"-constructs in the disassembly, and printed a lot of annoying
   undefined instructions.
   */
static long case_offset = 0;    /* Value of first element in switch */
static long case_offset_candidate = 0;    /* Value of first element in switch */
static long case_offset_counter = 0; /* How many more case-offsets to print. */
static long no_of_case_offsets = 0; /* Number of case offsets. */
#endif /* TRACE_CASE */

static int
number_of_bits(bits)
     unsigned int bits;
{
  int i;

  for (i = 0; bits != 0; bits &= bits-1)
    i++;

  return i;
}

#ifndef PRINT_IN_PREF_REPR
#define PRINT_IN_PREF_REPR(INSN,NBYTES,SIGNEDP,NUMBER_REPR,ADDR,BUF,STREAM) \
  print_in_preferred_representation(INSN,NBYTES,SIGNEDP,NUMBER_REPR,ADDR, \
				    BUF,STREAM)

static char *
print_in_preferred_representation(insn, nbytes, signedp,
                                  number_repr, addr, buffer,
                                  cp)
     unsigned int insn;
     int nbytes;
     int signedp;
     int number_repr;
     bfd_vma addr;
     unsigned char *buffer;
     char *cp;
{
  enum Representation { Hex, Dec, Reg, HexDig};

  enum Representation repr = (enum Representation) number_repr;

  int i = 0;
  char zero_suppress = 1;

  switch(repr) {
  case Hex:
    *cp++ = '0';
    *cp++ = 'x';
    /* Fall down into... */
  case HexDig:
    {
      int factor = 0;
      last_immediate = 0;

      for (i = nbytes-1; i > 0; i--)
        if (buffer[i] || !zero_suppress) {
          zero_suppress = 0;
          sprintf(cp,"%02x",buffer[i]);
          cp+= 2;
          factor += 8;
        }
    }

#if TRACE_CASE
    for (i = 0; i < nbytes; i++)
      last_immediate += buffer[i]*(1 << (8*i));
#endif /* TRACE_CASE */

    sprintf(cp,zero_suppress ? "%x" : "%02x",buffer[0]);
    cp += buffer[0] >= 16 || !zero_suppress ? 2 : 1;
    break;

  case Dec:
    { char buf[20];

      if (nbytes == 1)
	{
          last_immediate = buffer[0];
          sprintf(buf, signedp ? "%d" : "%u" ,
                  (int) (signedp ?
                         (last_immediate = (last_immediate > 127
                                            ? (256 - last_immediate) : last_immediate))
                         : last_immediate));
        }          
      else if (nbytes == 2)
        {
          last_immediate = buffer[0]+buffer[1]*256;

          sprintf(buf, signedp ? "%d" : "%u" ,
                  (int) (signedp ?
                         (last_immediate = (last_immediate > 32767
                                            ? (65536 - last_immediate) : last_immediate))
                         :  last_immediate));
        }
      else if (nbytes == 4)
        {
          last_immediate = buffer[0]+buffer[1]*256
		+ buffer[2]*65536 + buffer[3]*0x1000000;
          sprintf(buf, signedp ? "%ld" : "%lu" , last_immediate);
        }
      else {
	sprintf(cp,"??Dec?");
	cp += 6;
	cp = print_in_preferred_representation(insn,nbytes, 0, Hex, addr,
					       buffer, cp);
	*buf = 0;
      }

      strcpy(cp,buf);
      cp += strlen(buf);
    }
    break;
    
  case Reg:
    { char buf[20];
      sprintf(buf,
	      (buffer[0]&15) == 15 ? "pc"
	      : (buffer[0]&15) == 14 ? "sp" : "r%d",
		buffer[0]&15);
      sprintf(cp,buf);
      cp += strlen(buf);
    }
    break;
  }

  return cp;
}
#endif

#ifndef PRINT_HEX_DATA
#define PRINT_HEX_DATA 1
#endif

#ifndef PARSE_PREFIX
#define PARSE_PREFIX 1
#endif

int
cris_constraint(insn, buffer, addr, cs)
     unsigned int insn;
     unsigned char *buffer;
     bfd_vma addr;
     char *cs;
{
  int retval = 0;
  int tmp,i;

  char *s = cs;
  for (; *s; s++)
    switch (*s) {
    case 'm':
      if ((insn & 0x30) == 0x30)
	return -1;
      break;

    case 'S':
      /* A prefix operand. */
      if (was_prefix)
	break;
      else
	return -1;

    case 'B':
      /* A "push" prefix. Check for legal "push" size.
         In case of special register, it may be != 4. */
      if (was_prefix
	  && (strcmp(prefix_string,"sp-4") == 0
	      || strcmp(prefix_string,"sp-2") == 0
	      || strcmp(prefix_string,"sp-1") == 0))
	break;
      else
	return -1;

    case 'D':
      retval = (((insn >> 12)&15) == (insn & 15));
      if (!retval)
	return -1;
      else retval += 4;
      break;

    case 'P':
      tmp = (insn >> 12)&15;
      for (i = 0; i < NUMSPECREGS; i++)
	/* Since we match four bits, we will give a value of
	   4-1 = 3 in a match.  If there is a corresponding
	   exact match of a special register in another pattern, it
	   will get a value of 4, which will be higher.  This should
	   be correct in that an exact pattern would match better that
	   a general pattern.
	    Note that there is a reason for not returning zero; the
	   pattern for "clear" is partly  matched in the bit-pattern
	   (the two lower bits must be zero), while the bit-pattern
	   for a move from a special register is matched in the
	   register constraint. 
	    This also means we will will have a race condition if
	   there is a partly match in three bits in the bit pattern */  
	if (tmp == spec_regs[i].number) {
	  retval += 3;
	  break;
	}
      if (i == NUMSPECREGS)
	return -1;
      break;
    }

  return retval;
}

static unsigned
bytes_to_skip(insn, matched, template)
     unsigned int insn;
     int matched;
     char *template;
{
  unsigned to_skip = 2;
  char *s = template;

  for (; *s; s++)
    if (*s == 's' && insn & 0x800 && (insn & 15) == 15) {
      int tmp = 1 << ((insn >> 4) & (*template == 'z' ? 1 : 3));
      
      to_skip += cris_opcodes[matched].imm_oprnd_size == SIZE_FIX_32
	? 4 : cris_opcodes[matched].imm_oprnd_size == SIZE_SPEC_REG
	  ? (insn & 0x8000 ? 4 : 2) : tmp == 1
	    ? 2 : tmp;
    }  
    else if (*s == 'b')
      to_skip += 2;

  return to_skip;    
}

static char *
print_flags(insn, cp)
     unsigned int insn;
     char *cp;
{
  unsigned char byt = (((insn >> 8) & 0xf0) | (insn & 15));

  /* Use the Svinto definitions for disassembly.  The differences with Etrax
     vs. Svinto are (in interpretation of bit placement):
     Etrax 'd' <=> Svinto 'm', Etrax 'e' <=> Svinto 'b'. */
  static char fnames[] = "cvznxibm";
  int i;

  for (i = 0; i < 8; i++)
    if (byt & (1 << i))
      *cp++ = fnames[i];

  return cp;
}

/* WARNING Uses lots of magic and cris-opcode.h knowledge. */
static int
print_with_operands(matched, insn, buffer, addr, info)
     int matched;
     unsigned int insn;
     unsigned char *buffer;
     bfd_vma addr;
     disassemble_info *info;
{
  static char temp[80];
  char *tp = temp;
  static const char mode_char[] = "bwd?";
  static const char *cc_strings[] = {
    "hs",
    "lo",
    "ne",
    "eq",
    "vc",
    "vs",
    "pl",
    "mi",
    "ls",
    "hi",
    "ge",
    "lt",
    "gt",
    "le",
    "a",
    "ext",
  };

  int skip;

#define PRINT_REG(x) \
  do { char buf[1]; buf[0] = (x); \
	 tp = print_in_preferred_representation(insn, 1, 0, 2, \
						addr,buf,tp);} while (0)

#define IS_PREFIX (PARSE_PREFIX && *cs == 'p')
					   
  char *s,*cs = cris_opcodes[matched].args;
  int name_length = strlen(cris_opcodes[matched].name);

  skip = bytes_to_skip(insn,matched, cs);

  s = cs;
  if (PRINT_HEX_DATA && !IS_PREFIX) {
    int i;
    for (i = -was_prefix; i < skip; i++)
      (*info->fprintf_func)(info->stream,"%02x",buffer[i]);
    for (; (i+was_prefix) < 8; i++)
      (*info->fprintf_func)(info->stream,"  ");
    (*info->fprintf_func)(info->stream," ");
  }

#if TRACE_CASE  
  /* If to print data as offsets, then shortcut here. */
  if (case_offset_counter != 0)
    {
      (*info->fprintf_func)(info->stream,"case %d%s: -> 0x%lx",
              case_offset+no_of_case_offsets-case_offset_counter,
              case_offset_counter == 1 ? "/default" : "",
              (unsigned long) ((short) (buffer[0]+buffer[1]*256)
                               + (long) (addr - (no_of_case_offsets-case_offset_counter)*2 )));
      case_offset_counter--;

      /* The default case start (without the "sub" must be zero. */
      if (case_offset_counter == 0)
        case_offset = case_offset_candidate = 0;
      return 2;
    }
#endif /* TRACE_CASE */

  if (*s == 'p')
    s++;

  *temp = 0;

  if (*s == 'm' || *s == 'M' || *s == 'z') {
    *tp++ = '.';
    *tp++ = *s == 'M'
      ? (insn & 0x8000 ? 'd'
	 : insn & 0x4000 ? 'w' : 'b')
	: mode_char[(insn >> 4) & (*s == 'z' ? 1 : 3)];
    name_length += 2;
    s += 2;
  }
  
  if (!IS_PREFIX && *s != 'b')
    *tp++ = ' ';

  for (; *s; s++) {
    switch (*s) {
    case ' ':
    case ',':
      *tp++ = *s;
      break;

    case 'B':
      /* Recognize the prefix. */
      was_prefix = - was_prefix;
      break;

    case 'D':
    case 'r':
      PRINT_REG(insn & 15);
      break;
      
    case 'R':
      PRINT_REG((insn >> 12) & 15);
      break;

    case 'y':
    case 'S':
    case 's':
      /* [pc+] */
      if (insn & 0x400
	  && (insn & 15) == 15) {
	int tmp = 1 << ((insn >> 4) & (*cs == 'z' ? 1 : 3));

	tmp = /* Number of bytes */
	  cris_opcodes[matched].imm_oprnd_size == SIZE_FIX_32
	    ? 4 : cris_opcodes[matched].imm_oprnd_size == SIZE_SPEC_REG
	      ? (insn & 0x8000 ? 4 : 2) : tmp == 1 ? 2 : tmp;
	tp = print_in_preferred_representation
	  (insn,tmp,
	   /* Number as signed? */
	   (*cs == 'z' && (insn & 0x20)) || !strcmp(cris_opcodes[matched].name,"bdap"),
	   /* Representation, 0 = Hex 1 = Dec */
	   (*cs == 'z' && (insn & 0x20))
	   || (strcmp(cris_opcodes[matched].name,"bdap") == 0
	       && (tmp <= 2 || buffer[1+tmp] == 0)) ? 1 : 0,
	   addr, buffer+2,tp);
      }
      else {
	*tp++ = '[';
	if (was_prefix) {
	  if (insn & 0x400) {
	    PRINT_REG(insn & 15);
	    *tp++ = '=';
	  }
	  strcpy(tp,prefix_string);
	  tp += strlen(prefix_string);
	  was_prefix = - was_prefix;
	} else {
	  PRINT_REG(insn & 15);
	  if (insn & 0x400)
	    *tp++ = '+';
	}
	*tp++ = ']';
      }
      break;

    case 'x':
      PRINT_REG(insn >> 12);
      *tp++ = '.';
      *tp++ = mode_char[(insn >> 4) & 3];
      break;

    case 'I':
      {
	unsigned char buf[1];
	buf[0] = insn & 63;
	tp = print_in_preferred_representation(insn,1, 0, 1, addr,buf,tp);
      }
      break;

    case 'b':
      {
        int where = ((short) (buffer[2]+buffer[3]*256));
	unsigned char buf[4];

	where += addr+4;
	buf[0] = where;
	buf[1] = where >> 8;
	buf[2] = where >> 16;
	buf[3] = where >> 24;

	strcpy(tp,cc_strings[insn >> 12]);
	tp += strlen(cc_strings[insn >> 12]);
	*tp++ = ' ';
	tp = print_in_preferred_representation(insn,4, 0, 0, addr,buf,tp);
      }
      break;

    case 'c':
      {
	unsigned char buf[1];
	buf[0] = insn & 31;
	tp = print_in_preferred_representation(insn,1, 0, 1, addr,buf,tp);
      }
      break;

    case 'C':
      {
	unsigned char buf[1];
	buf[0] = insn & 15;
	tp = print_in_preferred_representation(insn,1, 0, 1, addr,buf,tp);
      }
      break;

    case 'o':
      {
	unsigned char offs[4];
	long offset = insn & 0xfe;
	if (insn & 1)
	  offset = offset|~0xff;
	offset += addr+2;
	offs[0] = offset;
	offs[1] = offset >> 8;
	offs[2] = offset >> 16;
	offs[3] = offset >> 24;
	tp = print_in_preferred_representation(insn,4, 0, 0, addr,offs,tp);
      }
      break;

    case 'O':
      tp = print_in_preferred_representation(insn,1, 1, 1, addr,buffer,tp);
      *tp++ = ',';
      PRINT_REG(insn >> 12);
      break;

    case 'f':
      tp = print_flags(insn,tp);
      break;

    case 'i':
      {
	char buf[1];
	buf[0] = (insn&32) ? (insn&31)|~31 : insn&31;
	tp = print_in_preferred_representation(insn,1, 1, 1, addr,buf,tp);
      }
      break;

    case 'P':
      {
	int i;

	for (i = 0; i < NUMSPECREGS; i++)
	  if (((insn >> 12)&15) == spec_regs[i].number) {
	    strcpy(tp,spec_regs[i].name);
	    tp += strlen(spec_regs[i].name);
	    break;
	  }
	if (i == NUMSPECREGS)
	  *tp++ = '?';
      }
      break;

    default:
      strcpy(tp,"???");
      tp += 3;
    }
  }

  *tp++ = 0;

  if (IS_PREFIX) {
      was_prefix = skip;
      skip = 0;

      /* Modify temp[] into parsed prefix mode in prefix_string. */
      if (strcmp(cris_opcodes[matched].name,"dip") == 0) {

	/* DIP prefix. Just copy as is. */
	strcpy(prefix_string,temp);

      } else if (strcmp(cs,"pO") == 0) {

	/* BDAP 8bit,r */
	/* Change "num,reg" into "reg+num" or, if num < 0, "reg-num" */
	char *tem = strchr(temp,',');
	strcpy(prefix_string,tem+1);
	if (*temp != '-')
	  strcat(prefix_string,"+");

	/* "reg+" or "reg". add into "reg+num" or "reg-num" */
	*tem = 0;
	strcat(prefix_string,temp);
      }
      else if (strcmp(cris_opcodes[matched].name,"biap") == 0) {

	/* Change ".m r,R" into "r+R.m" */
	strcpy(prefix_string,temp+2);
	*strchr(prefix_string,',') = '+';
	temp[2] = 0;
	strcat(prefix_string,temp);
      } else {

	/* (bdap).m s,r; change into "r+s.m" or "r+s"/"r-s", if value.
	   If s is a value, then it is not terminated by a ']'. */
	char *tem = strchr(temp,',');

	strcpy(prefix_string, tem+1);
	if (temp[2] != '-')
	  strcat(prefix_string,"+");
	*tem = 0;
	strcat(prefix_string,temp+2);

	/* Got "r+s"/"r-s". Check if to add .m */
	if (tem[-1] == ']') {
	  temp[2] = 0;
	  strcat(prefix_string,temp);
	}
      }
    }
  else {

    if (was_prefix > 0) {
      (*info->fprintf_func)(info->stream,"OOPS prefix [%s]!! ", prefix_string);
      was_prefix = - was_prefix;
    }

    (*info->fprintf_func)(info->stream,cris_opcodes[matched].name);

    if (*cs == 'p')
      cs++;

    (*info->fprintf_func)(info->stream,"%s%s",
	    *cs == 'b' || *cs == 'z' || *cs == 'm' || *cs == 'M'
	    ? "" : " ", temp);
    skip -= was_prefix;
    was_prefix = 0;

#if TRACE_CASE
    if (strncmp(cris_opcodes[matched].name,"sub",3) == 0)
      { case_offset_candidate = last_immediate; }

    /* Dont forget that it could also be an "add", if we start on
       negative case-values.  */
    else if (strncmp(cris_opcodes[matched].name,"add",3) == 0)
      {
        /* First check: Is it the casejump? "adds.w [pc+r%d.w],pc" */
        if (cris_opcodes[matched].name[3] == 's'
            && strncmp(temp,".w [pc+r",8) == 0)
          {
            char *s = temp+8;
            while (*s <= '9' && *s >= '0')
              s++;
            if (strncmp(s,".w],pc",6) == 0)
              {
                case_offset_counter = no_of_case_offsets;
                case_offset = case_offset_candidate;
              }
            else
              case_offset_candidate = 0;
          }
        else
          /* It is pretty safe to assume that no constants are fetched
             in a way that matches the above pattern! */
          case_offset_candidate = -last_immediate;
      }
    else
      {
        /* Check if "bound" told us the number of cases.  */
        if (strncmp(cris_opcodes[matched].name,"bound",5) == 0)
          { no_of_case_offsets = last_immediate+1; }
        else
          /* Then the sequence of switch-instructions was broken --
             assume default first-case again. */
          case_offset_candidate = 0;
      }
#endif /* TRACE_CASE */
  }

  return skip > 10 ? 10 : skip;
}


/* Print the cris instruction at address MEMADDR in debugged memory,
   on STREAM.  Returns length of the instruction, in bytes.  */

int
print_insn_cris_old(addr, buffer, info)
     bfd_vma addr;
     unsigned    char *buffer;
     struct disassemble_info *info;
{
  int i;

  unsigned char lobyte,hibyte;
  unsigned int insn;

  int max_level_of_match;
  int max_matched;

  int advance = 0;

  do {
    max_level_of_match = 0;
    max_matched = -1;

    lobyte = buffer[0];
    hibyte = buffer[1];

    insn = (hibyte << 8) + lobyte;

#if TRACE_CASE
    if (case_offset_counter != 0)
      max_matched = 0;          /* Anything !=  -1 will do. */
    else
#endif /* TRACE_CASE */
    for (i = 0; i < NUMOPCODES; i++) {
      
      int level_of_match;
      
      if ((cris_opcodes[i].match & insn) == cris_opcodes[i].match
	  && (cris_opcodes[i].lose & insn) == 0
	  && (level_of_match
	      = cris_constraint(insn, buffer, addr, cris_opcodes[i].args)) >= 0
	  && (level_of_match
	      += number_of_bits(cris_opcodes[i].match | cris_opcodes[i].lose))
	  > max_level_of_match) {
	max_matched = i;
	max_level_of_match = level_of_match;
	if (level_of_match == 16)
	  break;
      }
    }
    
    if (max_matched == -1) {
      char buf[20];
      (*info->fprintf_func)(info->stream,"??");
      print_in_preferred_representation(insn,2,0,0,addr,buffer,buf);
      (*info->fprintf_func)(info->stream,buf);
      advance += 2;
    }
    else
      advance = print_with_operands(max_matched,insn,buffer,addr,info);

    buffer += was_prefix;
    addr += was_prefix;
    
  } while (advance == 0);
  

  return advance;
}

int print_insn_cris(memaddr, info)
     bfd_vma memaddr;
     disassemble_info *info;
{
#define MAX_BYTES_PER_CRIS_INSN 10

  /* No instruction is allowed to be longer than 10 bytes.  I hope. */
  unsigned char buffer[MAX_BYTES_PER_CRIS_INSN];
  int status;
  int retval = -1;
  int i;

  /* There will be an "out of range" error after the last instruction.
     Reading pairs of decreasing number of bytes we hope that we will get at
     least the amount that print_insn_cris_old() will consume.
      If that was wrong, we let through the error message.  */
  for (i = MAX_BYTES_PER_CRIS_INSN; i > 0; i -= 2)
  {
    if ((status = (*info->read_memory_func) (memaddr, buffer, i, info))
        == 0)
    {
      break;
    }
  }

  /* If we did not get as much as we asked for, then clear the rest to make a
     reproducible result */
  if (i != MAX_BYTES_PER_CRIS_INSN)
  {
    memset(buffer+i, 0, MAX_BYTES_PER_CRIS_INSN-i);
  }

  if (i != 0)
  {
    retval = print_insn_cris_old(memaddr, buffer, info);
  }

  if (status != 0 && retval > i)
  {
    (*info->memory_error_func) (status, memaddr, info);
    return -1;
  }

  return retval;
}
/* End of cris-dis.c */
