/* cris-opcode.h -- Table of opcodes for the CGA-RISC processor.
   Copyright (C) 1989 Free Software Foundation, Inc.

This file is part of GAS, the GNU Assembler, and GDB, the GNU disassembler.

GAS/GDB is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GAS/GDB is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GAS or GDB; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#if !defined(__STDC__) && !defined(const)
#define const
#endif

/* REGISTERS */

/* Added 920330 by MA */
#define MAX_REG (15)
#define REG_SP  (14)
#define REG_PC  (15)

/* Added 920330 by MA */
struct spec_reg
{
  const char    *name;
  unsigned int  number;
};

static struct spec_reg spec_regs[] =
{
  { "vr",  1  },
  { "ccr", 5  },
  { "dcr0",6  },
  { "dcr1",7  },
  { "ibr", 9  },
  { "irp", 10 },
  { "srp", 11 },
  { "bar/dtp0", 12},            /* For disassembly use. */
  { "bar", 12 },
  { "dtp0",12 },
  { "dccr/dtp1",13 },           /* For disassembly use */
  { "dccr",13 },
  { "dtp1",13 },
  { "brp", 14 }
};

#define NUMSPECREGS ((sizeof spec_regs)/(sizeof spec_regs[0]))

/* Definitions for condition codes */
/* Added 920319 by MA */
/* Updated 920401 by MA */

#define CC_CC  0x0
#define CC_HS  0x0
#define CC_CS  0x1
#define CC_LO  0x1
#define CC_NE  0x2
#define CC_EQ  0x3
#define CC_VC  0x4
#define CC_VS  0x5
#define CC_PL  0x6
#define CC_MI  0x7
#define CC_LS  0x8
#define CC_HI  0x9
#define CC_GE  0xA
#define CC_LT  0xB
#define CC_GT  0xC
#define CC_LE  0xD
#define CC_A   0xE
#define CC_EXT 0xF


/* Added 920406 by MA */
/* 920713  MA  Changed to bit within byte instead of word */
#define AUTOINCR_BIT (0x04)


/* 920324 MA added PC_RELATIVE */
/* 920508 MA added pcrel_type */

/* Used as pcrel parameter in fix_new calls */
enum pcrel_type { NOT_PC_RELATIVE, PC_RELATIVE };

/* OPCODE-DEPENDENT CONSTANTS */

/* Updated 920508 by MA to Preliminary specification - CGARISC INTRUCTION SET Rev. 0.12 */


/* Prefixes */

/* Added 920402 by MA */
/* Changed 930408 by HP */
#define BDAP_QUICK_OPCODE  (0x0100)
#define BDAP_QUICK_Z_BITS  (0x0e00)

#define BIAP_OPCODE        (0x0540)
#define BIAP_Z_BITS        (0x0a80)

#define DIP_OPCODE         (0x0970)
#define DIP_Z_BITS         (0xf280)

/* Added 920326 by MA */
/* Changed 930408 by HP */
#define BDAP_INDIR_LOW     (0x40)
#define BDAP_INDIR_LOW_Z   (0x80)
#define BDAP_INDIR_HIGH    (0x09)
#define BDAP_INDIR_HIGH_Z  (0x02)

/* Not used, but this is how the above are used: /hp 930408 */
#define BDAP_INDIR_OPCODE  (BDAP_INDIR_HIGH * 0x0100 + BDAP_INDIR_LOW)
#define BDAP_INDIR_Z_BITS  (BDAP_INDIR_HIGH_Z * 0x100 + BDAP_INDIR_LOW_Z)
#define BDAP_PC_LOW        (BDAP_INDIR_LOW + REG_PC)
#define BDAP_INCR_HIGH     (BDAP_INDIR_HIGH + AUTOINCR_BIT)


/* Bcc quick */

/* Added 920713 by MA */
/* Changed 930408 by HP */
/* Don't forget to change Z_BITS if you change OPCODE !!! */
#define BRANCH_QUICK_Z_BITS  (0x0F00)
#define BRANCH_QUICK_LOW     (0)
#define BRANCH_QUICK_HIGH    (0)
#define BRANCH_QUICK_OPCODE  (BRANCH_QUICK_HIGH * 0x0100 + BRANCH_QUICK_LOW)


/* BA quick */

/* Added 920425 by MA */
/* 920713  MA  Defined in terms of the BRANCH_QUICK_HIGH constant */
#define BA_QUICK_HIGH (BRANCH_QUICK_HIGH + CC_A * 0x10)

/* Added 920702 by MA */
/* 920703  MA  Construct from low and high byte values */
#define BA_QUICK_OPCODE  (BA_QUICK_HIGH * 0x100 + BRANCH_QUICK_LOW)


/* Bcc [PC+] */

/* Added 920326 by MA */
/* Changed 930408 by HP */
#define BRANCH_PC_LOW            (0xFF)
#define BRANCH_INCR_HIGH         (0x0D)

/* Added 920425 by MA */
/* 920703  MA  Construct from low and high byte values */
#define BA_PC_INCR_OPCODE  ((BRANCH_INCR_HIGH + CC_A * 0x10) * 0x0100 + BRANCH_PC_LOW)


/* JUMP */

/* Added 920713 by MA */
/* Changed 930408 by HP */
/* Don't forget to change Z_BITS if you change OPCODE !!! */
/* Note that both special register 8 and 0 (both are zero-valued) are
   recognized as jump-instructions, but special-register 0 is generated.
   /H-P 970624*/
#define JUMP_INDIR_OPCODE  (0x0930)
#define JUMP_INDIR_Z_BITS  (0x72c0)

/* Added 920511 by MA */
/* 920713  MA  Expressed in terms of JUMP_INDIR_OPCODE */
#define JUMP_PC_INCR_OPCODE  (JUMP_INDIR_OPCODE + AUTOINCR_BIT * 0x0100 + REG_PC)


/* NOP */

/* Added 920713 by MA */
/* Changed 930408 by HP */
#define NOP_Z_BITS  (0xFAF0)

/* Added 920702 by MA */
/* Changed 930408 by HP */
/* Don't forget to change Z_BITS if you change OPCODE !!! */
#define NOP_OPCODE  (0x050F)

/* OPCODE TABLE */


/*
 * Structure of an opcode table entry.
 */

/* 920312 by MA : renamed sparc_opcode to cris_opcode, redeclared match and lose to unsigned int */
/* 920603 MA Added imm_oprnd_size field */
enum imm_oprnd_size_type { SIZE_NONE, SIZE_FIX_32, SIZE_SPEC_REG, SIZE_FIELD };

struct cris_opcode
{
    const char    *name;
    unsigned int  match;     /* Bits that must be set.  */
    unsigned int  lose;      /* Bits that must not be set.  */
    const char    *args;
    char          delayed;   /* Nonzero if this is a delayed branch */
                             /* instruction.                        */

    enum imm_oprnd_size_type imm_oprnd_size;  /* Size of immediate operands */
};


/*
   All cris opcodes are 16 bits.

   The match component is a mask saying which bits must match a
   particular opcode in order for an instruction to be an instance
   of that opcode.

   The args component is a string containing one character
   for each operand of the instruction.

Kinds of operands:
   B    Not realy an operand; outputs a "BDAP -size,SP" prefix for
           the PUSH instructions
   b    Non-matching operand, used for branch with 16-bit displacement.
        Only recognized in the disassembler.
   c    5-bit unsigned immediate in bits <4:0>
   C    4-bit unsigned immediate in bits <3:0>
   D    General register in bits <15:12> and <3:0>
   f    List of flags in bits <15:12> and <3:0>
   i    6-bit signed immediate in bits <5:0>
   I    6-bit unsigned immediate in bits <5:0>
   M    Size modifier (B, W or D) for CLEAR instructions
   m    Size modifier (B, W or D) in bits <5:4>
   o    [-128..127] word offset in bits <7:1> and <0>
   O    [-128..127] offset in bits <7:0> comma and general reg: n,r
   P    Special register in bits <15:12>
   p    Changed: tells that insn is a prefix insn. Must be 1st char HP 930416
   R    General register in bits <15:12>
   r    General register in bits <3:0>
   S    Source operand in bit <10> and a prefix; a 3-operand prefix
   s    Source operand in bits <10> and <3:0> and optionally a prefix;
           no prefix or an effective-address prefix
   x    Rs.m in bits <15:12> and <5:4>
   z    Size modifier (B or W) in bit <4>

*/


/* The order of the opcodes in this table is significant:
   
   * The assembler requires that all instances of the same mnemonic must be
     consecutive.  If they aren't, the assembler will bomb at runtime.

   * The disassembler should not care about the order of the opcodes.

   * The 'S' and 's' operand descriptors are syntactically overlapping -
     offset mode without an offset expression and indirect mode.

   * Register operands may be interpreted as immediate mode operands.  */


/* 920401  MA  First version completed                            */
/* 920508  MA  Updated to Preliminary specification - CGARISC     */
/*               INTRUCTION SET Rev. 0.12                         */
/* 920603  MA  Added imm_oprnd_size values                        */
/* 920604  MA  Added spaces after 'z' argument descriptors        */
/* 920626  MA  Added the PUSH and POP instructions                */
/* 920630  MA  Added NOP instruction, changed order of operands   */
/*               and added size modifier to the MUL and DIV       */
/*               instructions.                                    */
/* 920701  MA  Set the "delayed" field for delayed branch         */
/*               instructions.                                    */
/* 920713  MA  Use opcode constants in table for increased safety */
/* 920714  MA  Corrected test instruction operands                */
/* 920717  MA  Added the AX, CLEAR, DI and EI instructions        */
/* 930408  HP  Changed to comply with cris instruction set
                                               ver 0.31           */
/* 960802  HP  Added new Etrax 4 instructions: retb, lz, break,   */
/*             and related modifications; brp register            */
/* 961113  HP  Added svinto (Etrax V8) instructions               */
/* 970624  HP  Modified generated "jump"-opcode to using special  */
/*             register 0 as destination (was 8).  For reasons of */
/*             disassembly (etc), special register 8 is still     */
/*             recognized for jump.                               */
static struct cris_opcode cris_opcodes[] =
{
  { "abs",     0x06B0, 0x0940,             "r,R",     0, SIZE_NONE     },
  { "add",     0x0600, 0x09c0,             "m r,R",   0, SIZE_NONE     },
  { "add",     0x0A00, 0x01c0,             "m s,R",   0, SIZE_FIELD    },
  { "add",     0x0A00, 0x01c0,             "m S,D",   0, SIZE_NONE     },
  { "add",     0x0a00, 0x05c0,             "m S,R,r", 0, SIZE_NONE     },
  { "addi",    0x0500, 0x0Ac0,             "x,r",     0, SIZE_NONE     },
  { "addq",    0x0200, 0x0Dc0,             "I,R",     0, SIZE_NONE     },
  { "adds",    0x0420, 0x0Bc0,             "z r,R",   0, SIZE_NONE     },
  { "adds",    0x0820, 0x03c0,             "z s,R",   0, SIZE_FIELD    },
  { "adds",    0x0820, 0x03c0,             "z S,D",   0, SIZE_NONE     },
  { "adds",    0x0820, 0x07c0,             "z S,R,r", 0, SIZE_NONE     },
  { "addu",    0x0400, 0x0be0,             "z r,R",   0, SIZE_NONE     },
  { "addu",    0x0800, 0x03e0,             "z s,R",   0, SIZE_FIELD    },
  { "addu",    0x0800, 0x03e0,             "z S,D",   0, SIZE_NONE     },
  { "addu",    0x0800, 0x07e0,             "z S,R,r", 0, SIZE_NONE     },
  { "and",     0x0700, 0x08C0,             "m r,R",   0, SIZE_NONE     },
  { "and",     0x0B00, 0x00C0,             "m s,R",   0, SIZE_FIELD    },
  { "and",     0x0B00, 0x00C0,             "m S,D",   0, SIZE_NONE     },
  { "and",     0x0B00, 0x04C0,             "m S,R,r", 0, SIZE_NONE     },
  { "andq",    0x0300, 0x0CC0,             "i,R",     0, SIZE_NONE     },
  { "asr",     0x0780, 0x0840,             "m r,R",   0, SIZE_NONE     },
  { "asrq",    0x03a0, 0x0c40,             "c,R",     0, SIZE_NONE     },
  { "ax",      0x15B0, 0xEA4F,             "",        0, SIZE_NONE     },
  /* Should use branch #defines */
  { "b",       0x0dff, 0x0200,             "b",       0, SIZE_NONE     },
  { "ba",      BA_QUICK_OPCODE,
               0x0F00+(0xF-CC_A)*0x1000,   "o",       1, SIZE_NONE     },
  { "bcc",     BRANCH_QUICK_OPCODE+CC_CC*0x1000,
               0x0f00+(0xF-CC_CC)*0x1000,  "o",       1, SIZE_NONE     },
  { "bcs",     BRANCH_QUICK_OPCODE+CC_CS*0x1000,
               0x0f00+(0xF-CC_CS)*0x1000,  "o",       1, SIZE_NONE     },
  { "bdap",    BDAP_INDIR_OPCODE,
               BDAP_INDIR_Z_BITS,          "pm s,R",  0, SIZE_FIELD    },
  { "bdap",    BDAP_QUICK_OPCODE,
               BDAP_QUICK_Z_BITS,          "pO",      0, SIZE_NONE     },
  { "beq",     BRANCH_QUICK_OPCODE+CC_EQ*0x1000,
               0x0f00+(0xF-CC_EQ)*0x1000,  "o",       1, SIZE_NONE     },
  { "bext",    BRANCH_QUICK_OPCODE+CC_EXT*0x1000,
               0x0f00+(0xF-CC_EXT)*0x1000, "o",       1, SIZE_NONE     },
  { "bir",     BRANCH_QUICK_OPCODE+CC_EXT*0x1000,
               0x0f00+(0xF-CC_EXT)*0x1000, "o",       1, SIZE_NONE     },
  { "bge",     BRANCH_QUICK_OPCODE+CC_GE*0x1000,
               0x0f00+(0xF-CC_GE)*0x1000,  "o",       1, SIZE_NONE     },
  { "bgt",     BRANCH_QUICK_OPCODE+CC_GT*0x1000,
               0x0f00+(0xF-CC_GT)*0x1000,  "o",       1, SIZE_NONE     },
  { "bhi",     BRANCH_QUICK_OPCODE+CC_HI*0x1000,
               0x0f00+(0xF-CC_HI)*0x1000,  "o",       1, SIZE_NONE     },
  { "bhs",     BRANCH_QUICK_OPCODE+CC_HS*0x1000,
               0x0f00+(0xF-CC_HS)*0x1000,  "o",       1, SIZE_NONE     },
  { "biap",    BIAP_OPCODE, BIAP_Z_BITS,   "pm r,R",   0, SIZE_NONE     },
  { "ble",     BRANCH_QUICK_OPCODE+CC_LE*0x1000,
               0x0f00+(0xF-CC_LE)*0x1000,  "o",       1, SIZE_NONE     },
  { "blo",     BRANCH_QUICK_OPCODE+CC_LO*0x1000,
               0x0f00+(0xF-CC_LO)*0x1000,  "o",       1, SIZE_NONE     },
  { "bls",     BRANCH_QUICK_OPCODE+CC_LS*0x1000,
               0x0f00+(0xF-CC_LS)*0x1000,  "o",       1, SIZE_NONE     },
  { "blt",     BRANCH_QUICK_OPCODE+CC_LT*0x1000,
               0x0f00+(0xF-CC_LT)*0x1000,  "o",       1, SIZE_NONE     },
  { "bmi",     BRANCH_QUICK_OPCODE+CC_MI*0x1000,
               0x0f00+(0xF-CC_MI)*0x1000,  "o",       1, SIZE_NONE     },
  { "bmod",    0x0ab0, 0x0140,             "s,R",     0, SIZE_FIX_32   },
  { "bmod",    0x0ab0, 0x0140,             "S,D",     0, SIZE_NONE     },
  { "bmod",    0x0ab0, 0x0540,             "S,R,r",   0, SIZE_NONE     },
  { "bne",     BRANCH_QUICK_OPCODE+CC_NE*0x1000,
               0x0f00+(0xF-CC_NE)*0x1000,  "o",       1, SIZE_NONE     },
  { "bound",   0x05c0, 0x0A00,             "m r,R",   0, SIZE_NONE     },
  { "bound",   0x09c0, 0x0200,             "m s,R",   0, SIZE_FIELD    },
  { "bound",   0x09c0, 0x0200,             "m S,D",   0, SIZE_NONE     },
  { "bound",   0x09c0, 0x0600,             "m S,R,r", 0, SIZE_NONE     },
  { "bpl",     BRANCH_QUICK_OPCODE+CC_PL*0x1000,
               0x0f00+(0xF-CC_PL)*0x1000,  "o",       1, SIZE_NONE     },
  { "break",   0xe930, 0x16c0,             "C",       0, SIZE_NONE     },
  { "bstore",  0x0af0, 0x0100,             "s,R",     0, SIZE_FIX_32   },
  { "bstore",  0x0af0, 0x0100,             "S,D",     0, SIZE_NONE     },
  { "bstore",  0x0af0, 0x0500,             "S,R,r",   0, SIZE_NONE     },
  { "btst",    0x04F0, 0x0B00,             "r,R",     0, SIZE_NONE     },
  { "btstq",   0x0380, 0x0C60,             "c,R",     0, SIZE_NONE     },
  { "bvc",     BRANCH_QUICK_OPCODE+CC_VC*0x1000,
               0x0f00+(0xF-CC_VC)*0x1000,  "o",       1, SIZE_NONE     },
  { "bvs",     BRANCH_QUICK_OPCODE+CC_VS*0x1000,
               0x0f00+(0xF-CC_VS)*0x1000,  "o",       1, SIZE_NONE     },
  { "clear",   0x0670, 0x3980,             "M r",     0, SIZE_NONE     },
  { "clear",   0x0A70, 0x3180,             "M y",     0, SIZE_NONE     },
  { "clear",   0x0A70, 0x3180,             "M S",     0, SIZE_NONE     },
  { "clearf",  0x05F0, 0x0A00,             "f",       0, SIZE_NONE     },
  { "cmp",     0x06C0, 0x0900,             "m r,R",   0, SIZE_NONE     },
  { "cmp",     0x0Ac0, 0x0100,             "m s,R",   0, SIZE_FIELD    },
  { "cmp",     0x0Ac0, 0x0100,             "m S,D",   0, SIZE_NONE     },
  { "cmpq",    0x02C0, 0x0D00,             "i,R",     0, SIZE_NONE     },
  { "cmps",    0x08e0, 0x0300,             "z s,R",   0, SIZE_FIELD    },
  { "cmps",    0x08e0, 0x0300,             "z S,D",   0, SIZE_NONE     },
  { "cmpu",    0x08c0, 0x0320,             "z s,R" ,  0, SIZE_FIELD    },
  { "cmpu",    0x08c0, 0x0320,             "z S,D",   0, SIZE_NONE     },
  { "di",      0x25F0, 0xDA0F,             "",        0, SIZE_NONE     },
  { "dip",     DIP_OPCODE, DIP_Z_BITS,     "ps",      0, SIZE_FIX_32   },
  { "div",     0x0980, 0x0640,             "m R,r",   0, SIZE_FIELD    },
  { "dstep",   0x06f0, 0x0900,             "r,R",     0, SIZE_NONE     },
  { "ei",      0x25B0, 0xDA4F,             "",        0, SIZE_NONE     },
  { "jbrc",    0x69b0, 0x9640,             "r",       0, SIZE_NONE     },
  { "jbrc",    0x6930, 0x92c0,             "s",       0, SIZE_FIX_32   },
  { "jbrc",    0x6930, 0x92c0,             "S",       0, SIZE_NONE     },
  { "jir",     0xA9b0, 0x5640,             "r",       0, SIZE_NONE     },
  { "jir",     0xA930, 0x52c0,             "s",       0, SIZE_FIX_32   },
  { "jir",     0xA930, 0x52c0,             "S",       0, SIZE_NONE     },
  { "jirc",    0x29b0, 0xd640,             "r",       0, SIZE_NONE     },
  { "jirc",    0x2930, 0xd2c0,             "s",       0, SIZE_FIX_32   },
  { "jirc",    0x2930, 0xd2c0,             "S",       0, SIZE_NONE     },
  { "jsr",     0xB9b0, 0x4640,             "r",       0, SIZE_NONE     },
  { "jsr",     0xB930, 0x42c0,             "s",       0, SIZE_FIX_32   },
  { "jsr",     0xB930, 0x42c0,             "S",       0, SIZE_NONE     },
  { "jsrc",    0x39b0, 0xc640,             "r",       0, SIZE_NONE     },
  { "jsrc",    0x3930, 0xc2c0,             "s",       0, SIZE_FIX_32   },
  { "jsrc",    0x3930, 0xc2c0,             "S",       0, SIZE_NONE     },
  /* Note that both special register 8 and 0 (both are zero-valued) are
     recognized as jump-instructions, but special-register 0 is generated.
     /H-P 970624*/
  { "jump",    0x09b0, 0x7640,             "r",       0, SIZE_NONE     },
  { "jump",    JUMP_INDIR_OPCODE,
               JUMP_INDIR_Z_BITS,          "s",       0, SIZE_FIX_32   },
  { "jump",    JUMP_INDIR_OPCODE,
               JUMP_INDIR_Z_BITS,          "S",       0, SIZE_NONE     },
  { "link",    0x0B70, 0x0080,             "R,y",     0, SIZE_FIX_32   },
  { "link",    0x0B70, 0x0080,             "D,S",     0, SIZE_NONE     },
  { "lsl",     0x04C0, 0x0B00,             "m r,R",   0, SIZE_NONE     },
  { "lslq",    0x03c0, 0x0C20,             "c,R",     0, SIZE_NONE     },
  { "lsr",     0x07C0, 0x0800,             "m r,R",   0, SIZE_NONE     },
  { "lsrq",    0x03e0, 0x0C00,             "c,R",     0, SIZE_NONE     },
  { "lz",      0x0730, 0x08C0,             "r,R",     0, SIZE_NONE     },
  { "move",    0x0640, 0x0980,             "m r,R",   0, SIZE_NONE     },
  { "move",    0x0630, 0x09c0,             "r,P",     0, SIZE_NONE     },
  { "move",    0x0670, 0x0980,             "P,r",     0, SIZE_NONE     },
  { "move",    0x0BC0, 0x0000,             "m R,y",   0, SIZE_FIELD    },
  { "move",    0x0BC0, 0x0000,             "m D,S",   0, SIZE_NONE     },
  { "move",    0x0A40, 0x0180,             "m s,R",   0, SIZE_FIELD    },
  { "move",    0x0A40, 0x0180,             "m S,D",   0, SIZE_NONE     },
  { "move",    0x0A30, 0x01c0,             "s,P",     0, SIZE_SPEC_REG },
  { "move",    0x0A30, 0x01c0,             "S,P",     0, SIZE_NONE     },
  { "move",    0x0A70, 0x0180,             "P,y",     0, SIZE_SPEC_REG },
  { "move",    0x0A70, 0x0180,             "P,S",     0, SIZE_NONE     },
  { "movem",   0x0BF0, 0x0000,             "R,y",     0, SIZE_FIX_32   },
  { "movem",   0x0BF0, 0x0000,             "D,S",     0, SIZE_NONE     },
  { "movem",   0x0BB0, 0x0040,             "s,R",     0, SIZE_FIX_32   },
  { "movem",   0x0BB0, 0x0040,             "S,D",     0, SIZE_NONE     },
  { "moveq",   0x0240, 0x0D80,             "i,R",     0, SIZE_NONE     },
  { "movs",    0x0460, 0x0B80,             "z r,R",   0, SIZE_NONE     },
  { "movs",    0x0860, 0x0380,             "z s,R",   0, SIZE_FIELD    },
  { "movs",    0x0860, 0x0380,             "z S,D",   0, SIZE_NONE     },
  { "movu",    0x0440, 0x0Ba0,             "z r,R",   0, SIZE_NONE     },
  { "movu",    0x0840, 0x03a0,             "z s,R",   0, SIZE_FIELD    },
  { "movu",    0x0840, 0x03a0,             "z S,D",   0, SIZE_NONE     },
  { "mstep",   0x07f0, 0x0800,             "r,R",     0, SIZE_NONE     },
  { "mul",     0x0900, 0x06c0,             "m R,r",   0, SIZE_NONE     },
  { "neg",     0x0580, 0x0A40,             "m r,R",   0, SIZE_NONE     },
  { "nop",     NOP_OPCODE, NOP_Z_BITS,     "",        0, SIZE_NONE     },
  { "not",     0x8770, 0x7880,             "r",       0, SIZE_NONE     },
  { "or",      0x0740, 0x0880,             "m r,R",   0, SIZE_NONE     },
  { "or",      0x0B40, 0x0080,             "m s,R",   0, SIZE_FIELD    },
  { "or",      0x0B40, 0x0080,             "m S,D",   0, SIZE_NONE     },
  { "or",      0x0B40, 0x0480,             "m S,R,r", 0, SIZE_NONE     },
  { "orq",     0x0340, 0x0C80,             "i,R",     0, SIZE_NONE     },
  { "pop",     0x0E6E, 0x0191,             "R",       0, SIZE_NONE     },
  { "pop",     0x0e3e, 0x01c1,             "P",       0, SIZE_NONE     },
  { "push",    0x0FEE, 0x0011,             "BR",      0, SIZE_NONE     },
  { "push",    0x0E7E, 0x0181,             "BP",      0, SIZE_NONE     },
  { "ret",     0xB67F, 0x4980,             "",        1, SIZE_NONE     },
  { "retb",    0xe67f, 0x1980,             "",        1, SIZE_NONE     },
  { "reti",    0xA67F, 0x5980,             "",        1, SIZE_NONE     },
  { "sa",      0x0530+CC_A*0x1000,
               0x0AC0+(0xf-CC_A)*0x1000,   "r",       0, SIZE_NONE     },
  { "scc",     0x0530+CC_CC*0x1000,
               0x0AC0+(0xf-CC_CC)*0x1000,  "r",       0, SIZE_NONE     },
  { "scs",     0x0530+CC_CS*0x1000,
               0x0AC0+(0xf-CC_CS)*0x1000,  "r",       0, SIZE_NONE     },
  { "seq",     0x0530+CC_EQ*0x1000,
               0x0AC0+(0xf-CC_EQ)*0x1000,  "r",       0, SIZE_NONE     },
  { "setf",    0x05b0, 0x0A40,             "f",       0, SIZE_NONE     },

  { "sext",    0x0530+CC_EXT*0x1000,
               0x0AC0+(0xf-CC_EXT)*0x1000, "r",       0, SIZE_NONE     },
  { "sge",     0x0530+CC_GE*0x1000,
               0x0AC0+(0xf-CC_GE)*0x1000,  "r",       0, SIZE_NONE     },
  { "sgt",     0x0530+CC_GT*0x1000,
               0x0AC0+(0xf-CC_GT)*0x1000,  "r",       0, SIZE_NONE     },
  { "shi",     0x0530+CC_HI*0x1000,
               0x0AC0+(0xf-CC_HI)*0x1000,  "r",       0, SIZE_NONE     },
  { "shs",     0x0530+CC_HS*0x1000,
               0x0AC0+(0xf-CC_HS)*0x1000,  "r",       0, SIZE_NONE     },
  { "sle",     0x0530+CC_LE*0x1000,
               0x0AC0+(0xf-CC_LE)*0x1000,  "r",       0, SIZE_NONE     },
  { "slo",     0x0530+CC_LO*0x1000,
               0x0AC0+(0xf-CC_LO)*0x1000,  "r",       0, SIZE_NONE     },
  { "sls",     0x0530+CC_LS*0x1000,
               0x0AC0+(0xf-CC_LS)*0x1000,  "r",       0, SIZE_NONE     },
  { "slt",     0x0530+CC_LT*0x1000,
               0x0AC0+(0xf-CC_LT)*0x1000,  "r",       0, SIZE_NONE     },
  { "smi",     0x0530+CC_MI*0x1000,
               0x0AC0+(0xf-CC_MI)*0x1000,  "r",       0, SIZE_NONE     },
  { "sne",     0x0530+CC_NE*0x1000,
               0x0AC0+(0xf-CC_NE)*0x1000,  "r",       0, SIZE_NONE     },
  { "spl",     0x0530+CC_PL*0x1000,
               0x0AC0+(0xf-CC_PL)*0x1000,  "r",       0, SIZE_NONE     },
  { "sub",     0x0680, 0x0940,             "m r,R",   0, SIZE_NONE     },
  { "sub",     0x0a80, 0x0140,             "m s,R",   0, SIZE_FIELD    },
  { "sub",     0x0a80, 0x0140,             "m S,D",   0, SIZE_NONE     },
  { "sub",     0x0a80, 0x0540,             "m S,R,r", 0, SIZE_NONE     },
  { "subq",    0x0280, 0x0d40,             "I,R",     0, SIZE_NONE     },
  { "subs",    0x04a0, 0x0b40,             "z r,R",   0, SIZE_NONE     },
  { "subs",    0x08a0, 0x0340,             "z s,R",   0, SIZE_FIELD    },
  { "subs",    0x08a0, 0x0340,             "z S,D",   0, SIZE_NONE     },
  { "subs",    0x08a0, 0x0740,             "z S,R,r", 0, SIZE_NONE     },
  { "subu",    0x0480, 0x0b60,             "z r,R",   0, SIZE_NONE     },
  { "subu",    0x0880, 0x0360,             "z s,R",   0, SIZE_FIELD    },
  { "subu",    0x0880, 0x0360,             "z S,D",   0, SIZE_NONE     },
  { "subu",    0x0880, 0x0760,             "z S,R,r", 0, SIZE_NONE     },
  { "svc",     0x0530+CC_VC*0x1000,
               0x0AC0+(0xf-CC_VC)*0x1000,  "r",       0, SIZE_NONE     },
  { "svs",     0x0530+CC_VS*0x1000,
               0x0AC0+(0xf-CC_VS)*0x1000,  "r",       0, SIZE_NONE     },
 /* "swapn" is the same as NOT */
  { "swapn",   0x8770, 0x7880,             "r",       0, SIZE_NONE     },
  { "swapw",   0x4770, 0xb880,             "r",       0, SIZE_NONE     },
  { "swapnw",  0xc770, 0x3880,             "r",       0, SIZE_NONE     },
  { "swapb",   0x2770, 0xd880,             "r",       0, SIZE_NONE     },
  { "swapnb",  0xA770, 0x5880,             "r",       0, SIZE_NONE     },
  { "swapwb",  0x6770, 0x9880,             "r",       0, SIZE_NONE     },
  { "swapnwb", 0xE770, 0x1880,             "r",       0, SIZE_NONE     },
  { "swapr",   0x1770, 0xe880,             "r",       0, SIZE_NONE     },
  { "swapnr",  0x9770, 0x6880,             "r",       0, SIZE_NONE     },
  { "swapwr",  0x5770, 0xa880,             "r",       0, SIZE_NONE     },
  { "swapnwr", 0xd770, 0x2880,             "r",       0, SIZE_NONE     },
  { "swapbr",  0x3770, 0xc880,             "r",       0, SIZE_NONE     },
  { "swapnbr", 0xb770, 0x4880,             "r",       0, SIZE_NONE     },
  { "swapwbr", 0x7770, 0x8880,             "r",       0, SIZE_NONE     },
  { "swapnwbr", 0xf770, 0x0880,             "r",       0, SIZE_NONE     },

  { "test",    0x0640, 0x0980,             "m D",     0, SIZE_NONE     },
  { "test",    0x0b80, 0xf040,             "m s",     0, SIZE_FIELD    },
  { "test",    0x0b80, 0xf040,             "m S",     0, SIZE_NONE     },
  { "xor",     0x07B0, 0x0840,             "r,R",     0, SIZE_NONE     }
};

#define NUMOPCODES ((sizeof cris_opcodes)/(sizeof cris_opcodes[0]))







