/* cris.c -- Assembler for the CGA-RISC processor
   Copyright (C) 1989 Free Software Foundation, Inc.


This file is part of GAS, the GNU Assembler.

GAS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GAS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GAS; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */


/*!***************************************************************************/
/*!                                                                          */
/*!  FILE NAME:   cris.c                                                     */
/*!                                                                          */
/*!  DESCRIPTION: This file contains the machine dependent part of GNU as    */
/*!               for the CGA-RISC processor.                                */
/*!                                                                          */
/*!                                                                          */
/*!  GLOBAL       md_estimate_size_before_relax                              */
/*!  FUNCTIONS:   md_convert_frag                                            */
/*!                                                                          */
/*!               md_create_short_jump                                       */
/*!               md_create_long_jump                                        */
/*!                                                                          */
/*!               md_begin                                                   */
/*!               md_end                                                     */
/*!               md_assemble                                                */
/*!               md_atof                                                    */
/*!               md_number_to_chars                                         */
/*!               md_number_to_imm                                           */
/*!               md_number_to_disp                                          */
/*!               md_number_to_field                                         */
/*!               md_ri_to_chars                                             */
/*!               emit_relocations                                           */
/*!                                                                          */
/*!                                                                          */
/*!  LOCAL        process_instruction                                        */
/*!  FUNCTIONS :  get_bwd_size_modifier                                      */
/*!               get_bw_size_modifier                                       */
/*!               get_gen_reg                                                */
/*!               get_spec_reg                                               */
/*!               get_no_or_eff_addr_prefix_oprnd                            */
/*!               get_3_oprnd_prefix_oprnd                                   */
/*!               get_expression                                             */
/*!               get_flags                                                  */
/*!               spec_reg_size                                              */
/*!                                                                          */
/*!               gen_bdap                                                   */
/*!               branch_disp                                                */
/*!                                                                          */
/*!                                                                          */
/*!--------------------------------------------------------------------------*/
/*!                                                                          */
/*!  HISTORY:                                                                */
/*!                                                                          */
/*!  DATE      NAME                CHANGES                                   */
/*!  ----      ----                -------                                   */
/*!  920312    Mikael Asker        Initial version created by copying        */
/*!                                  sparc.c in GAS v 1.38.1                 */
/*!  920330    Mikael Asker        Included compiler.h                       */
/*!  930409    Hans-Peter Nilsson  Updated to version 0.31 of CRIS spec,     */
/*!                                  fixed some bugs, added new.             */
/*!  930830    HP                  Added some warnings for immediate values  */
/*!                                not in operand range, support for 'y'     */
/*!                                operand descriptor.                       */
/*!  950209    HP                  Bugfix in gen_bdap()                      */
/*!  960702    HP                  Moved to binutils-2.6, BFD_ASSEMBLER      */
/*!  971117    HP                  Updated C.C. flag definitions to include  */
/*!                                Svinto (seems there are no notes about    */
/*!                                previous Svinto modifications here).      */
/*!                                                                          */
/*!***************************************************************************/


#include <stdio.h>
#include <ctype.h>

/* This is ok on all current platforms (960703) at Axis, so why make
   it more difficult? */
typedef unsigned char byte;
typedef unsigned long udword;
typedef long dword;
typedef unsigned short uword;
typedef short word;
typedef int bool;

#define TRUE 1
#define FALSE 0

#ifdef __STDC__

#define PROTOTYPES
#define NO_RET void
#define NO_PARAMS void
#define CONST const

#else

#define NO_RET
#define NO_PARAMS
#define CONST

#endif

#include "opcode/cris-opc.h"
#include "as.h"

#include "tc-cris.h"

byte foo_any;
static NO_RET process_instruction();  /* 920327 MA renamed sparc_ip to process_instruction */
static bool get_bwd_size_modifier();
static bool get_bw_size_modifier();
static bool get_gen_reg();
static bool get_spec_reg();
static bool get_no_or_eff_addr_prefix_oprnd();
static bool get_3_oprnd_prefix_oprnd();
static bool get_expression();
static bool get_flags();
static byte spec_reg_size();

static NO_RET gen_bdap();
static byte branch_disp();
static NO_RET gen_cond_branch_32();

/* Static or global?  Used at all?  Other name in binutils-2.6? */
static void md_number_to_imm();
static void md_number_to_disp();
static void md_number_to_field();
static void md_ri_to_chars();
void emit_relocations();

/* handle of the OPCODE hash table */
static struct hash_control *op_hash = NULL;

static void s_bss();
extern void s_globl(), cons();

/* 920422  MA  Changed to process cris pseudoops                        */
/* 920713  MA  Removed .word pseudoop; the SPARC flag is no longer used */
/*               in read.c.                                             */
const pseudo_typeS md_pseudo_table[] = {
    { "bss",        s_bss,      0 },
    { "dword",      cons,       4 },
    { "global",     s_globl,    0 },
    { NULL,         0,          0 },
};

static bool O_option = FALSE;   /* Added 920811 by MA
                                   Meaning changed 960723 HP */

/* FRAG HANDLING */

/* Added 920320 by MA */
/* 920326 MA added STATE_LENGTH_MASK and STATE_MAX_LENGTH constants */
/* 920515 MA implemented 32-bit conditional branch case             */

/*
*
*   For Cris, we encode the relax_substateTs ( in e.g. fr_substate ) as :
*
*                       2                 1                 0
*      ---/ /--+-----------------+-----------------+-----------------+
*              |  what state ?   |            how long ?             |
*      ---/ /--+-----------------+-----------------+-----------------+
*
*   The "how long" bits are 00 = byte, 01 = word, 10 = long.
*   This is a Un*x convention.
*   Not all lengths are legit for a given value of (what state).

groups for Cris address relaxing :

1. Bcc
   length of byte, word

2. BDAP
   length of byte, word
*/

#define STATE_CONDITIONAL_BRANCH     (1)
#define STATE_BASE_PLUS_DISP_PREFIX  (2)

#define STATE_LENGTH_MASK            (3)
#define STATE_BYTE                   (0)
#define STATE_WORD                   (1)
#define STATE_LONG                   (2)
#define STATE_UNDF                   (3)  /* Symbol undefined in pass1 */
#define STATE_MAX_LENGTH             (3)


/* These displacements are relative to    */
/* the adress following the opcode word   */
/* of the instruction.  The first letter  */
/* is Byte, Word.  2nd letter is Forward, */
/* Backward.                              */

/* 920512 MA Corrected length bit field width in ENCODE_RELAX */
/*             and displacement values.                       */
/* 930408 HP Changed BDAP entry to include dword              */
#define BRANCH_BF  ( 254)
#define BRANCH_BB  (-256)
#define BRANCH_WF  (2+ 32767)
#define BRANCH_WB  (2+-32768)

#define BDAP_BF    ( 127)
#define BDAP_BB    (-128)
#define BDAP_WF    ( 32767)
#define BDAP_WB    (-32768)


#define C(a,b) ENCODE_RELAX(a,b)
 /* This macro has no side effects. */
#define ENCODE_RELAX(what,length) (((what) << 2) + (length))

const relax_typeS md_relax_table[] = {
  { 1,         1,         0, 0      },  /* error sentinel              (0,0) */
  { 1,         1,         0, 0      },  /* unused                      (0,1) */
  { 1,         1,         0, 0      },  /* unused                      (0,2) */
  { 1,         1,         0, 0      },  /* unused                      (0,3) */
  { BRANCH_BF, BRANCH_BB, 0, C(1,1) },  /* Bcc  o                      (1,0) */
  { BRANCH_WF, BRANCH_WB, 2, C(1,2) },  /* Bcc  [PC+]                  (1,1) */
  { 0,         0,        10, 0      },  /* BE, BA, JUMP (external) or  (1,2) */
                                        /*   JUMP         (always) or        */
                                        /*   Bnot_cc, JUMP (default).        */
  { 1,         1,         0, 0      },  /* unused                      (1,3) */
  { BDAP_BF,   BDAP_BB,   0, C(2,1) },  /* BDAP o                      (2,0) */
  { BDAP_WF,   BDAP_WB,   2, C(2,2) },  /* BDAP.[bw] [PC+]             (2,1) */
  { 0,         0,         4, 0      },  /* BDAP.d [PC+]                (2,2) */
};

#undef C
#undef BRANCH_BF
#undef BRANCH_BB
#undef BRANCH_WF
#undef BRANCH_WB
#undef BDAP_BF
#undef BDAP_BB
#undef BDAP_WF
#undef BDAP_WB

/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: md_estimate_size_before_relax                            */
/*#                                                                          */
/*#  PARAMETERS:    frag_ptr        Pointer to frag to be processed          */
/*#                 segment_type    What segment the frag belongs to         */
/*#                                                                          */
/*#  CHANGES:       Sets correct frag_ptr->fr_subtype value                  */
/*#                                                                          */
/*#  RETURNS:       Initial guess for frag_ptr->fr_var                       */
/*#                                                                          */
/*#  SIDE EFFECTS:  None.                                                    */
/*#                                                                          */
/*#  DESCRIPTION:   Prepares machine-dependent frags for relaxation.         */
/*#                                                                          */
/*#                 Called just before relax(). Any symbol that is now       */
/*#                 undefined will not become defined.                       */
/*#                                                                          */
/*#                 Return the correct fr_subtype in the frag.               */
/*#                                                                          */
/*#                 Return the initial "guess for fr_var" to caller. The     */
/*#                 guess for fr_var is ACTUALLY the growth beyond fr_fix.   */
/*#                 Whatever we do to grow fr_fix or fr_var contributes to   */
/*#                 our returned value.                                      */
/*#                                                                          */
/*#                 Although it may not be explicit in the frag, pretend     */
/*#                 fr_var starts with a value.                              */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME                CHANGES                                   */
/*#  ----      ----                -------                                   */
/*#  920324    Mikael Asker        Initial version                           */
/*#  920515    Mikael Asker        Complement condition codes in conditional */
/*#                                  branches correctly                      */
/*#  920624    Mikael Asker        Output word displacements in BDAP:s       */
/*#                                  correctly                               */
/*#  920702    Mikael Asker        Use gen_cond_branch_32                    */
/*#  920811    Mikael Asker        Changed BDAP offset overflow handling     */
/*#  930409    Hans-Peter Nilsson  No overflow on BDAP - will use dword size */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

int md_estimate_size_before_relax(frag_ptr,segment_type)
     register fragS  *frag_ptr;
     segT    segment_type;   /* N_DATA or N_TEXT. */
{

  register int   old_fr_fix;

  old_fr_fix = frag_ptr->fr_fix;
  switch (frag_ptr->fr_subtype)
  {

    case ENCODE_RELAX( STATE_CONDITIONAL_BRANCH, STATE_UNDF ) :
      if (S_GET_SEGMENT(frag_ptr->fr_symbol) == segment_type)
      {   /* The symbol lies in the same segment - a relaxable case */
        frag_ptr->fr_subtype = ENCODE_RELAX( STATE_CONDITIONAL_BRANCH,
          STATE_BYTE);
      }
      else {   /* Not relaxable */
        char *write_ptr;
        frag_ptr->fr_fix += 2 + 2 + 4 + 2;
        write_ptr = frag_ptr->fr_literal + old_fr_fix;
        gen_cond_branch_32(
          frag_ptr->fr_opcode,
          write_ptr,
          frag_ptr,
          frag_ptr->fr_symbol,
          (symbolS *)NULL,
          frag_ptr->fr_offset
        );
        frag_wane(frag_ptr);
      }
      break;

    case ENCODE_RELAX( STATE_BASE_PLUS_DISP_PREFIX, STATE_UNDF ) :
      /* This cannot happen: [rx+previous_symbol_in_text] cannot be
         relaxed to anything else but a 32bit value.  Code still left
	 here as a warning to others.
	  Come to think of it, you CAN relax it to something. Imagine:
	   move.x [r+32_bit_symbol],r
           move.x [32_bit_symbol],r
           move.x 32_bit_symbol,r
	  can be shortened by a word (16 bits) if we are close to the
	  beginning (or end) of the function.  Then maybe we could change the
	  32_bit_symbol into a "bdap [pc+offset_to_constant_pool]", and put
	  the 32_bit_symbol there.  Weird, to say the least. This
	  requires an extra 16-bit read, while still not shortening
	  the code size (counting in the symbol).  But its an interesting thought.
	 */
#if 0
      if (S_GET_SEGMENT(frag_ptr->fr_symbol) == segment_type)
      {    /* The symbol lies in the same segment - a relaxable case */
        frag_ptr->fr_subtype = ENCODE_RELAX( STATE_BASE_PLUS_DISP_PREFIX,
					    STATE_BYTE);
      }
      else
#endif
	if (S_GET_SEGMENT(frag_ptr->fr_symbol) != absolute_section) {
	  /* Go for dword if not absolute or same segment. */
	  frag_ptr->fr_subtype = ENCODE_RELAX( STATE_BASE_PLUS_DISP_PREFIX,
					      STATE_LONG);
	  frag_ptr->fr_var += 4;
      }
      else {   /* Absolute expression */
        dword value;
	int size = 0;

        value = S_GET_VALUE(frag_ptr->fr_symbol) + frag_ptr->fr_offset;
        if ((value < -32768) || (value > 32767)) {   /* Outside word range */
	  size = 2;
        }
	else
	  size = 1;		/* Or one (byte), but that will be
				   taken care of. */

        if ((value >= -128) && (value <= 127)) {  /* Byte displacement */
          (frag_ptr->fr_opcode)[0] = value;
        }
        else {
          /* Word or dword displacement */
          char *write_ptr;

          (frag_ptr->fr_opcode)[0] = BDAP_PC_LOW+size*16;
          (frag_ptr->fr_opcode)[1] &= 0xF0;
          (frag_ptr->fr_opcode)[1] |= BDAP_INCR_HIGH;
          frag_ptr->fr_fix += 1 << size;
          write_ptr = frag_ptr->fr_literal + old_fr_fix;
          md_number_to_chars(write_ptr,value, 1 << size);
        }
	frag_wane(frag_ptr);
      }
      break;

    default:
      break;

  }
  return ( frag_ptr->fr_var + (frag_ptr->fr_fix - old_fr_fix) );

}   /* md_estimate_size_before_relax() */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: md_convert_frag                                          */
/*#                                                                          */
/*#  PARAMETERS:    frag_ptr    Pointer to frag to be processed              */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       Nothing.                                                 */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   Performs post processing of machine-dependent frags      */
/*#                   after relaxation.                                      */
/*#                                                                          */
/*#                 Called after relax() is finished.                        */
/*#                                                                          */
/*#                 In:  Address of frag.                                    */
/*#                      fr_type == rs_machine_dependent.                    */
/*#                      fr_subtype is what the address relaxed to.          */
/*#                                                                          */
/*#                 Out: Any fixSs and constants are set up.                 */
/*#                      Caller will turn frag into a ".space 0".            */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920323    Mikael Asker       Initial version                            */
/*#  920326    Mikael Asker       Introduced the BRANCH_PC_AUTOINCREMENT_LOW */
/*#                                 and BDAP_PC_AUTOINCREMENT_LOW constants. */
/*#  920508    Mikael Asker       Use new opcode-dependent constants in      */
/*#                                 file cris-opcode.h                       */
/*#  920512    Mikael Asker       Changed branch byte displacement, renamed  */
/*#                                 some variables                           */
/*#  920515    Mikael Asker       Handle 32-bit conditional branch case      */
/*#  920702    Mikael Asker       Use gen_cond_branch_32                     */
/*#  930408    Hans-Peter Nilsson Bdap may now have 32-bit displacement.     */
/*#  960703    Hans-Peter Nilsson Changed parameters for binutils-2.6.       */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

void
md_convert_frag (abfd, sec, frag_ptr)
     bfd *abfd;
     segT sec;
     register fragS *frag_ptr;
{

  register char       *var_part_ptr;  /* Pointer to first byte in variable-  */
                                      /*   sized part of the frag.           */

  register char       *opcode_ptr;   /* Pointer to first opcode byte in frag */
  register byte       length_code;    /* 2=long 1=word 0=byte                */

  register short int  var_part_size;  /* Size in bytes of variable-          */
                                      /*   sized part of frag.               */

  /* Added to fr_fix: incl. ALL var chars. */
  register symbolS  *symbolP;
  register udword   var_part_offset;
  register udword   address_of_var_part;
  /* Where, in file space, is _var of *fragP? */
  register udword   target_address;
  /* Where, in file space, does addr point? */

  know (frag_ptr->fr_type == rs_machine_dependent);
  length_code = frag_ptr->fr_subtype & STATE_LENGTH_MASK;
  know (length_code >= 0 && length_code < STATE_MAX_LENGTH);
  var_part_offset = frag_ptr->fr_fix;
  var_part_ptr = frag_ptr->fr_literal + var_part_offset;
  opcode_ptr = frag_ptr->fr_opcode;

  symbolP = frag_ptr->fr_symbol;
  target_address = (symbolP ? S_GET_VALUE(symbolP) + frag_ptr->fr_symbol->sy_frag->fr_address
                    : 0 ) + frag_ptr->fr_offset;
  address_of_var_part = frag_ptr->fr_address + var_part_offset;
  switch (frag_ptr->fr_subtype)
  {

    case ENCODE_RELAX ( STATE_CONDITIONAL_BRANCH, STATE_BYTE ) :
      opcode_ptr[0] = branch_disp((word)
        (target_address - address_of_var_part));
      var_part_size = 0;
      break;

    case ENCODE_RELAX ( STATE_CONDITIONAL_BRANCH, STATE_WORD ) :
      /* Quick immediate -> PC autoincrement */
        opcode_ptr[0] = BRANCH_PC_LOW;
        opcode_ptr[1] &= 0xF0;
        opcode_ptr[1] |= BRANCH_INCR_HIGH;
      md_number_to_chars (var_part_ptr, (long)(target_address -
        (address_of_var_part + 2)), 2);
      var_part_size = 2;
      break;

    case ENCODE_RELAX ( STATE_CONDITIONAL_BRANCH, STATE_LONG ) :
      gen_cond_branch_32(
        frag_ptr->fr_opcode,
        var_part_ptr,
        frag_ptr,
        frag_ptr->fr_symbol,
        (symbolS *)NULL,
        frag_ptr->fr_offset
      );
      var_part_size = 2 + 2 + 4 + 2;
      break;

    case ENCODE_RELAX ( STATE_BASE_PLUS_DISP_PREFIX, STATE_BYTE ) :
      var_part_ptr[0] = target_address - (address_of_var_part + 1);
      var_part_size = 0;
      break;

    case ENCODE_RELAX ( STATE_BASE_PLUS_DISP_PREFIX, STATE_WORD ) :
      /* Quick immediate -> PC autoincrement */
        opcode_ptr[0] = BDAP_PC_LOW + (1 << 4);
        opcode_ptr[1] &= 0xF0;
        opcode_ptr[1] |= BDAP_INCR_HIGH;
      md_number_to_chars (var_part_ptr, (long)(target_address), 2);
      var_part_size = 2;
      break;

    case ENCODE_RELAX ( STATE_BASE_PLUS_DISP_PREFIX, STATE_LONG ) :
      /* word -> dword */
      opcode_ptr[0] = BDAP_PC_LOW + (2 << 4);
      opcode_ptr[1] &= 0xf0;
      opcode_ptr[1] |= BDAP_INCR_HIGH;
      if (frag_ptr->fr_symbol == NULL)
	md_number_to_chars (var_part_ptr, frag_ptr->fr_offset, 4);
      else
	fix_new(frag_ptr, var_part_ptr - frag_ptr->fr_literal, 4,
		frag_ptr->fr_symbol, frag_ptr->fr_offset,
		NOT_PC_RELATIVE, CRIS_RELOC_32);
      var_part_size = 4;
      break;

    default:
      BAD_CASE (frag_ptr->fr_subtype);
      break;
    }
  frag_ptr->fr_fix += var_part_size;

}  /* md_convert_frag */

/* BROKEN-WORD FUNCTION SUPPORT */

int md_short_jump_size = 6;  /* Changed 920713 by MA because of delay slots */
int md_long_jump_size = 6;   /* Changed 920326 by MA */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: md_create_short_jump                                     */
/*#                                                                          */
/*#  PARAMETERS:    store_ptr  Where to store the jump instruction           */
/*#                 from_addr  Address of the jump instruction               */
/*#                 to_addr    Destination address of the jump               */
/*#                 frag       Which frag the destination address operand    */
/*#                              lies in                                     */
/*#                 to_symbol  Destination symbol                            */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       Nothing.                                                 */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   Generates a short jump around a secondary jump table.    */
/*#                 Used by the broken-word function                         */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920326    Mikael Asker       Initial version                            */
/*#  920424    Mikael Asker       Adjusted branch displacement               */
/*#  920425    Mikael Asker       Optimize "short" short jumps               */
/*#  920713    Mikael Asker       Put NOP:s in delay slots                   */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

void
md_create_short_jump(store_ptr, from_addr, to_addr, frag, to_symbol)
    char     *store_ptr;
    addressT     from_addr, to_addr;
    fragS    *frag;
    symbolS  *to_symbol;
{
  dword distance;

  distance = to_addr - from_addr;
  if ((-254 <= distance) && (distance <= 256)) {  /* "Short" short jump */
    /* BA  distance - 2 */
      store_ptr[0] = branch_disp(distance-2);
      store_ptr[1] = BA_QUICK_HIGH;
    /* NOP */
      md_number_to_chars(store_ptr+2, NOP_OPCODE, 2);
  }
  else {  /* "Long" short jump */
    /* BA (PC+) */
      md_number_to_chars(store_ptr,BA_PC_INCR_OPCODE,2);
    /* .WORD  distance - 4 */
      md_number_to_chars(store_ptr+2, (long)(distance-4), 2);
    /* nop */
      md_number_to_chars(store_ptr+4, NOP_OPCODE, 2);
  }
}  /* md_create_short_jump */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: md_create_long_jump                                      */
/*#                                                                          */
/*#  PARAMETERS:    store_ptr  Where to store the jump instruction           */
/*#                 from_addr  Address of the jump instruction               */
/*#                 to_addr    Destination address of the jump               */
/*#                 frag       Which frag the destination address operand    */
/*#                              lies in                                     */
/*#                 to_symbol  Destination symbol                            */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       Nothing.                                                 */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   Generates a long jump in a secondary jump table.         */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920327    Mikael Asker       Initial version                            */
/*#  920425    Mikael Asker       Optimize "short" long jumps                */
/*#  920604    Miakel Asker       Changed place in frag and inserted for-    */
/*#                                 gotten relocation type in fix_new call   */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

void
md_create_long_jump(store_ptr, from_addr, to_addr, frag, to_symbol)
    char     *store_ptr;
    addressT     from_addr, to_addr;
    fragS    *frag;
    symbolS  *to_symbol;
{
  dword distance;

  distance = to_addr - from_addr;
  if ((-32763 <= distance) && (distance <= 32772)) {  /* "Short" long jump */
    md_create_short_jump(store_ptr,from_addr,to_addr,frag,to_symbol);
  }
  else {  /* "Long" long jump */
    /* JUMP (PC+) */
      md_number_to_chars(store_ptr,JUMP_PC_INCR_OPCODE,2);
    /* .DWORD to_addr */
      fix_new(
        frag,
        store_ptr + 2 - frag->fr_literal,
        4,
        to_symbol,
        (long)0,
        NOT_PC_RELATIVE,
        CRIS_RELOC_32
      );
  }
}  /* md_create_long_jump */

/* 920605 MA  Changed the a_machtype field to 0xFF which is an invalid value */
/*              on SUN systems.                                              */
#if 0
int omagic  =  (0x1FF << 16) | OMAGIC;  /* Magic number for header */
#endif

/* This array holds the chars that always start a comment.  If the
    pre-processor is disabled, these aren't very useful */
/* Changed 920312 by MA */
const char comment_chars[] = ";";	/* JF removed '|' from comment_chars */

/* This array holds the chars that only start a comment at the beginning of
   a line.  If the line seems to have the form '# 123 filename'
   .line and .file directives will appear in the pre-processed output */
/* Note that input_file.c hand checks for '#' at the beginning of the
   first line of the input file.  This is because the compiler outputs
   #NO_APP at the beginning of its output. */
/* Also note that '/*' will always start a comment */
const char line_comment_chars[] = "#";
const char line_separator_chars[] = "";

/* Chars that can be used to separate mant from exp in floating point nums */
/* Changed 920605 by MA */
char EXP_CHARS[] = "eE";

/* Chars that mean this number is a floating point constant */
/* As in 0f12.456 */
/* or    0d1.2345e12 */
/* Changed 920605 by MA */
char FLT_CHARS[] = "rRsSfFdDxXeEpP";

/* Also be aware that MAXIMUM_NUMBER_OF_CHARS_FOR_FLOAT may have to be
   changed in read.c .  Ideally it shouldn't have to know about it at all,
   but nothing is ideal around here.
 */
/* 920331 MA changed reloc_info_sparc to reloc_info_cris */


#ifdef __STDC__
#if 0
static void print_insn(struct sparc_it *insn);
#endif
#else
#if 0
static void print_insn();
#endif
#endif

/* 920327 MA Replaced special_case and SPECIAL_CASE_SET by prefix */
/* 920403 MA deleted prefix */

/* 920422 MA deleted s_reserve, s_common, s_seg, s_data_1, s_proc */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: s_bss                                                    */
/*#                                                                          */
/*#  PARAMETERS:    None                                                     */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       Nothing.                                                 */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This procedure handles .bss directives.                  */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920422    Mikael Asker       Initial version                            */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

static void
s_bss(NO_PARAMS)
{
  as_fatal("The .bss pseudoop is not implemented !!!");
}


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: md_begin                                                 */
/*#                                                                          */
/*#  PARAMETERS:    None                                                     */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       Nothing.                                                 */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This function is called once, at assembler startup time. */
/*#                 It should set up all the tables, etc. that the MD part   */
/*#                 of the assembler will need.                              */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920312    Mikael Asker       Changed sparc_opcodes to cris_opcodes      */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

void md_begin()
{
  register char *retval = NULL;
  bool lose = FALSE;
  register unsigned int i = 0;

  op_hash = hash_new();
  if (op_hash == NULL)
    as_fatal("Virtual memory exhausted");

  while (i < NUMOPCODES)
    {
      const char *name = cris_opcodes[i].name;
      retval = hash_insert(op_hash, name, &cris_opcodes[i]);
      if(retval != NULL && *retval != '\0')
	{
	  fprintf (stderr, "internal error: can't hash `%s': %s\n",
		   cris_opcodes[i].name, retval);
	  lose = TRUE;
	}
      do
	{
	  if (cris_opcodes[i].match & cris_opcodes[i].lose)
	    {
	      fprintf (stderr, "internal error: losing opcode: `%s' \"%s\"\n",
		       cris_opcodes[i].name, cris_opcodes[i].args);
	      lose = TRUE;
	    }
	  ++i;
	} while (i < NUMOPCODES
		 && !strcmp(cris_opcodes[i].name, name));
    }

  if (lose)
    as_fatal ("Broken assembler.  No assembly attempted.");

}  /* md_begin */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: md_end                                                   */
/*#                                                                          */
/*#  PARAMETERS:    None                                                     */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       Nothing.                                                 */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This function is called once, at assembler startup time. */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920312    Mikael Asker       Changed sparc_opcodes to cris_opcodes      */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

void md_end()
{
    return;
}


/* Created 920522 by MA */
/* 920603 MA Added reloc field */
/* 920626 MA Added PREFIX_PUSH */

enum prefix_kind_type { PREFIX_NONE, PREFIX_BDAP, PREFIX_BIAP, PREFIX_DIP,
                        PREFIX_PUSH };

typedef struct cris_prefix_type {
  enum prefix_kind_type  kind;
  byte                   base_reg_number;
  expressionS            expr;
  char                   *mode_name_ptr, *error_msg_ptr;
  uword                  opcode;
  enum bfd_reloc_code_real        reloc;
} cris_prefix_type;


/* 920312 MA Renamed sparc_it to cris_it, retyped opcode to unsigned */
/* 920327 MA Replaced the_insn and set_insn with output_prefix and output_instruction */
/* 920410 MA changed struct cris_it to typedef cris_instruction_type, moved output_instruction and output_prefix into md_assemble */
/* 920511 MA redeclared pcrel to pc_rel_type, added branch_inst field */
/* 920603 MA Replaced reloc field by imm_oprnd_size field */

typedef struct cris_instruction_type {
    char             *error;
    uword            opcode;
    struct nlist     *nlistp;
    expressionS      expr;
    enum pcrel_type  pcrel;
    enum bfd_reloc_code_real  reloc;
    bool             branch_inst;
    byte             imm_oprnd_size;
} cris_instruction_type;


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: md_assemble                                              */
/*#                                                                          */
/*#  PARAMETERS:    str   Pointer to string containing the source line       */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       Nothing.                                                 */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   Assemble one instruction.                                */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920312    Mikael Asker       Initial version                            */
/*#  920327    Mikael Asker       Renamed toP to write_ptr, the_insn to      */
/*#                                 output_instructon and sparc_ip to        */
/*#                                 process_instruction. Replaced            */
/*#                                 special_case test by prefix test.        */
/*#  920403    Mikael Asker       Replaced global prefix variable by         */
/*#                                 prefixed_inst.                           */
/*#  920410    Mikael Asker       Renamed output_prefix to prefix            */
/*#  920413    Mikael Asker       Added BDAP offset handling                 */
/*#  920504    Mikael Asker       Corrected handling of symbol-dependent     */
/*#                                 instruction stuff.                       */
/*#  920511    Mikael Asker       Added branch instruction handling          */
/*#  920522    Mikael Asker       Redeclared prefix to cris_prefix_type,     */
/*#                                 removed prefixed_inst.                   */
/*#  920527    Mikael Asker       Completed BIAP and DIP prefix handling     */
/*#  920603    Mikael Asker       Output absolute mode addresses and         */
/*#                                 immediate values.                        */
/*#  920626    Mikael Asker       Added PREFIX_PUSH handling                 */
/*#  920702    Mikael Asker       Handle X_add_number in conditional         */
/*#                                 branch operand expressions. Use          */
/*#                                 gen_cond_branch_32.                      */
/*#  920803    Mikael Asker       Correct size values in prefixes in         */
/*#                                 PUSH:es of special registers.            */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

void md_assemble(str)
  char *str;
{

  cris_instruction_type  output_instruction;
  cris_prefix_type       prefix;
  char                   *opcode_ptr, *p;

  assert(str);   /* From <assert.h>; for debugging */
  process_instruction(str,&output_instruction,&prefix);

  switch (prefix.kind) {

    case PREFIX_NONE :  break;

    case PREFIX_BDAP :  gen_bdap(prefix.base_reg_number,&(prefix.expr),
                          prefix.mode_name_ptr);
      /* Kludge to use when we only want prefix insn; no branch can */
      /* have prefix. */
                        if (output_instruction.branch_inst)
			  return;
                        break;

    case PREFIX_BIAP :
    case PREFIX_DIP  :
                        opcode_ptr = frag_more(2);
                        /* Put out the prefix opcode */
                          md_number_to_chars(opcode_ptr,
                            (long)(prefix.opcode),2);
                        if (prefix.reloc != CRIS_RELOC_NONE) {
                          /* Put out absolute mode address */
                          p = frag_more(4);
                          know(prefix.expr.X_subtract_symbol == 0); /*
                                                                      Or what? */
                          fix_new(
                            frag_now,                         /* which frag */
                            (p - frag_now->fr_literal),       /* where      */
                            4,                                /* size       */
                            prefix.expr.X_add_symbol,
                            prefix.expr.X_add_number,
                            NOT_PC_RELATIVE,
                            prefix.reloc
                          );
                        }
                        break;

    case PREFIX_PUSH :  opcode_ptr = frag_more(2);
                        /* Put out the prefix opcode */
      /* WARNING : Next line contains magic about opcodes.
         Change the '== 7' to something appropriate. */
                          if (((output_instruction.opcode & 0xF0) >> 4 ) == 7 )
                          {
                            /* Special register */
                            opcode_ptr[0] = - spec_reg_size((
                              output_instruction.opcode & 0xF000) >> 12);
                          }
                          else {
                            /* General register */
                            opcode_ptr[0] = -4;
                          }
                          opcode_ptr[1] = 0xE0 + (BDAP_QUICK_OPCODE >> 8);
                        break;

    default :           as_fatal("internal error - bad prefix type : %d",
                          prefix.kind);

  }  /* switch (prefix.kind) */

  opcode_ptr = frag_more(2);

  /* Put out the instruction opcode */
    md_number_to_chars(opcode_ptr, (long)(output_instruction.opcode), 2);

  /* put out the symbol-dependent instruction stuff */
    if (output_instruction.branch_inst) {
      segT  to_seg;
      bool  is_undefined;
      byte  length_code;

      is_undefined = (output_instruction.expr.X_op != O_constant
                      && (to_seg = S_GET_SEGMENT(output_instruction.expr.X_add_symbol))
                      == undefined_section);
      if (output_instruction.expr.X_op == O_constant 
          || to_seg == now_seg || is_undefined ) {
        /* If is_undefined, then the expression may BECOME now_seg */
        if (is_undefined) {
          length_code = STATE_UNDF;
        }
        else {
          length_code = STATE_BYTE;
        }
        frag_var(
          rs_machine_dependent,
          10,
          0,
          ENCODE_RELAX(STATE_CONDITIONAL_BRANCH,length_code),
          output_instruction.expr.X_add_symbol,
          output_instruction.expr.X_add_number,
          opcode_ptr
        );
      }
      else {  /* (to_seg != now_seg) && (to_seg != undefined_section) */
        gen_cond_branch_32(
          opcode_ptr,
          frag_more(10),
          frag_now,
          output_instruction.expr.X_add_symbol,
          (symbolS *)NULL,
          output_instruction.expr.X_add_number
        );
      }
    }  /* Branch instruction */
    else if (output_instruction.imm_oprnd_size > 0) {   /* Immediate operand */
      enum bfd_reloc_code_real  reloc;
      switch (output_instruction.imm_oprnd_size) {
        case 2  :  reloc = CRIS_RELOC_16;   break;
        case 4  :  reloc = CRIS_RELOC_32;   break;
        default :  as_fatal("INTERNAL ERROR : Illegal immediate operand size");
      }
      p = frag_more(output_instruction.imm_oprnd_size);
      know(output_instruction.expr.X_subtract_symbol == 0) /* Or what? */
      fix_new(
        frag_now,                                  /* which frag */
        (p - frag_now->fr_literal),                /* where */
        output_instruction.imm_oprnd_size,         /* size */
        output_instruction.expr.X_add_symbol,
        output_instruction.expr.X_add_number,
        output_instruction.pcrel,
        reloc
      );
    }   /* Immediate operand */
    else if (output_instruction.reloc != CRIS_RELOC_NONE) {
      /* Quick immediate operand */
      know(output_instruction.expr.X_subtract_symbol == 0);/* Or else... */
      fix_new(
        frag_now,                                   /* which frag */
        (opcode_ptr - frag_now->fr_literal),        /* where      */
        2,                                          /* size       */
        output_instruction.expr.X_add_symbol,
        output_instruction.expr.X_add_number,
        output_instruction.pcrel,
        output_instruction.reloc
      );      
    }

}  /* md_assemble */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: process_instruction                                      */
/*#                                                                          */
/*#  PARAMETERS:    str                      Pointer to string containing    */
/*#                                            the source line               */
/*#                                                                          */
/*#                 output_instruction_ptr   Pointer to structure containing */
/*#                                            the resulting opcode and      */
/*#                                            operands.                     */
/*#                                                                          */
/*#                 prefix_ptr               Pointer to structure containing */
/*#                                            the optional instruction      */
/*#                                            prefix                        */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       Nothing.                                                 */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   Parse an assembly language instruction.                  */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920312    Mikael Asker       Initial version                            */
/*#  920312    Mikael Asker       Changed sparc_opcodes to cris_opcodes,     */
/*#                                 sparc_opcode insn to cris_opcode         */
/*#                                 instruction.                             */
/*#  920327    Mikael Asker       Renamed sparc_ip to process_instruction    */
/*#  920401    Mikael Asker       Changed opcode recognition to stop at      */
/*#                                 points before size modifiers.  Added m,  */
/*#                                 s and S argument descriptors.            */
/*#  920407    Mikael Asker       Corrected 's' prefixtest & reg_number      */
/*#                                 insert                                   */
/*#  920410    Mikael Asker       Added expr_ptr and error_ptr parameters in */
/*#                                 get_expression calls. Added              */
/*#                                 output_instruction_ptr and prefix_ptr    */
/*#                                 parameters.                              */
/*#  920504    Mikael Asker       Added 'o' argument descriptor              */
/*#  920511    Mikael Asker       Added output_instruction_ptr->branch_inst  */
/*#  920522    Mikael Asker       Redeclared prefix_ptr to *cris_prefix_type,*/
/*#                                 removed prefixed_inst_ptr parameter.     */
/*#  920527    Mikael Asker       Added mode declaration                     */
/*#  920603    Mikael Asker       Initialize prefix_ptr->reloc, process      */
/*#                                 immediate mode expressions.              */
/*#  920604    Mikael Asker       Added 'x' and 'z' argument descriptors.    */
/*#  920626    Mikael Asker       Added 'B' argument descriptor.             */
/*#  920701    Mikael Asker       Made opcode recognition case-insensitive   */
/*#  920717    Mikael Asker       Added 'M' argument descriptor              */
/*#  930408    Hans-Peter Nilsson Added 'O','f' and fixed other bugs.        */
/*#  930701    HP                 Bug solved; reg_number and spec_reg_number */
/*#                               not used as their name implied.  Caused    */
/*#                               bug with instructions matching "move s,P". */
/*#  930830    HP                 Added 'y' descriptor; like 's' but         */
/*#                               does not allow constants.  Added warning   */
/*#                               messages for immediate values not in       */
/*#                               operand range.                             */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

static NO_RET process_instruction(
  str, output_instruction_ptr, prefix_ptr
)
    char                   *str;
    cris_instruction_type  *output_instruction_ptr;
    cris_prefix_type       *prefix_ptr;
{

#define NO_CHAR (0)

    char                *s;
    char                modified_char = NO_CHAR;
    const char          *args;
    struct cris_opcode  *instruction;
    char                *argsStart;
    bool                match = FALSE;
    byte                mode, reg_number;         /* Added 920330 by MA */
    byte                size_bits;                /* Added 920401 by MA */

    /* Find end of opcode mnemonic */
      for (argsStart = str; islower(*argsStart) || isupper(*argsStart);
      ++argsStart)
	;

    /* Put a '\0' after the mnemonic */
      modified_char = NO_CHAR;
      switch (*argsStart) {

        case '\0':
          break;

        case '.':
          /* Put back the modified character later */
            modified_char = *argsStart;

          /* FALLTHROUGH */

        case ' ':
          /* Consume the character after the mnemonic and replace */
          /* it with '\0'.                                        */
            *argsStart++ = '\0';
          break;

        default:
          as_bad("Unknown opcode: `%s'", str);
          exit(1);

      }

    /* Translate mnemonic to lower-case */
      for (s = str; *s != '\0'; ++s) {
        if (isupper(*s)) {
          *s += 32;
        }
      }

    if ((instruction = (struct cris_opcode *) hash_find(op_hash, str)) == NULL)
    {
	as_bad("Unknown opcode: `%s'", str);
	return;
    }

    switch (modified_char) {
      case NO_CHAR :  break;
      default      :  *--argsStart = modified_char;
    }

    s = argsStart;
    for (;;) {   /* Try to match an opcode table slot */

      bool  imm_expr_found;
      byte  spec_reg_number;

        /* Initialize *prefix_ptr */
          prefix_ptr->kind = PREFIX_NONE;
          prefix_ptr->reloc = CRIS_RELOC_NONE;

        /* Initialize *output_instruction_ptr */
          bzero(output_instruction_ptr, sizeof(*output_instruction_ptr));
          output_instruction_ptr->opcode = instruction->match;
          output_instruction_ptr->reloc = CRIS_RELOC_NONE;
          output_instruction_ptr->branch_inst = FALSE;
          output_instruction_ptr->imm_oprnd_size = 0;

        imm_expr_found = FALSE;

	/*
	 * Build the opcode, checking as we go to make
	 * sure that the operands match
	 */
	for (args = instruction->args; ; ++args) {
	    switch (*args) {

	    case '\0':  /* end of args */
		if (*s == '\0') {
		    match = TRUE;
		}
		break;

	    case '+':
		if (*s == '+') {
		    ++s;
		    continue;
		}
		if (*s == '-') {
		    ++s;	/* HP: added 930409 */
		    continue;
		}
		break;

	    case '[':   /* these must match exactly */
	    case ']':
	    case ',':
	    case ' ':
		if (*s++ == *args)
		    continue;
		break;

	    case '#':   /* must be at least one digit */
		if (isdigit(*s++)) {
		    while (isdigit(*s)) {
			++s;
		    }
		    continue;
		}
		break;

            case 'B':   /* Not realy an operand; outputs a "BDAP -size,SP" */
                        /* prefix for the PUSH instructions                */
                     prefix_ptr->kind = PREFIX_PUSH;
                     continue;

		/* This should not be matched in the assembler. It is a
		   branch with 16-bit displacement.  The assembler
		   will create them from the 8-bit flavor when
		   necessary. */
	      case 'b':
		break;

            case 'c':  /* 5-bit unsigned immediate in bits <4:0> */
                     if (! get_expression(&s,
                     &(output_instruction_ptr->expr),
                     &(output_instruction_ptr->error))) {
                       break;
                     }
                     else {
		       if (output_instruction_ptr->expr.X_op == O_constant
			   && (output_instruction_ptr->expr.X_add_number < 0
			       || output_instruction_ptr->expr.X_add_number > 31))
			   as_bad("Immediate value not in 5 bit unsigned range: %ld; truncated",
				   output_instruction_ptr->expr.X_add_number);
                       output_instruction_ptr->reloc = CRIS_RELOC_UNSIGNED_5;
                       continue;
                     }

            case 'C':  /* 4-bit unsigned immediate in bits <3:0> */
                     if (! get_expression(&s,
                     &(output_instruction_ptr->expr),
                     &(output_instruction_ptr->error))) {
                       break;
                     }
                     else {
		       if (output_instruction_ptr->expr.X_op == O_constant
			   && (output_instruction_ptr->expr.X_add_number < 0
			       || output_instruction_ptr->expr.X_add_number > 15))
			   as_bad("Immediate value not in 4 bit unsigned range: %ld; truncated",
				   output_instruction_ptr->expr.X_add_number);
                       output_instruction_ptr->reloc = CRIS_RELOC_UNSIGNED_4;
                       continue;
                     }

            case 'D':  /* General register in bits <15:12> and <3:0> */
                     if (! get_gen_reg(&s,&reg_number)) {
                       break;
                     }
                     else {
                       output_instruction_ptr->opcode |= reg_number /* << 0 */;
                       output_instruction_ptr->opcode |= reg_number << 12;
                       continue;
                     }

	      case 'f': {
		byte flags = 0;
		if (!get_flags(&s,&flags))
		  break;
		output_instruction_ptr->opcode
		  |= ((flags & 0xf0) << 8) | (flags & 0xf);
		continue;
	      }

            case 'i':  /* 6-bit signed immediate in bits <5:0> */
                     if (! get_expression(&s,
                     &(output_instruction_ptr->expr),
                     &(output_instruction_ptr->error))) {
                       break;
                     }
                     else {
		       if (output_instruction_ptr->expr.X_op == O_constant
			   && (output_instruction_ptr->expr.X_add_number < -32
			       || output_instruction_ptr->expr.X_add_number > 31))
			   as_bad("Immediate value not in 6 bit range: %ld; truncated",
				   output_instruction_ptr->expr.X_add_number);
                       output_instruction_ptr->reloc = CRIS_RELOC_SIGNED_6;
                       continue;
                     }

            case 'I':  /* 6-bit unsigned immediate in bits <5:0> */
                     if (! get_expression(&s,
                     &(output_instruction_ptr->expr),
                     &(output_instruction_ptr->error))) {
                       break;
                     }
                     else {
		       if (output_instruction_ptr->expr.X_op == O_constant
			   && (output_instruction_ptr->expr.X_add_number < 0
			       || output_instruction_ptr->expr.X_add_number > 63))
			   as_bad("Immediate value not in 6 bit unsigned range: %ld; truncated",
				   output_instruction_ptr->expr.X_add_number);
                       output_instruction_ptr->reloc = CRIS_RELOC_UNSIGNED_6;
                       continue;
		     }

            case 'M':  /* Size modifier (B, W or D) for CLEAR instructions */
                     if (! get_bwd_size_modifier(&s,&size_bits)) {
                       break;
                     }
                     else {
                       switch (size_bits) {
                         case 0 :
                           output_instruction_ptr->opcode |= 0 << 12;  break;
                         case 1 :
                           output_instruction_ptr->opcode |= 4 << 12;  break;
                         case 2 :
                           output_instruction_ptr->opcode |= 8 << 12;  break;
                       }
                       continue;
                     }

            case 'm':  /* Size modifier (B, W or D) in bits <5:4> */
                     if (! get_bwd_size_modifier(&s,&size_bits)) {
                       break;
                     }
                     else {
                       output_instruction_ptr->opcode |= size_bits << 4;
                       continue;
                     }

            case 'o':  /* branch expression */
                     if (! get_expression(&s,
                     &(output_instruction_ptr->expr),
                     &(output_instruction_ptr->error))) {
                       break;
                     }
                     else {
                       output_instruction_ptr->branch_inst = TRUE;
                       continue;
		     }

	      case 'O':		/* BDAP expression any size. */
                     if (! get_expression(&s,
					  &(prefix_ptr->expr),
					  &(output_instruction_ptr->error))) {
                       break;
                     }
                     else {
		       byte base_reg;
		       if (*s != ',')
			 break;
		       s++;
		       if (!get_gen_reg(&s,&(prefix_ptr->base_reg_number)))
			   break;
		       /* Kludge: say we have a branch with
			  a prefix, to void the "real" instruction.
			   Since 'O' is used with bdap, we have no
			   "real" instruction. */
		       prefix_ptr->kind = PREFIX_BDAP;
                       output_instruction_ptr->branch_inst = TRUE;
                       continue;
		     }
		
            case 'P':  /* Special register in bits <15:12> */
                     if (! get_spec_reg(&s,&spec_reg_number)) {
                       break;
                     }
                     else {
                       output_instruction_ptr->opcode |= spec_reg_number << 12;
                       continue;
                     };

/* HP: not used any more */
#if 0
            case 'p':  /* Special register in bits <3:0> */
                     if (! get_spec_reg(&s,&spec_reg_number)) {
                       break;
                     }
                     else {
                       output_instruction_ptr->opcode |= spec_reg_number /* << 0 */;
                       continue;
                     };
#else
/* HP: used to tell disassembler this is a prefix insn. Ignore in assembler. */
	      case 'p':
		continue;
#endif
            case 'R':  /* General register in bits <15:12> */
                     if (! get_gen_reg(&s,&reg_number)) {
                       break;
                     }
                     else {
                       output_instruction_ptr->opcode |= reg_number << 12;
                       continue;
                     }

            case 'r':  /* General register in bits <3:0> */
                     if (! get_gen_reg(&s,&reg_number)) {
                       break;
                     }
                     else {
                       output_instruction_ptr->opcode |= reg_number /* << 0 */;
                       continue;
                     }

            case 'S' :  /* Source operand in bit <10> and a prefix; a 3-operand prefix. */
                     if (! get_3_oprnd_prefix_oprnd(&s,prefix_ptr)) {
                       break;
                     }
                     else {
                       continue;
                     }

            case 's' :  /* Source operand in bits <10>, <3:0> and optionally a prefix */
                        /* no prefix or an effective address prefix. */
                     if (! get_no_or_eff_addr_prefix_oprnd(&s,prefix_ptr,&mode,
                     &reg_number,&imm_expr_found,
                     &(output_instruction_ptr->expr))) {
                       break;
                     }
                     else {
                       if (prefix_ptr->kind != PREFIX_NONE) {  /* Prefix */
                         output_instruction_ptr->opcode |=
                           (AUTOINCR_BIT * 0x0100);
                       }
                       else {  /* No prefix */
                         output_instruction_ptr->opcode |= (mode << 10);
                       }
                       output_instruction_ptr->opcode |=
                         reg_number /* << 0 */ ;
                       continue;
                     }

            case 'x':  /* Rs.m in bits <15:12> and <5:4> */
                     if (! get_gen_reg(&s,&reg_number)) {
                       break;
                     }
                     else if (! get_bwd_size_modifier(&s,&size_bits)) {
                       break;
                     }
                     else {
                       output_instruction_ptr->opcode |=
                         ( reg_number << 12 ) | ( size_bits << 4 );
                       continue;
                     }

            case 'y' :  /* Source operand in bits <10>, <3:0> and optionally a prefix */
                        /* no prefix or an effective address prefix. */
	                /* Note the difference to 's': it does not allow an */
	                /* immediate expression. */
                     if (! get_no_or_eff_addr_prefix_oprnd(&s,prefix_ptr,&mode,
							   &reg_number,&imm_expr_found,
							   &(output_instruction_ptr->expr))
			 || imm_expr_found) {
                       break;
                     }
                     else {
                       if (prefix_ptr->kind != PREFIX_NONE) {  /* Prefix */
                         output_instruction_ptr->opcode |=
                           (AUTOINCR_BIT * 0x0100);
                       }
                       else {  /* No prefix */
                         output_instruction_ptr->opcode |= (mode << 10);
                       }
                       output_instruction_ptr->opcode |=
                         reg_number /* << 0 */ ;
                       continue;
                     }

            case 'z':  /* Size modifier (B or W ) in bit <4> */
                     if (! get_bw_size_modifier(&s,&size_bits)) {
                       break;
                     }
                     else {
                       output_instruction_ptr->opcode |= size_bits << 4;
                       continue;
                     }

	    default:
		/* Dont abort in my nose. Give appropriate error msg. */
		as_fatal("Internal: Cannot find match character '%c'",*args);
	    }
	    break;
	}
        /* Error */
	if (match == FALSE)
	  {
	    /* Args don't match.  */
	    if (&instruction[1] - cris_opcodes < NUMOPCODES
		&& !strcmp(instruction->name, instruction[1].name))
	      {
		++instruction;
		s = argsStart;
		continue;
	      }
	    else
	      {
		as_bad("Illegal operands");
		return;
	      }
	  }
        else   /* Match */
          if (imm_expr_found) {   /* Immediate mode operand */
            /* Compute operand length */
            switch (instruction->imm_oprnd_size) {
              case SIZE_NONE :
                as_fatal("INTERNAL ERROR : No immediate operand size");
                break;
              case SIZE_FIX_32 :
                output_instruction_ptr->imm_oprnd_size = 4;
                break;
              case SIZE_SPEC_REG :
                switch (spec_reg_size(spec_reg_number)) {
                  case 1 :
		       if (output_instruction_ptr->expr.X_op == O_constant
			   && (output_instruction_ptr->expr.X_add_number < -128
			       || output_instruction_ptr->expr.X_add_number > 255))
			   as_bad("Immediate value not in 8 bit range: %ld; truncated",
				   output_instruction_ptr->expr.X_add_number);
                  case 2 :
		       if (output_instruction_ptr->expr.X_op == O_constant
			   && (output_instruction_ptr->expr.X_add_number < -32768
			       || output_instruction_ptr->expr.X_add_number > 65535))
			   as_bad("Immediate value not in 16 bit range: %ld; truncated",
				   output_instruction_ptr->expr.X_add_number);
                    output_instruction_ptr->imm_oprnd_size = 2;
                    break;
                  case 4 :
                    output_instruction_ptr->imm_oprnd_size = 4;
                    break;
	        }
                break;
              case SIZE_FIELD :
                switch (size_bits) {
                  case 0 :
		    if (output_instruction_ptr->expr.X_op == O_constant
			&& (output_instruction_ptr->expr.X_add_number < -128
			    || output_instruction_ptr->expr.X_add_number > 255))
		      as_bad("Immediate value not in 8 bit range: %ld; truncated",
			      output_instruction_ptr->expr.X_add_number);
                  case 1 :
		    if (output_instruction_ptr->expr.X_op == O_constant
			&& (output_instruction_ptr->expr.X_add_number < -32768
			    || output_instruction_ptr->expr.X_add_number > 65535))
		      as_bad("Immediate value not in 16 bit range: %ld; truncated",
			      output_instruction_ptr->expr.X_add_number);
                    output_instruction_ptr->imm_oprnd_size = 2;
                    break;
                  case 2 :
                    output_instruction_ptr->imm_oprnd_size = 4;
                    break;
                }
            }
          }
	break;
    }

    return;
}  /* process_instruction */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: get_bwd_size_modifier                                    */
/*#                                                                          */
/*#  PARAMETERS:    cPP             Pointer to pointer to string starting    */
/*#                                   with the size modifier                 */
/*#                                                                          */
/*#                 size_bits_ptr   Pointer to variable containing the size  */
/*#                                   bits                                   */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       TRUE iff a correct size modifier is found.               */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This function gets a BWD size modifier from the string   */
/*#                 pointed out by *cPP. *cPP is advanced to the character   */
/*#                 following the size modifier.                             */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920402    Mikael Asker       Initial version                            */
/*#  920803    Miakel Asker       Made the procedure case-insensitive        */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

static bool get_bwd_size_modifier(cPP,size_bits_ptr)
  char  **cPP;
  byte  *size_bits_ptr;
{
  if (*(*cPP) != '.') {
    return FALSE;
  }
  else {
    (*cPP)++;   /* Consume '.' */
    switch (*(*cPP)) {

      case 'B' :
      case 'b' :
        *size_bits_ptr = 0;   break;

      case 'W' :
      case 'w' :
        *size_bits_ptr = 1;   break;

      case 'D' :
      case 'd' :
        *size_bits_ptr = 2;   break;

      default  :  return FALSE;

    }
    (*cPP)++;  /* Consume size letter */
    return TRUE;
  }
}  /* get_bwd_size_modifier */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: get_bw_size_modifier                                     */
/*#                                                                          */
/*#  PARAMETERS:    cPP             Pointer to pointer to string starting    */
/*#                                   with the size modifier                 */
/*#                                                                          */
/*#                 size_bits_ptr   Pointer to variable containing the size  */
/*#                                   bits                                   */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       TRUE iff a correct size modifier is found.               */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This function gets a BW size modifier from the string    */
/*#                 pointed out by *cPP. *cPP is advanced to the character   */
/*#                 following the size modifier.                             */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920604    Mikael Asker       Initial version                            */
/*#  920803    Mikael Asker       Made the procedure case-insensitive        */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

static bool get_bw_size_modifier(cPP,size_bits_ptr)
  char  **cPP;
  byte  *size_bits_ptr;
{
  if (*(*cPP) != '.') {
    return FALSE;
  }
  else {
    (*cPP)++;   /* Consume '.' */
    switch (*(*cPP)) {

      case 'B' :
      case 'b' :
        *size_bits_ptr = 0;   break;

      case 'W' :
      case 'w' :
        *size_bits_ptr = 1;   break;

      default  :  return FALSE;

    }
    (*cPP)++;  /* Consume size letter */
    return TRUE;
  }
}  /* get_bw_size_modifier */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: get_gen_reg                                              */
/*#                                                                          */
/*#  PARAMETERS:    cPP              Pointer to pointer to string begining   */
/*#                                    with a general register designator    */
/*#                                                                          */
/*#                 reg_number_ptr   Pointer to byte containing the register */
/*#                                    number                                */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       TRUE iff a correct general register designator is found. */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This function gets a general register from the string    */
/*#                 pointed out by *cPP. *cPP is advanced to the character   */
/*#                 following the general register designator.               */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920330    Mikael Asker       Initial version                            */
/*#  920612    Mikael Asker       Replaced = in conditions by ==             */
/*#  920622    Mikael Asker       Consume 'C' in "PC" and 'P' in "SP"        */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

static bool get_gen_reg(cPP,reg_number_ptr)
  char  **cPP;
  byte  *reg_number_ptr;
{
  char *saved_str_ptr;

  saved_str_ptr = *cPP;
  switch (*(*cPP)) {
    case 'P' :
    case 'p' :
      (*cPP)++;   /* Consume 'P' or 'p' */
      if (! ((*(*cPP) == 'C') || (*(*cPP) == 'c')) ) {
        *cPP = saved_str_ptr;
        return FALSE;
      }
      else {   /* PC = Rreg_pc */
        (*cPP)++;   /* Consume 'C' or 'c' */
        *reg_number_ptr = REG_PC;
        return TRUE;
      }
    case 'R' :
    case 'r' :
      (*cPP)++;   /* Consume 'R' or 'r' */
      if (! isdigit(*(*cPP))) {
        *cPP = saved_str_ptr;
        return FALSE;
      }
      else {   /* Digit */
        *reg_number_ptr = *(*cPP) - '0';
        (*cPP)++;   /* Consume digit */
        if (! isdigit(*(*cPP))) {   /* No more digits */
          return TRUE;
        }
        else {   /* One more digit */
          *reg_number_ptr = 10 * *reg_number_ptr + (*(*cPP) - '0');
          if (*reg_number_ptr > MAX_REG) {
            *cPP = saved_str_ptr;
            return FALSE;
          }
          else {   /* Rn, 0 <= n <= MAX_REG */
            (*cPP)++;   /* Consume second digit */
            return TRUE;
          }
        }
      }
    case 'S' :
    case 's' :
      (*cPP)++;   /* Consume 'S' or 's' */
      if (! ((*(*cPP) == 'P') || (*(*cPP) == 'p'))) {
        *cPP = saved_str_ptr;
        return FALSE;
      }
      else {   /* SP = Rreg_sp */
        (*cPP)++;   /* Consume 'P' or 'p' */
        *reg_number_ptr = REG_SP;
        return TRUE;
      }
    default :
      *cPP = saved_str_ptr;
      return FALSE;
  }
}  /* get_gen_reg */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: get_spec_reg                                             */
/*#                                                                          */
/*#  PARAMETERS:    cPP              Pointer to pointer to string starting   */
/*#                                    with a special register designator    */
/*#                                                                          */
/*#                 reg_number_ptr   Pointer to byte containing the register */
/*#                                    number                                */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       TRUE iff a correct special register designator is found. */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This function gets a special register from the string    */
/*#                 pointed out by *cPP. *cPP is advanced to the character   */
/*#                 following the special register designator.               */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920330    Mikael Asker       Initial version                            */
/*#  920612    Mikael Asker       Replaced = in conditions by ==             */
/*#  920622    Mikael Asker       End-of-string test only on char_ptr_2.     */
/*#                                  Consume register name on success.       */
/*#  920803    Mikael Asker       Added translation to lower case            */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

static bool get_spec_reg(cPP,reg_number_ptr)
  char  **cPP;
  byte  *reg_number_ptr;
{
  byte  index;
  char  *char_ptr_1, *char_ptr_2;

  index = 0;
  do {
    char_ptr_1 = *cPP;
    char_ptr_2 = spec_regs[index].name;
    while ( (*char_ptr_2 != '\0') && (tolower(*char_ptr_1) == *char_ptr_2) )
    {
      char_ptr_1++;
      char_ptr_2++;
    }
    index++;
  } while ( (index < NUMSPECREGS) && (*char_ptr_2 != '\0') );
  if (*char_ptr_2 != '\0') {
    return FALSE;
  }
  else {
    *cPP = char_ptr_1;   /* Consume register name */
    *reg_number_ptr = spec_regs[index-1].number;
    return TRUE;
  }
}  /* get_spec_reg */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: get_no_or_eff_addr_prefix_oprnd                          */
/*#                                                                          */
/*#  PARAMETERS:    cPP                  Pointer to pointer to string        */
/*#                                        beginning with the operand        */
/*#                                                                          */
/*#                 prefix_ptr           Pointer to structure containing an  */
/*#                                        optional instruction prefix       */
/*#                                                                          */
/*#                 mode_ptr             Pointer to byte containing the      */
/*#                                        mode bits                         */
/*#                                                                          */
/*#                 src_reg_number_ptr   Pointer to byte containing the      */
/*#                                        source register number in the     */
/*#                                        instruction                       */
/*#                                                                          */
/*#                 imm_expr_found_ptr   Pointer to a boolean indicating if  */
/*#                                        an immediate expression is found  */
/*#                                                                          */
/*#                 imm_expr_ptr         Pointer to a structure containing   */
/*#                                        an optional immediate expression  */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       TRUE iff a correct indirect operand is found.            */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This function gets an unprefixed or effective-address    */
/*#                 indirect operand from the string pointed out by *cPP.    */
/*#                 *cPP is advanced to the character following the indirect */
/*#                 operand.                                                 */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920406    Mikael Asker       Initial version                            */
/*#  920407    Mikael Asker       Consume '+' in "Indexed with assign" and   */
/*#                                 "Offset with assign" modes and generate  */
/*#                                 opcodes correctly in these modes         */
/*#  920410    Mikael Asker       Added expr_ptr and error_ptr parameters in */
/*#                                 get_expression calls                     */
/*#  920415    Mikael Aaker       When offset with assign mode, return mode  */
/*#                                 name in prefix_ptr->error. Handle offset */
/*#                                 expressions beginning with '-'.          */
/*#  920428    Mikael Asker       Corrected error in call to get_expression  */
/*#  920520    Mikael Asker       Use gen_bdap                               */
/*#  920522    Mikael Asker       Redeclared prefix_ptr to *cris_prefix_type,*/
/*#                                 removed prefixed_inst_ptr parameter      */
/*#  920527    Mikael Asker       Added mode_ptr and inst_expr_ptr parame-   */
/*#                                 ters and MODE_* constants. Consume '-'   */
/*#                                 in offset with assign mode.              */
/*#  920603    Mikael Asker       Renamed inst_expr_ptr to imm_expr_ptr,     */
/*#                                 added imm_expr_found_ptr parameter       */
/*#  920624    Mikael Asker       Handle SEG_DIFFERENCE offset expressions   */
/*#                                  correctly - don't consume the '-'       */
/*#                                  character before the get_expression     */
/*#                                  call and don't negate                   */
/*#                                  prefix_ptr->expr->X_add_number.         */
/*#  930409    Hans-Peter Nilsson Added [rx=ry+[rz].bwd] & [rx=ry+[rz+].bwd] */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

#define MODE_INDIR    (0)
#define MODE_AUTOINC  (1)

static bool get_no_or_eff_addr_prefix_oprnd(
  cPP,prefix_ptr,mode_ptr,src_reg_number_ptr,imm_expr_found_ptr,imm_expr_ptr
)
  char              **cPP;
  cris_prefix_type  *prefix_ptr;
  byte              *mode_ptr, *src_reg_number_ptr;
  bool              *imm_expr_found_ptr;
  expressionS       *imm_expr_ptr;
{
  *imm_expr_found_ptr = FALSE;   /* Assume no immediate mode expression */
  if (*(*cPP) == '[') {  /* Found '[' */
    /* Indirect, autoincrement, indexed with assign or offset with assign mode */
    (*cPP)++;  /* Consume '[' */
    if (! get_gen_reg(cPP,src_reg_number_ptr)) {  /* Register missing */
      return FALSE;
    }
    else {  /* Register found */
      switch (*(*cPP)) {
        case ']' :  /* ']' found */
          /* Indirect mode */
          prefix_ptr->kind = PREFIX_NONE;
          *mode_ptr = MODE_INDIR;
          break;
        case '+' :  /* '+' found */
          /* Auto-increment mode */
          prefix_ptr->kind = PREFIX_NONE;
          *mode_ptr = MODE_AUTOINC;
          (*cPP)++;  /* Consume '+' */
          break;
        case '=' :  /* '=' found */
          /* Indexed with assign or offset with assign mode */
          (*cPP)++;  /* Consume '=' */
          if (! get_gen_reg(cPP,&(prefix_ptr->base_reg_number))) {
            /* Register missing */
            return FALSE;
          }
          else {  /* Register found */
            if (*(*cPP) == '+') {  /* '+' found */
              byte index_reg_number;
              (*cPP)++;  /* Consume '+' */
	      if (**cPP == '[') {
		byte size_bits;
		/* [rx=ry+[rz].s] or [rx=ry+[rz+].s] */
		(*cPP)++;
		if (!get_gen_reg(cPP,&index_reg_number))
		  return FALSE;
		/* Fake a biap since bdap is used for dealing with
		   hairy fixes.  We know the size of this thang. */
		prefix_ptr->kind = PREFIX_BIAP;
		prefix_ptr->opcode
		  = BDAP_INDIR_OPCODE + (prefix_ptr->base_reg_number << 12)
		    + index_reg_number;
		if (**cPP == '+') {
		  /* [rx=ry+[rz+... */
		  (*cPP)++;
		  prefix_ptr->opcode |= AUTOINCR_BIT << 8;
		  prefix_ptr->mode_name_ptr
		    = "indir autoinc (and sign-ext) offset with assign";
		} else prefix_ptr->mode_name_ptr
		  = "indir (sign-ext) offset with assign";
		if (**cPP != ']')
		  return FALSE;
		(*cPP)++;
		if (!get_bwd_size_modifier(cPP,&size_bits))
		  return FALSE;
		prefix_ptr->opcode |= size_bits << 4;
		/* Now we got [rx=ry+[rz+].s or [rx=ry+[rz].s;  Expect ']'. */
		break;
	      }
	      else
              if (get_gen_reg(cPP,&index_reg_number)) {  /* Register found */
                byte size_bits;
                /* Indexed with assign mode */
                prefix_ptr->kind = PREFIX_BIAP;
                prefix_ptr->opcode = BIAP_OPCODE + ( index_reg_number << 12 ) +
                  prefix_ptr->base_reg_number /* << 0 */ ;
                if (! get_bwd_size_modifier(cPP,&size_bits)) {
                  /* Size missing */
                  return FALSE;
		}
                else {  /* Size found */
                  prefix_ptr->opcode |= size_bits << 4;
                  break;
                }  /* Size found */
              }  /* Register found */
              else if (get_expression(cPP,&(prefix_ptr->expr),
              &(prefix_ptr->error_msg_ptr))) {
                /* Expression found */
                /* Offset with assign mode */
                prefix_ptr->kind = PREFIX_BDAP;
                prefix_ptr->mode_name_ptr = "offset with assign";
                break;
              }
              else {  /* Neither register nor expression found */
                return FALSE;
              }
            }  /* '+' found */
            else if (*(*cPP) == '-') {  /* '-' found */
              /* Offset with assign mode */
              if (! get_expression(cPP,&(prefix_ptr->expr),
              &(prefix_ptr->error_msg_ptr))) {
                /* No expression */
                return FALSE;
              }
              else {  /* Expression found */
                /* Offset with assign mode */
                prefix_ptr->kind = PREFIX_BDAP;
                prefix_ptr->mode_name_ptr = "offset with assign";
                break;
              }
            }  /* '-' found */
            else {  /* Neither '+' nor '-' */
              return FALSE;
            }
          }  /* Register found */
        default :  /* Neither ']' nor '+' nor '=' */
          return FALSE;
      }
    }  /* Register found */
    if (*(*cPP) != ']') {  /* Missing ']' */
      return FALSE;
    }
    else  {  /* ']' found */
      (*cPP)++;  /* Consume ']' */
      return TRUE;
    }
  }  /* Found '[' */
  else if (get_expression(cPP,imm_expr_ptr,&(prefix_ptr->error_msg_ptr))) {
    /* Expression found */
    /* Immediate mode */
    prefix_ptr->kind = PREFIX_NONE;
    *mode_ptr = MODE_AUTOINC;
    *src_reg_number_ptr = REG_PC;
    *imm_expr_found_ptr = TRUE;
    return TRUE;
  }  /* Expression found */
  else {  /* Neither '[' nor expression found */
    return FALSE;
  }
}  /* get_no_or_eff_addr_prefix_oprnd */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: get_3_oprnd_prefix_oprnd                                 */
/*#                                                                          */
/*#  PARAMETERS:    cPP          Pointer to pointer to string begining       */
/*#                                with the operand                          */
/*#                                                                          */
/*#                 prefix_ptr   Pointer to structure containing an          */
/*#                                instruction prefix                        */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       TRUE iff a correct indirect operand is found.            */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This function gets an indirect operand in a three-       */
/*#                 address instruction from the string pointed out by *cPP. */
/*#                 *cPP is advanced to the character following the indirect */
/*#                 operand.                                                 */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920401    Mikael Asker       Initial version                            */
/*#  920407    Mikael Asker       Moved the declarations for reg_number,     */
/*#                                 base_reg_number, index_reg_number and    */
/*#                                 size_bits to internal blocks to increase */
/*#                                 readability.                             */
/*#  920410    Mikael Asker       Added expr_ptr and error_ptr parameters in */
/*#                                 get_expression calls.                    */
/*#  920415    Mikael Asker       When offset mode, return mode name in      */
/*#                                 prefix_ptr->error. Handle offset expres- */
/*#                                 sions beginning with '-'.                */
/*#  920520    Mikael Asker       Use gen_bdap                               */
/*#  920522    Mikael Asker       Redeclared prefix_ptr to *cris_prefix_type */
/*#  920603    Mikael Asker       RELOC_32 in absolute mode                  */
/*#  920624    Mikael Asker       Handle SEG_DIFFERENCE offset expressions   */
/*#                                  correctly - don't consume the '-'       */
/*#                                  character before the get_expression     */
/*#                                  call and don't negate                   */
/*#                                  prefix_ptr->expr->X_add_number.         */
/*#  920804    Mikael Asker       Offset mode operands without offset        */
/*#                                  expressions are now possible.           */
/*#  930409    Hans-Peter Nilsson Added [rx+[ry+].bwd] & [rx+[ry].bwd]       */
/*#                               and solved [[rx+]] bug.                    */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

static bool get_3_oprnd_prefix_oprnd(cPP,prefix_ptr)
  char              **cPP;
  cris_prefix_type  *prefix_ptr;
{
  if (*(*cPP) != '[') {  /* Missing '[' */
    return FALSE;
  }
  else {  /* '[' found */
    byte reg_number;

    (*cPP)++;  /* Consume '[' */
    if (*(*cPP) == '[') {  /* A second '[' */
      /* A double-indirect mode */
      (*cPP)++;  /* Consume second '[' */
      prefix_ptr->kind = PREFIX_DIP;
      prefix_ptr->opcode = DIP_OPCODE;
      if (! get_gen_reg(cPP,&reg_number)) {
        return FALSE;
      }
      else {  /* Register found */
        prefix_ptr->opcode |= reg_number /* << 0 */ ;
        if (*(*cPP) == '+') {  /* '+' found */
          /* Double-indirect autoincrement mode */
	  /* Changed (*(*cPP))++ to (*cPP)++ /HP 930409. */
          (*cPP)++;  /* Consume '+' */
          prefix_ptr->opcode |= AUTOINCR_BIT << 8;  /* bit_10 = TRUE */
        }
        else {  /* Double-indirect mode */
          /* bit_10 = FALSE */
        }
        if (*(*cPP) != ']') {  /* First ']' missing */
          return FALSE;
        }
        else {  /* First ']' found */
          (*cPP)++;  /* Consume first ']' */
        }
      }  /* Register found */
    }  /* A second '[' */
    else if (get_gen_reg(cPP,&(prefix_ptr->base_reg_number))) {
      /* Register found */
      /* Indexed or offset mode */
      if (*(*cPP) == '+') {  /* '+' found */
        byte index_reg_number;

        (*cPP)++;  /* Consume '+' */

	if (**cPP == '[') {
	  /* [rx+[.. */
	  byte size_bits;
	  (*cPP)++;

	  if (!get_gen_reg(cPP,&index_reg_number))
	    return FALSE;

	  /* Fake a non-bdap type so we dont get into fix-trouble.
	     This bdap has fixed size. */
	  prefix_ptr->kind = PREFIX_BIAP;
	  prefix_ptr->opcode
	    = BDAP_INDIR_OPCODE + (prefix_ptr->base_reg_number << 12)
	      + index_reg_number;

	  /* [rx+[ry... */
	  if (**cPP == '+') {
	    /* [rx+[ry+... */
	    (*cPP)++;
	    prefix_ptr->opcode |= AUTOINCR_BIT << 8;
	    prefix_ptr->mode_name_ptr = "indir autoincr sign-extended offset";
	  } else
	    prefix_ptr->mode_name_ptr = "indir sign-extended offset";

	  if (**cPP != ']')
	    return FALSE;
	  (*cPP)++;
	  
	  /* [rx+[ry+].. or [rx+[ry].. Expect .bwd */
	  if (!get_bwd_size_modifier(cPP,&size_bits))
	    return FALSE;
	  prefix_ptr->opcode |= size_bits << 4;

	  /* [rx+[ry+].bwd or [rx+[ry].bwd;  Expect ']' */
	} else
        if (get_gen_reg(cPP,&index_reg_number)) {  /* Register found */
          /* Indexed mode */
          byte size_bits;
          prefix_ptr->kind = PREFIX_BIAP;
          prefix_ptr->opcode = BIAP_OPCODE |
            (prefix_ptr->base_reg_number /* << 0 */) |
            (index_reg_number << 12);
          if (! get_bwd_size_modifier(cPP,&size_bits)) {  /* Missing size */
            return FALSE;
          }
          else {  /* Size found */
            prefix_ptr->opcode |= ( size_bits << 4 );
          }
        }  /* Register found */
        else if (get_expression(cPP,&(prefix_ptr->expr),
        &(prefix_ptr->error_msg_ptr))) {
          /* Expression found */
          /* Offset mode */
          prefix_ptr->kind = PREFIX_BDAP;
          prefix_ptr->mode_name_ptr = "offset";
        }
        else { /*  Neither register nor expression */
          return FALSE;
        }
      }  /* '+' found */
      else if (*(*cPP) == '-') {  /* '-' found */
        /* Offset mode */
        if (! get_expression(cPP,&(prefix_ptr->expr),
        &(prefix_ptr->error_msg_ptr))) {
          /* No expression */
          return FALSE;
        }
        else {  /* Expression found */
          /* Offset mode */
          prefix_ptr->kind = PREFIX_BDAP;
          prefix_ptr->mode_name_ptr = "offset";
        }
      }  /* '-' found */
      else if (*(*cPP) == ']') {  /* ']' found */
        /* Offset mode without expression */
        (*cPP)++;  /* Consume ']' */
        prefix_ptr->expr.X_op = O_constant;
        prefix_ptr->expr.X_add_number = 0;
        prefix_ptr->expr.X_add_symbol = NULL;
        prefix_ptr->expr.X_op_symbol = NULL;
        prefix_ptr->kind = PREFIX_BDAP;
        prefix_ptr->mode_name_ptr = "offset";
        return TRUE;
      }
      else {  /* Neither '+' nor '-' nor ']' */
        return FALSE;
      }
    }  /* Register found */
    else if (get_expression(cPP,&(prefix_ptr->expr),
    &(prefix_ptr->error_msg_ptr))) {
      /* Expression found */
      /* Absolute mode */
      prefix_ptr->kind = PREFIX_DIP;
      prefix_ptr->opcode = DIP_OPCODE | (AUTOINCR_BIT * 0x0100) | REG_PC;
      prefix_ptr->reloc = CRIS_RELOC_32;
    }
    else {  /* Neither '[' nor register nor expression */
      return FALSE;
    };
    if (*(*cPP) != ']') {  /* ']' missing */
      return FALSE;
    }
    else {  /* ']' found */
      (*cPP)++;  /* Consume ']' */
      return TRUE;
    }
  }  /* '[' found */
}  /* get_3_oprnd_prefix_oprnd */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: get_expression                                           */
/*#                                                                          */
/*#  PARAMETERS:    cPP         Pointer to pointer to string beginning with  */
/*#                                the expression                            */
/*#                                                                          */
/*#                 expr_ptr    Pointer to structure containing the          */
/*#                                expression                                */
/*#                                                                          */
/*#                 error_ptr   Pointer to charracter array for storing      */
/*#                                error messages                            */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       TRUE iff a correct expression is found.                  */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This function gets an expression from the string pointed */
/*#                 out by *cPP. *cPP is advanced to the character following */
/*#                 the expression.                                          */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920330    Mikael Asker       Initial version                            */
/*#  920331    Mikael Asker       Renamed getExpression to get_expression,   */
/*#                                 changed return type to bool and the_insn */
/*#                                 to output_instruction. Renamed save_in   */
/*#                                 to saved_input_line_ptr.                 */
/*#  920410    Mikael Asker       Added expr_ptr and error_ptr parameters    */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

static bool get_expression(cPP,expr_ptr,error_ptr)
    char         **cPP;
    expressionS  *expr_ptr;
    char         **error_ptr;
{
    char *saved_input_line_ptr;
    segT exp;

    saved_input_line_ptr = input_line_pointer;
    input_line_pointer = *cPP;
    exp = expression(expr_ptr);
    if (exp !=  absolute_section
        && exp != text_section
        && exp != data_section
        && exp != bss_section
        && exp != undefined_section
        && exp != expr_section)
    {
      *error_ptr = "bad segment";
      input_line_pointer = saved_input_line_ptr;
      return FALSE;
    }
    *cPP = input_line_pointer;
    input_line_pointer = saved_input_line_ptr;
    return TRUE;
}  /* get_expression */

/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: get_flags                                                */
/*#                                                                          */
/*#  PARAMETERS:    spp         Pointer to pointer to string beginning with  */
/*#                                the expression                            */
/*#                                                                          */
/*#                 flag_ptr    Pointer to byte to return the flags          */
/*#                                expression                                */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       TRUE iff a correct flags expression is found.            */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This function gets a sequence of flag characters from    */
/*#                 *spp. *spp is advanced to the character following        */
/*#                 the expression. The flag characters are consecutive.     */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  930409    Hans-Peter Nilsson Initial version                            */
/*#--------------------------------------------------------------------------*/

static bool
get_flags(spp,flagsp)
    char         **spp;
    byte        *flagsp;
{
  for (;;) {
    switch (**spp) {
    case 'd': case 'D': case 'm': case 'M':
      *flagsp |= 0x80;
      break;
    case 'e': case 'E': case 'b': case 'B':
      *flagsp |= 0x40;
      break;
    case 'i': case 'I':
      *flagsp |= 0x20;
      break;
    case 'x': case 'X':
      *flagsp |= 0x10;
      break;
    case 'n': case 'N':
      *flagsp |= 0x8;
      break;
    case 'z': case 'Z':
      *flagsp |= 0x4;
      break;
    case 'v': case 'V':
      *flagsp |= 0x2;
      break;
    case 'c': case 'C':
      *flagsp |= 1;
      break;
    default:
      if (**spp != ','
	  && **spp != ' '
	  && **spp != 0
	  && **spp != '\t'
	  && **spp != '\r'
	  && **spp != '\n')
	return FALSE;
      else
	return TRUE;
    }
    (*spp)++;
  }
}

/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: spec_reg_size                                            */
/*#                                                                          */
/*#  PARAMETERS:    spec_reg_number  Number of the special register of which */
/*#                                  the size is to be returned              */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       The size of the special register in bytes                */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This function returns the size of a special register.    */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920603    Mikael Asker       Initial version                            */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

static byte spec_reg_size(spec_reg_number)
  byte spec_reg_number;
{
  if ( /*(0 <= spec_reg_number) &&*/ (spec_reg_number <= 3)) {
    return 1;
  }
  else if ((4 <= spec_reg_number) && (spec_reg_number <= 7)) {
    return 2;
  }
  else {   /* spec_reg_number >= 8 */
    return 4;
  }
}


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: gen_bdap                                                 */
/*#                                                                          */
/*#  PARAMETERS:    base_reg_number   Byte containing the base register      */
/*#                                     number                               */
/*#                                                                          */
/*#                 expr_ptr          Pointer to structure containing the    */
/*#                                     offset expression                    */
/*#                                                                          */
/*#                 mode_name_ptr     Pointer to string containing the name  */
/*#                                     of the addressing mode which gene-   */
/*#                                     rates the prefix; used in error      */
/*#                                     messages                             */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       Nothing.                                                 */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This function generates code and fixes for a BDAP        */
/*#                   prefix.                                                */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920520    Mikael Asker       Initial version                            */
/*#  920623    Mikael Asker       Added forward reference handling           */
/*#  920624    Mikael Asker       Included offset value in frag_var call     */
/*#                                  Corrected absolute segment test         */
/*#  920811    Mikael Asker       Changed BDAP offset overflow handling      */
/*#  950209    Hans-Peter Nilsson Bugfix for [rX+expr]; the expr will        */
/*#                               be max 4 bytes, not 2.                     */ 
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

static NO_RET gen_bdap(base_reg_number,expr_ptr,mode_name_ptr)
  byte         base_reg_number;
  expressionS  *expr_ptr;
  char         *mode_name_ptr;
{
  uword  opcode;
  char   *opcode_ptr;

  /* Put out the prefix opcode; assume quick immediate mode */
    opcode = BDAP_QUICK_OPCODE | (base_reg_number << 12);  
    opcode_ptr = frag_more(2);
    md_number_to_chars(opcode_ptr, (long)opcode, 2);

  /* BDAP offsets aren't PC-relative.    */

  if (expr_ptr->X_op == O_constant) {  /* Absolute expression */
    dword value;
    int size;

    value = expr_ptr->X_add_number;
    if ((value < -32768) || (value > 32767)) {   /* Outside range */
      size = 2;
    } else
      size = 1;

    if ((value >= -128) && (value <= 127)) {  /* Byte displacement */
      opcode_ptr[0] = value;
    }
    else {   /* (value >= -32768) && (value <= 32767) */
      /* Word displacement */
      char  *p;

      opcode_ptr[0] = BDAP_PC_LOW + size * 16;
      opcode_ptr[1] &= 0xF0;
      opcode_ptr[1] |= BDAP_INCR_HIGH;
      p = frag_more( 1 << size);
      md_number_to_chars(p,value, 1 << size);
    }
  }  /* Absolute expression */
  else {
    /* The expression is not defined yet but may become absolute */
    frag_var(rs_machine_dependent,
             4, /* Bug fixed; was 2. It will ALWAYS be MAX 4 bytes (?
                   for what I know...) and I do not understand why it
                   took so long for this bug to creep up (must've
                   overwritten LOTS of stuff!) */
             0,
	     ENCODE_RELAX(STATE_BASE_PLUS_DISP_PREFIX,STATE_UNDF),
	     expr_ptr->X_add_symbol, expr_ptr->X_add_number, opcode_ptr);
  }

}  /* gen_bdap */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: branch_disp                                              */
/*#                                                                          */
/*#  PARAMETERS:    offset   The displacement value in bytes                 */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       A byte containing the encoded displacement.              */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This function encodes a branch displacement in the range */
/*#                 -256..254 in the form used by CGA-RISC conditional       */
/*#                 branch instructions.                                     */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920428    Mikael Asker       Initial version                            */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

static byte branch_disp(offset)
  word offset;
{
  byte disp;

  disp = offset & 0xFE;
  if (offset < 0) {
    disp |= 1;
  }
  return disp;
}


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: gen_cond_branch_32                                       */
/*#                                                                          */
/*#  PARAMETERS:    opcode_ptr        Pointer to the word containing the     */
/*#                                     original 8-bit branch instruction    */
/*#                                                                          */
/*#                 write_ptr         Pointer to "extension area" following  */
/*#                                     the first instruction word           */
/*#                                                                          */
/*#                 frag_ptr          Pointer to the frag containing the     */
/*#                                     instruction                          */
/*#                                                                          */
/*#                 add_symbol        Parts of the destination address       */
/*#                 subtract_symbol     expression                           */
/*#                 add_number                                               */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       Nothing.                                                 */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This procedures generates code and fixes for a 32-bit    */
/*#                 conditional branch instruction created by "extending"    */
/*#                 an existing 8-bit branch instruction.                    */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920702    Mikael Asker       Initial version                            */
/*#  920811    Mikael Asker       Added warning message                      */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

static NO_RET gen_cond_branch_32(opcode_ptr,write_ptr,frag_ptr,add_symbol,
  subtract_symbol,add_number)

  char      *opcode_ptr, *write_ptr;
  fragS     *frag_ptr;
  symbolS   *add_symbol, *subtract_symbol;
  long int  add_number;

{
  if (O_option)
  {
    as_warn("32-bit conditional branch generated");
  }
  /* opcode_ptr+10:  Bcc     . - 6      */
    write_ptr[8] = branch_disp(-2-6);
    write_ptr[9] = opcode_ptr[1];
  /* opcode_ptr:     BA      . + 10     */
    md_number_to_chars(opcode_ptr,(long)(BA_QUICK_OPCODE+8),2);
  /* opcode_ptr+2:   NOP                */
    md_number_to_chars(write_ptr,(long)NOP_OPCODE,2);
  /* opcode_ptr+4:   JUMP    [PC+]      */
    md_number_to_chars(write_ptr+2,(long)JUMP_PC_INCR_OPCODE,2);
  /* opcode_ptr+6:   .DWORD  expression */
    if ((add_symbol == NULL) && (subtract_symbol == NULL)) {
      /* Absolute address */
      md_number_to_chars(write_ptr+4, add_number, 4);
    }
    else {   /* Not absolute */
      know(subtract_symbol == 0); /* Or else... */
      fix_new(
        frag_ptr,
        write_ptr + 4 - frag_ptr->fr_literal,
        4,
        add_symbol,
        add_number,
        NOT_PC_RELATIVE,
        CRIS_RELOC_32
      );
    }
}   /* gen_cond_branch_32 */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: md_atof                                                  */
/*#                                                                          */
/*#  PARAMETERS:    type    A character from FLTCHARS that describes what    */
/*#                           kind of floating-point number is wanted        */
/*#                                                                          */
/*#                 litP    A pointer to an array that the result should be  */
/*#                           stored in.                                     */
/*#                                                                          */
/*#                 sizeP   A pointer to an integer where the size of the    */
/*#                           result is stored.                              */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       An error message or an empty string for success.         */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This is identical to the md_atof in m68k.c.              */
/*#                                                                          */
/*#                 Turn a string in input_line_pointer into a floating      */
/*#                 point constant of type type, and store the appropriate   */
/*#                 bytes in *litP.  The number of LITTLENUMS emitted is     */
/*#                 stored in *sizeP.                                        */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920312    Mikael Asker       Initial version                            */
/*#  920605    Mikael Asker       Changed                     */
/*#  930409    Hans-Peter Nilsson Modified to output littlenums to correct   */
/*#                               previously bad endianness.                 */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

/* Equal to MAX_PRECISION in atof-ieee.c */
#define MAX_LITTLENUMS 6

char *md_atof(type,litP,sizeP)
  char  type;
  char  *litP;
  int   *sizeP;
{
        byte            prec;
	LITTLENUM_TYPE  words[MAX_LITTLENUMS];
	LITTLENUM_TYPE  *wordP;
	char            *t;

	switch(type) {
	case 'f':
	case 'F':
	case 's':
	case 'S':
		prec = 2;
		break;

	case 'd':
	case 'D':
	case 'r':
	case 'R':
		prec = 4;
		break;

	case 'x':
	case 'X':
		prec = 6;
		break;

	case 'p':
	case 'P':
		prec = 6;
		break;

	default:
		*sizeP=0;
		return "Bad call to MD_ATOF()";
	}
	t=atof_ieee(input_line_pointer,type,words);
	if(t)
		input_line_pointer=t;

	*sizeP=prec * sizeof(LITTLENUM_TYPE);
	for(wordP=words+prec-1; prec--; wordP--) {
		md_number_to_chars(litP,(long)(*wordP),sizeof(LITTLENUM_TYPE));
		litP+=sizeof(LITTLENUM_TYPE);
	}
	return 0;
}  /* md_atof */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: md_number_to_chars                                       */
/*#                                                                          */
/*#  PARAMETERS:    buf   Pointer to an array where the result should be     */
/*#                         stored.                                          */
/*#                                                                          */
/*#                 val   The value to store                                 */
/*#                                                                          */
/*#                 n     The number of bytes in "value" that should be      */
/*#                         stored                                           */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       Nothing.                                                 */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This procedure is called to turn a C long int, short int */
/*#                 or char into a series of bytes that represents the       */
/*#                 number on the target machine.                            */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920312    Mikael Asker       Initial version                            */
/*#  920406    Mikael Asker       Changed for correct byte-order             */
/*#  920504    Mikael Asker       Better coding of the switch statement      */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

void md_number_to_chars(buf,val,n)
    char  *buf;
    valueT  val;
    int   n;
{

    switch(n) {

      case 4:
        buf[3] = ( val >> 24 ) & 0xFF;
        buf[2] = ( val >> 16 ) & 0xFF;

      case 2:
        buf[1] = ( val >> 8 ) & 0xFF;

      case 1:
        buf[0] = val & 0xFF;
	break;

      default:
	/* Dont abort in my nose. Give appropriate error msg. */
	as_fatal("INTERNAL ERROR : Bad n=%d md_to_chars",n);

    }
    return;

}  /* md_number_to_chars */


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: md_number_to_imm                                         */
/*#                                                                          */
/*#  PARAMETERS:    buf        Pointer to an array where the result should   */
/*#                              be stored.                                  */
/*#                                                                          */
/*#                 val        The value to store                            */
/*#                                                                          */
/*#                 n          The number of bytes in "val" that should be   */
/*#                              stored                                      */
/*#                                                                          */
/*#                 fixP       The fix to be applied to the bit field        */
/*#                              starting at *buf                            */
/*#                                                                          */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       Nothing.                                                 */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This procedure is called to turn a C long int, short int */
/*#                 or char into a series of bytes that represents the       */
/*#                 number on the target machine. The purpose of this        */
/*#                 procedure is the same as that of md_number_to_chars but  */
/*#                 this procedure can handle general bit field fixes.       */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920312    Mikael Asker       Initial version                            */
/*#  920331    Mikael Asker       Changed to process CGA-RISC relocations.   */
/*#  920504    Mikael Asker       Added RELOC_8, RELOC_16, RELOC_32,         */
/*#                                  RELOC_DISP8 and RELOC_BDISP8            */
/*#                                  relocations.                            */
/*#  930830    HP                 Added warnings when                        */
/*#                               not in range for the operand size.         */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

static void
md_number_to_imm(buf,val,n, fixP)
    char  *buf;
    long  val;
    int   n;
    fixS  *fixP;
{
    segT sym_seg;

    assert(n <= 4);
    assert(fixP);

    /* For inter-segment relocations, it seems we should put the
       relative "vma" for the other segment in the object data (to
       stay exact binary "compatible") for the relocation. . */
    if (fixP->fx_addsy
        && (sym_seg = S_GET_SEGMENT(fixP->fx_addsy)) != now_seg)
    {
      val += sym_seg->vma;
    }

    switch (fixP->fx_r_type) {

      /* Strange enough, we put the addend into the object code as
         well as the reloc addend.  Who cares...  except keep it that
         way, to simplify regression tests on the object file
         contents. */

      case CRIS_RELOC_32:
      /* No use having warnings here, since most unix has long==int==32bit */
        buf[3] = ( val >> 24 ) & 0xFF;
        buf[2] = ( val >> 16 ) & 0xFF;
        buf[1] = ( val >> 8 ) & 0xFF;
        buf[0] = val & 0xFF;
	break;

      case CRIS_RELOC_16:
	if (val > 0xffff || val < -32768)
	  as_bad("Value not in 16 bit range: %ld; truncated",val);
        buf[1] = ( val >> 8 ) & 0xFF;
        buf[0] = val & 0xFF;
	break;

      case CRIS_RELOC_8:
	if (val > 0xff || val < -128)
	  as_bad("Value not in 8 bit range: %ld; truncated",val);
        if (! fixP->fx_addsy)
        {
          buf[0] = val & 0xFF;
        }
	break;

      case CRIS_RELOC_UNSIGNED_4 :
	if (val > 15 || val < 0)
	  as_bad("Value not in 4 bit unsigned range: %ld; truncated",val);
        if (! fixP->fx_addsy) {
          buf[0] |= val & 0x0F;
        }
        break;

      case CRIS_RELOC_UNSIGNED_5 :
	if (val > 31 || val < 0)
	  as_bad("Value not in 5 bit unsigned range: %ld; truncated",val);
        if (! fixP->fx_addsy) {
          buf[0] |= val & 0x1F;
        }
        break;

      case CRIS_RELOC_SIGNED_6 :
	if (val > 31 || val < -32)
	  as_bad("Value not in 6 bit range: %ld; truncated",val);
        if (! fixP->fx_addsy) {
          buf[0] |= val & 0x3F;
        }
        break;

      case CRIS_RELOC_UNSIGNED_6 :
	if (val > 63 || val < 0)
	  as_bad("Value not in 6 bit unsigned range: %ld; truncated",val);
        if (! fixP->fx_addsy) {
          buf[0] |= val & 0x3F;
        }
        break;

      case CRIS_RELOC_DISP8:
	if (val > 255 || val < -128)
	  as_bad("Value not in 8 bit range: %ld; truncated",val);
        if (! fixP->fx_addsy)
        {
          buf[0] = val;
        }
        break;

      case CRIS_RELOC_BDISP8:
        if (! fixP->fx_addsy)
        {
          buf[0] = branch_disp((word)val);
        }
        break;

      case CRIS_RELOC_NONE:
        /* May actually happen automatically.  For example at broken
           words... (if it was no broken word). */
        if (! fixP->fx_addsy)
        {
          md_number_to_chars(buf, val, n);
        }
        break;

      default:
        as_fatal("bad relocation type: 0x%02x", fixP->fx_r_type);
        break;

    }
    return;
}  /* md_number_to_imm */

/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: md_number_to_disp                                        */
/*#                                                                          */
/*#  PARAMETERS:    buf                                                      */
/*#                 val                                                      */
/*#                 n                                                        */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       Nothing.                                                 */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This procedure should never be called for cris.          */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920312    Mikael Asker       Initial version                            */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

static void
md_number_to_disp(buf,val,n)
    char	*buf;
    long	val;
{
    fprintf(stderr, "md_number_to_disp\n");
    abort();
}


/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: md_number_to_field                                       */
/*#                                                                          */
/*#  PARAMETERS:    buf                                                      */
/*#                 val                                                      */
/*#                 n                                                        */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       Nothing.                                                 */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This procedure should never be called for cris.          */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920312    Mikael Asker       Initial version                            */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

static void
md_number_to_field(buf,val,fix)
    char  *buf;
    long  val;
    void  *fix;
{
    fprintf(stderr, "md_number_to_field\n");
    abort();
}

/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  FUNCTION NAME: md_parse_option                                          */
/*#                                                                          */
/*#  PARAMETERS:    argP   Pointer to option.  If the option is multi-       */
/*#                          character, argP is advanced past the end of the */
/*#                          option.  If this wasn't done every letter in    */
/*#                          the option would be treated as a separate       */
/*#                          single-character option.                        */
/*#                                                                          */
/*#                 cntP   Provided for accessing arguments                  */
/*#                 vecP                                                     */
/*#                                                                          */
/*#  CHANGES:                                                                */
/*#                                                                          */
/*#  RETURNS:       Nothing.                                                 */
/*#                                                                          */
/*#  SIDE EFFECTS:                                                           */
/*#                                                                          */
/*#  DESCRIPTION:   This routine processes machine-dependent command line    */
/*#                 options.  It is called once for each option on the       */
/*#                 command line that the machine-independent part of GAS    */
/*#                 does not understand.                                     */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/
/*#                                                                          */
/*#  HISTORY:                                                                */
/*#                                                                          */
/*#  DATE      NAME               CHANGES                                    */
/*#  ----      ----               -------                                    */
/*#  920312    Mikael Asker       Initial version                            */
/*#  920717    Mikael Asker       Added -h option                            */
/*#  920811    Mikael Asker       Correct -O option handling                 */
/*#                                                                          */
/*#--------------------------------------------------------------------------*/

#define INVALID_OPTION  (0)
#define VALID_OPTION    (1)

int md_parse_option(arg,argP)
    int   arg;
    char  *argP;
{
  switch (arg) {

    case 'H' :
    case 'h' :
      md_show_usage(stdout);
      exit(0);  /* Don't continue */

    case 'O' :
      O_option = TRUE;
      return VALID_OPTION;

    default:
      return INVALID_OPTION;

  }
}   /* md_parse_option */

/* Round up a section size to the appropriate boundary. */
valueT
md_section_align (segment, size)
     segT segment;
     valueT size;
{
  /* Round all sects to multiple of 4, except the bss section. */
  if (segment == bss_section)
    return size;
  return (size + 3) & ~3;
}


arelent *
tc_gen_reloc (section, fixp)
     asection *section;
     fixS *fixp;
{
  arelent *rel;
  bfd_reloc_code_real_type code;

  switch(fixp->fx_r_type)
    {
    case BFD_RELOC_32:
    case BFD_RELOC_16:
    case BFD_RELOC_8:
      code = fixp->fx_r_type;
      break;
    default:
      as_bad_where(fixp->fx_file, fixp->fx_line, 
                   "Semantics error. This type of operand can not be relocated, it must be a constant");
      return 0;
    }

  rel = (arelent *) bfd_alloc_by_size_t (stdoutput, sizeof (arelent));
  assert (rel != 0);
  rel->sym_ptr_ptr = &fixp->fx_addsy->bsym;
  rel->address = fixp->fx_frag->fr_address + fixp->fx_where;
  if (fixp->fx_pcrel)
    rel->addend = fixp->fx_addnumber; /* @Q@ Is this correct? */
  else
    rel->addend = fixp->fx_offset; /* At least *this* is correct. */

  rel->howto = bfd_reloc_type_lookup (stdoutput, code);
  if (!rel->howto)
    {
      const char *name;

      name = S_GET_NAME (fixp->fx_addsy);
      if (name == NULL)
	name = "<unknown>";
      as_fatal ("Cannot generate relocation type for symbol %s, code %s",
		name, bfd_get_reloc_code_name (code));
    }

  return rel;
}

void
md_show_usage (stream)
     FILE *stream;
{
      fprintf(stream,"--  acris version 2.03,  MA/HP, Axis Communications 1993\n");
      fprintf(stream,"\n");
      fprintf(stream,"*** Usage : acris [switches] [-o object_file] [input1...]\n");
      fprintf(stream,"Switches :\n");
      fprintf(stream,"  -f   : Turns off pre-processing.\n");
      fprintf(stream,"  -h   : Don't execute, print this help text.\n");
      fprintf(stream,"  -L   : Include local labels in object file.\n");
      fprintf(stream,"  -O   : Warn when branches are expanded to jumps.\n");
      fprintf(stream,"  -R   : Fold data segment into text segment.\n");
      fprintf(stream,"  -W   : Repress warnings.\n");
      fprintf(stream,"  Some other switches are accepted for compatibility reasons\n");
      fprintf(stream,"  but are ignored.  These are described in the users manual.\n");
      fprintf(stream,"Object_file  : Output file in the a.out format described by %s",
        "the users manual.\n");
      fprintf(stream,"                 Default output file is \"a.out\".\n");
      fprintf(stream,"Input1 ...   : Input files in the source format described by %s",
        "the users manual.\n");
      fprintf(stream,"                 Default input file is standard input.\n");
      fprintf(stream,"Description  : Assembler for the CGA-RISC processor.\n");
}

struct option md_longopts[] = {
  {NULL, no_argument, NULL, 0}
};

size_t md_longopts_size = sizeof(md_longopts);
const char *md_shortopts = "hHO";


/* Apply a fixS (fixup of an instruction or data that we didn't have
   enough info to complete immediately) to the data in a frag.
*/
int
md_apply_fix (fixP, valp)
     fixS *fixP;
     valueT *valp;
{
  long val = *valp;

  char *buf = fixP->fx_where + fixP->fx_frag->fr_literal;

  if (fixP->fx_bit_fixP || fixP->fx_im_disp != 0)
    {				/* Bitfields to fix, sigh */
      as_bad_where(fixP->fx_file, fixP->fx_line, "Invalid relocation");
    }
  else
  {
    md_number_to_imm (buf, val, fixP->fx_size, fixP);
  }

  return 1;
}

long
md_pcrel_from (fixP)
     fixS *fixP;
{
  valueT addr = fixP->fx_where + fixP->fx_frag->fr_address;

  as_bad_where(fixP->fx_file, fixP->fx_line, "Invalid relocation");
  return fixP->fx_size + addr;
}

/* We have no need to default values of symbols.  */

symbolS *
md_undefined_symbol (name)
     char *name;
{
  return 0;
}
