/* BFD backend for Cris binaries, file aout-cris.c
   Copyright (C) 1996 Free Software Foundation, Inc.
   Contributed by Hans-Peter Nilsson (hp@axis.se)
   Based on sunos.c

This file is part of BFD, the Binary File Descriptor library.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include "bfd.h"

#define N_HEADER_IN_TEXT(x) 0
#define BYTES_IN_WORD 4
#define N_TXTOFF(x) 32
#define ENTRY_CAN_BE_ZERO
#define TEXT_START_ADDR 0
#define SEGMENT_SIZE 2          /* Maybe change to 4? */

/* For some reason, if the a.out file has Z_MAGIC, then
   adata(abfd).exec_bytes_size is not used, but rather
   adata(abfd).zmagic_disk_block_size, even though the exec_header is
   *not* included in the text segment.  A simple workaround is to
   #define ZMAGIC_DISK_BLOCK_SIZE, which is used if defined; otherwise
   TARGET_PAGE_SIZE is used. */
#define ZMAGIC_DISK_BLOCK_SIZE N_TXTOFF(0)

#define TARGET_PAGE_SIZE SEGMENT_SIZE
#define TARGETNAME "a.out-cris"
#define N_SHARED_LIB(x) 0
#define DEFAULT_ARCH bfd_arch_cris

/* Disable a few braindamages in aoutx.h; relocs are assumed to follow
 a specific format, although canonicalizing hook-functions exist. */
#define USE_CANONICALIZED_RELOCS

#define MY_exec_hdr_flags 1

#define MY(OP) CAT(cris_,OP)

#define USE_EXTENDED_RELOC

#define MY_write_object_contents MY(write_object_contents)
static boolean MY(write_object_contents)();

#define MY_swap_ext_reloc_out MY(swap_ext_reloc_out)
static void MY(swap_ext_reloc_out)();

#define MY_swap_ext_reloc_in MY(swap_ext_reloc_in)
static void MY(swap_ext_reloc_in)();

#define MY_put_ext_reloc MY(put_ext_reloc)
static void MY(put_ext_reloc)();

#define MY_set_sizes MY(set_sizes)
static boolean MY(set_sizes)();

/* To set back reloc_size to ext, we make MY(set_sizes) be called
   through this construct (will not happen with
   bfd_default_set_arch_mach() ) */
#define MY_set_arch_mach NAME(aout,set_arch_mach)
#define SET_ARCH_MACH(BFD,EXEC) MY_set_arch_mach(BFD,DEFAULT_ARCH,N_MACHTYPE(EXEC))

/* Output extended relocation information to a file in target byte order. */

#define RELOC_EXT_BITS_EXTERN_CRIS 0x80
#define RELOC_EXT_BITS_TYPE_CRIS 3
#define RELOC_EXT_BITS_TYPE_SH_CRIS 0

/* Allow for "alternative" relocation usage (i.e. *not* using
   swap_{ext,std}_reloc_{in,out}() ) */
#define RELOC_EXT_BITS_EXTERN_LITTLE RELOC_EXT_BITS_EXTERN_CRIS
#define RELOC_EXT_BITS_TYPE_LITTLE RELOC_EXT_BITS_TYPE_CRIS
#define RELOC_EXT_BITS_TYPE_SH_LITTLE RELOC_EXT_BITS_TYPE_SH_CRIS

/* Include generic functions (some are overrided above). */
#include "aout32.c"

#include "aout-target.h"

/* The "CHOOSE_RELOC()" construct is malformed; "#ifdef" should have
   been used instead of "#if".
    We want 12-bytes relocation records. 
*/
static boolean
MY(write_object_contents) (abfd)
     bfd *abfd;
{
  struct external_exec exec_bytes;
  struct internal_exec *execp = exec_hdr (abfd);
  bfd_size_type text_size; /* dummy vars */
  file_ptr text_end;

  obj_reloc_entry_size (abfd) = RELOC_EXT_SIZE;

  if (bfd_get_arch(abfd) == bfd_arch_cris)
  {
    N_SET_MACHTYPE(*execp, M_CRIS);
  }

  N_SET_FLAGS (*execp, aout_backend_info (abfd)->exec_hdr_flags);

  if (adata(abfd).magic == undecided_magic)
    NAME(aout,adjust_sizes_and_vmas) (abfd, &text_size, &text_end);
  
  execp->a_syms = bfd_get_symcount (abfd) * EXTERNAL_NLIST_SIZE;
  execp->a_entry = bfd_get_start_address (abfd);
  
  execp->a_trsize = ((obj_textsec (abfd)->reloc_count) *
                     obj_reloc_entry_size (abfd));
  execp->a_drsize = ((obj_datasec (abfd)->reloc_count) *
                     obj_reloc_entry_size (abfd));
  NAME(aout,swap_exec_header_out) (abfd, execp, &exec_bytes);
  
  if (bfd_seek (abfd, (file_ptr) 0, SEEK_SET) != 0) return false;
  if (bfd_write ((PTR) &exec_bytes, 1, EXEC_BYTES_SIZE, abfd)
      != EXEC_BYTES_SIZE)
    return false;
  /* Now write out reloc info, followed by syms and strings */
  
  if (bfd_get_outsymbols (abfd) != (asymbol **) NULL
      && bfd_get_symcount (abfd) != 0)
  {
    if (bfd_seek (abfd, (file_ptr)(N_SYMOFF(*execp)), SEEK_SET)
        != 0)
      return false;
    
    if (! NAME(aout,write_syms)(abfd)) return false;
    
    if (bfd_seek (abfd, (file_ptr)(N_TRELOFF(*execp)), SEEK_SET)
        != 0)
      return false;
    
    if (!NAME(aout,squirt_out_relocs)(abfd, obj_textsec (abfd)))
      return false;
    if (bfd_seek (abfd, (file_ptr)(N_DRELOFF(*execp)), SEEK_SET)
        != 0)
      return false;
    
    if (!NAME(aout,squirt_out_relocs)(abfd, obj_datasec (abfd)))
      return false;
  }

  return true;
}

/* Similar to swap_ext_reloc_out. (This should be unified) */
static void
MY(put_ext_reloc)(abfd, r_extern, r_index, value, howto, erel, addend)
     bfd *abfd;
     int r_extern;
     int r_index;
     long value;
     reloc_howto_type *howto;
     struct reloc_ext_external *erel;
     bfd_vma addend;
{
  PUT_WORD (abfd, value, erel->r_address);

  erel->r_index[2] = r_index >> 16;
  erel->r_index[1] = r_index >> 8;
  erel->r_index[0] = r_index;
  erel->r_type[0] =
    (r_extern ? RELOC_EXT_BITS_EXTERN_CRIS : 0)
    | (howto->type << RELOC_EXT_BITS_TYPE_SH_CRIS);

  PUT_WORD (abfd, addend, erel->r_addend);
}

static void
MY(swap_ext_reloc_out) (abfd, g, natptr)
     bfd *abfd;
     arelent *g;
     register struct reloc_ext_external *natptr;
{
  int r_index;
  int r_extern;
  unsigned int r_type;
  unsigned int r_addend;
  asymbol *sym = *(g->sym_ptr_ptr);
  asection *output_section = sym->section->output_section;

  PUT_WORD (abfd, g->address, natptr->r_address);

  r_type = (unsigned int) g->howto->type;

  r_addend = g->addend;
  if ((sym->flags & BSF_SECTION_SYM) != 0)
    r_addend += (*(g->sym_ptr_ptr))->section->output_section->vma;

  /* If this relocation is relative to a symbol then set the
     r_index to the symbols index, and the r_extern bit.

     Absolute symbols can come in in two ways, either as an offset
     from the abs section, or as a symbol which has an abs value.
     check for that here.  */

  if (bfd_is_abs_section (bfd_get_section (sym)))
    {
      r_extern = 0;
      r_index = 0;
    }
  else if ((sym->flags & BSF_SECTION_SYM) == 0)
    {
      if (bfd_is_und_section (bfd_get_section (sym))
	  || (sym->flags & BSF_GLOBAL) != 0)
	r_extern = 1;
      else
	r_extern = 0;
      r_index = (*(g->sym_ptr_ptr))->KEEPIT;
    }
  else
    {
      /* Just an ordinary section */
      r_extern = 0;
      r_index = output_section->target_index;
    }

  /* The relocation type is the same as the canonical ones, but only
     the first 3 are used: RELOC_8, RELOC_16, RELOC_32.
     We may change this later, but keep a check for the moment. */
  if (r_type > 2)
  {
    (*_bfd_error_handler)
      ("%s: Illegal relocation type exported: %d",
       bfd_get_filename (abfd), r_type);
    
    bfd_set_error(bfd_error_wrong_format);
  }

  /* now the fun stuff */
  natptr->r_index[2] = r_index >> 16;
  natptr->r_index[1] = r_index >> 8;
  natptr->r_index[0] = r_index;
  natptr->r_type[0] =
     (r_extern? RELOC_EXT_BITS_EXTERN_CRIS: 0)
      | (r_type << RELOC_EXT_BITS_TYPE_SH_CRIS);

  PUT_WORD (abfd, r_addend, natptr->r_addend);
}

/* Translate cris-reloc into standard reloc */
static void
MY(swap_ext_reloc_in) (abfd, bytes, cache_ptr, symbols, symcount)
     bfd *abfd;
     struct reloc_ext_external *bytes;
     arelent *cache_ptr;
     asymbol **symbols;
     bfd_size_type symcount;
{
  unsigned int r_index;
  int r_extern;
  unsigned int r_type;
  struct aoutdata *su = &(abfd->tdata.aout_data->a);

  cache_ptr->address = (GET_SWORD (abfd, bytes->r_address));

  /* now the fun stuff */
  r_index =  (bytes->r_index[2] << 16)
    | (bytes->r_index[1] << 8)
    |  bytes->r_index[0];
  r_extern = (0 != (bytes->r_type[0] & RELOC_EXT_BITS_EXTERN_CRIS));
  r_type   =       (bytes->r_type[0]) >> RELOC_EXT_BITS_TYPE_SH_CRIS;

  if (r_type > 2)
  {
    (*_bfd_error_handler)
      ("%s: Illegal relocation type imported: %d",
       bfd_get_filename (abfd), r_type);
    
    bfd_set_error(bfd_error_wrong_format);
  }
    
  cache_ptr->howto =  howto_table_ext + r_type;

  if (r_extern && r_index > symcount)
    {
      (*_bfd_error_handler)
        ("%s: Bad relocation record imported: %d",
         bfd_get_filename (abfd), r_index);
    
      bfd_set_error(bfd_error_wrong_format);
      /* We could arrange to return an error, but it might be useful
         to see the file even if it is bad.  */
      r_extern = 0;
      r_index = N_ABS;
    }

  MOVE_ADDRESS(GET_SWORD(abfd, bytes->r_addend));
}

static boolean
MY(set_sizes)(abfd)
     bfd *abfd;
{
  /* Just as the default in aout-target.h ... */

  adata(abfd).page_size = TARGET_PAGE_SIZE;

#ifdef SEGMENT_SIZE
  adata(abfd).segment_size = SEGMENT_SIZE;
#else
  adata(abfd).segment_size = TARGET_PAGE_SIZE;
#endif

#ifdef ZMAGIC_DISK_BLOCK_SIZE
  adata(abfd).zmagic_disk_block_size = ZMAGIC_DISK_BLOCK_SIZE;
#else
  adata(abfd).zmagic_disk_block_size = TARGET_PAGE_SIZE;
#endif

  adata(abfd).exec_bytes_size = EXEC_BYTES_SIZE;

  /* ... except for that we have the extended reloc 
   (I don't really want to make aoutx.h more dirty by adding a check
   on bfd_arch_cris in NAME(aout,set_arch_mach) since it really does
   not belong there - should check on USE_EXTENDED_RELOC instead). */

  obj_reloc_entry_size (abfd) = RELOC_EXT_SIZE;
  
  return true;

}

/* End of aout-cris.c */
