/** \AMPFileSource.cpp
* 
* Implementation of AMP FileSource
*
* (C) Copyright 2010, Axis Communications AB, LUND, SWEDEN
****************************************************************************/


/****************** INCLUDE FILES SECTION ***********************************/

#include <Streams.h>

#include "AMPFileSource.h"

/****************** DEFINITION SECTION **************************************/

const GUID MEDIATYPE_AxisCommunications = {0x34f14b, 0x454c, 0x400e, 
                                           {0xa0, 0x75, 0xd8, 0xba, 0x5e, 0x14, 0x92, 0xc8}};

typedef struct tagAMP_MEDIA_TYPE {
  GUID    majortype;
  GUID    subtype;
  GUID    formattype;
  ULONG   cbFormat;
  BYTE    rgbFormat[1];
} AMP_MEDIA_TYPE;

typedef struct tagAXIS_MEDIA_TYPE_V1 {
  GUID      majortype;
  GUID      subtype;
  BOOL      bFixedSizeSamples;
  BOOL      bTemporalCompression;
  ULONG     lSampleSize;
  GUID      formattype;
  ULONG     cbFormat;
  BYTE      formatdata[1];
} AXIS_MEDIA_TYPE_V1;

enum AMP_VIDEO_SAMPLE_TYPE {
  AMP_VST_UNKNOWN = 0,
  AMP_VST_MPEG4_VIDEO_CONFIG = 200,
  AMP_VST_MPEG4_VIDEO_IVOP = 201,
  AMP_VST_MPEG4_VIDEO_PVOP = 202,
  AMP_VST_MPEG4_VIDEO_BVOP = 203,
  AMP_VST_MPEG4_VIDEO_FRAGMENT = 204,
  AMP_VST_MJPEG_VIDEO = 205,
  AMP_VST_H264_VIDEO_CONFIG = 206,
  AMP_VST_H264_VIDEO_IDR = 207,
  AMP_VST_H264_VIDEO_NON_IDR = 208,
  AMP_VST_H264_VIDEO_SEI = 209,
  AMP_VST_MPEG4_AUDIO = 300,
  AMP_VST_METADATA_ONVIF = 400
};

enum AMP_SAMPLE {
  AMP_S_SYNCPOINT = 0x01,
  AMP_S_INCOMPLETE = 0x04,
  AMP_S_STARTTIMEVALID = 0x10,
  AMP_S_STOPTIMEVALID = 0x100
};

AMPFileSource::AMPFileSource()
: myFileHandle(NULL)
, myStreamMediaTypes(NULL)
, myNumberOfStreams(0)
, isLoaded(false)
, myVideoStreamIndex(-1)
, myAudioStreamIndex(-1)
, myNextSampleSize(0)
{

}

AMPFileSource::~AMPFileSource()
{
  (void)this->Close();
}

HRESULT
AMPFileSource::Close()
{
  if (myFileHandle)
  {
    CloseHandle(myFileHandle);
  }
  if(myStreamMediaTypes != NULL)
  {
    for(DWORD aStreamIndex = 0; aStreamIndex < myNumberOfStreams; aStreamIndex++)
    {
      DeleteMediaType(myStreamMediaTypes[aStreamIndex]);
    }
    delete [] myStreamMediaTypes;
  }

  myFileHandle = NULL;
  myStreamMediaTypes = NULL;

  myNumberOfStreams = 0;
  isLoaded = false;
  myVideoStreamIndex = -1;
  myAudioStreamIndex = -1;
  myNextSampleSize = 0;

  return S_OK;
}

HRESULT
AMPFileSource::Load(_TCHAR* theFile)
{
  HRESULT hr = S_OK;
  GUID aGuid;
  BYTE aFileVersion = 0;
  DWORD aReturn;
  BYTE* aBuffer = NULL;
  DWORD aBufferSize;

  if(isLoaded)
  {
    (void)this->Close();
  }

  _ASSERT(myStreamMediaTypes == NULL);

  try
  {
    myFileHandle = CreateFile(theFile, GENERIC_READ, 0, NULL, OPEN_EXISTING, 0, NULL);
    if (myFileHandle == INVALID_HANDLE_VALUE)
    {
      return E_FAIL;
    }

    ReadFile(myFileHandle, &aBufferSize, 4, &aReturn, NULL);
    aBuffer = new BYTE[aBufferSize];
    ReadFile(myFileHandle, aBuffer, aBufferSize, &aReturn, NULL);

    BYTE* aReadPtr = aBuffer;

    CopyMemory(&aGuid, aReadPtr, 16);
    if(aGuid == MEDIATYPE_AxisCommunications)
    {
      aReadPtr += 16; // MEDIATYPE_AxisCommunications
      aFileVersion = *aReadPtr;
      aReadPtr += 1; // Version

      myNumberOfStreams = *aReadPtr;
      aReadPtr += 1;
      myNumberOfStreams = min(myNumberOfStreams,MAX_NUMBER_OF_STREAMS);
    }
    else
    {
      // single AMP_MEDIA_TYPE
      aFileVersion = 0;
      myNumberOfStreams = 1;
    }

    myStreamMediaTypes = new AM_MEDIA_TYPE*[myNumberOfStreams];

    for(DWORD aStreamIndex = 0; aStreamIndex < myNumberOfStreams; aStreamIndex++)
    {
      if(aFileVersion == 0)
      {
        _ASSERT(myNumberOfStreams == 1);

        AMP_MEDIA_TYPE* anAmpMediaType = (AMP_MEDIA_TYPE*)aReadPtr;

        myStreamMediaTypes[aStreamIndex] = (AM_MEDIA_TYPE*)CoTaskMemAlloc(sizeof(AM_MEDIA_TYPE));
        ZeroMemory(myStreamMediaTypes[aStreamIndex],sizeof(AM_MEDIA_TYPE));
        myStreamMediaTypes[aStreamIndex]->majortype = anAmpMediaType->majortype;
        myStreamMediaTypes[aStreamIndex]->subtype = anAmpMediaType->subtype;
        myStreamMediaTypes[aStreamIndex]->formattype = anAmpMediaType->formattype;

        BYTE* aFormatBlock = (BYTE*)CoTaskMemAlloc(anAmpMediaType->cbFormat);
        CopyMemory(aFormatBlock, anAmpMediaType->rgbFormat, anAmpMediaType->cbFormat);

        myStreamMediaTypes[aStreamIndex]->cbFormat = anAmpMediaType->cbFormat;
        myStreamMediaTypes[aStreamIndex]->pbFormat = aFormatBlock;

        aReadPtr += sizeof(AMP_MEDIA_TYPE) - sizeof(BYTE) + anAmpMediaType->cbFormat;
      }
      else if (aFileVersion == 1)
      {
        AXIS_MEDIA_TYPE_V1* anAxisMediaType = (AXIS_MEDIA_TYPE_V1*)aReadPtr;

        myStreamMediaTypes[aStreamIndex] = (AM_MEDIA_TYPE*)CoTaskMemAlloc(sizeof(AM_MEDIA_TYPE));
        ZeroMemory(myStreamMediaTypes[aStreamIndex],sizeof(AM_MEDIA_TYPE));
        myStreamMediaTypes[aStreamIndex]->majortype = anAxisMediaType->majortype;
        myStreamMediaTypes[aStreamIndex]->subtype = anAxisMediaType->subtype;
        myStreamMediaTypes[aStreamIndex]->formattype = anAxisMediaType->formattype;
        myStreamMediaTypes[aStreamIndex]->bFixedSizeSamples = anAxisMediaType->bFixedSizeSamples;
        myStreamMediaTypes[aStreamIndex]->bTemporalCompression = anAxisMediaType->bTemporalCompression;
        myStreamMediaTypes[aStreamIndex]->lSampleSize = anAxisMediaType->lSampleSize;

        BYTE* aFormatBlock = (BYTE*)CoTaskMemAlloc(anAxisMediaType->cbFormat);
        CopyMemory(aFormatBlock, anAxisMediaType->formatdata, anAxisMediaType->cbFormat);

        myStreamMediaTypes[aStreamIndex]->cbFormat = anAxisMediaType->cbFormat;
        myStreamMediaTypes[aStreamIndex]->pbFormat = aFormatBlock;

        aReadPtr += sizeof(AXIS_MEDIA_TYPE_V1) - sizeof(BYTE) + anAxisMediaType->cbFormat;
      } 
      else
      {
        // unsupported AMP file version
        hr = E_FAIL;
        break;
      }
    }

    delete [] aBuffer;
  }
  catch (...)
  {
    if(aBuffer != NULL)
    {
      delete [] aBuffer;
    }
    return E_FAIL;
  }

  for(DWORD aStreamIndex = 0; aStreamIndex < myNumberOfStreams; aStreamIndex++)
  {
    if(myStreamMediaTypes[aStreamIndex]->majortype == MEDIATYPE_Video)
    {
      _ASSERT(myVideoStreamIndex == -1); // AMP format inherently only support one video stream
      myVideoStreamIndex = aStreamIndex;
    }
    else if(myStreamMediaTypes[aStreamIndex]->majortype == MEDIATYPE_Audio)
    {
      _ASSERT(myAudioStreamIndex == -1); // AMP format inherently only support one audio stream
      myAudioStreamIndex = aStreamIndex;
    }
  }

  if(SUCCEEDED(hr))
  {
    isLoaded = true;
  }

  return hr;
}

HRESULT
AMPFileSource::GetStreamInfo(AM_MEDIA_TYPE** theMediaType, DWORD theIndex)
{
  if(!isLoaded)
  {
    return E_FAIL;
  }

  if(theIndex < myNumberOfStreams)
  {
    *theMediaType = CreateMediaType(myStreamMediaTypes[theIndex]);
    return S_OK;
  }
  return S_FALSE;
}

HRESULT
AMPFileSource::GetNextSampleProperties(DWORD* theStreamIndex, DWORD* theSampleSize,
                                       REFERENCE_TIME* theStartTime, REFERENCE_TIME* theStopTime,
                                       BOOL* isSyncPoint)
{
  DWORD aReturn;
  DWORD aSampleType;
  DWORD someFlags;

  if(!isLoaded)
  {
    return E_FAIL;
  }

  if(theStartTime == NULL || theStopTime == NULL || isSyncPoint == NULL || 
     theStreamIndex == NULL || theSampleSize == NULL)
  {
    return E_POINTER;
  }

  ReadFile(myFileHandle, &aSampleType, 4, &aReturn, NULL);
  if (aReturn != 4)
  {
    // end-of-file
    return S_FALSE;
  }

  // Read headers from file
  ReadFile(myFileHandle, &someFlags, 4, &aReturn, NULL);
  ReadFile(myFileHandle, theStartTime, 8, &aReturn, NULL);
  ReadFile(myFileHandle, theStopTime, 8, &aReturn, NULL);
  ReadFile(myFileHandle, &myNextSampleSize, 4, &aReturn, NULL);

  if(aSampleType == AMP_VST_MPEG4_AUDIO) //audio
  {
    _ASSERT(myAudioStreamIndex >= 0);
    *theStreamIndex = myAudioStreamIndex;
  }
  else if(aSampleType != AMP_VST_METADATA_ONVIF) // video
  {
    _ASSERT(myVideoStreamIndex >= 0);
    *theStreamIndex = myVideoStreamIndex;
  }

  *isSyncPoint = (someFlags&AMP_S_SYNCPOINT) > 0;
  *theSampleSize = myNextSampleSize;

  return S_OK;
}

HRESULT
AMPFileSource::FillBuffer(BYTE* theBuffer, DWORD theBufferLength)
{
  HRESULT hr = S_OK;
  DWORD aReturn;

  if(!isLoaded)
  {
    return E_FAIL;
  }

  if(theBuffer == NULL)
  {
    return E_POINTER;
  }

  if(myNextSampleSize == 0)
  {
    // Call GetNextSampleProperties before calling this function
    return E_FAIL;
  }

  if(myNextSampleSize > theBufferLength)
  {
    return E_INVALIDARG;
  }

  ReadFile(myFileHandle, theBuffer, myNextSampleSize, &aReturn, NULL);
  if (aReturn != myNextSampleSize)
  {
    // unexpected end-of-file
    hr = E_FAIL;
  }
  myNextSampleSize = 0;
  return hr;
}

HRESULT
AMPFileSource::SkipBuffer()
{
  HRESULT hr = S_OK;

  if(!isLoaded)
  {
    return E_FAIL;
  }

  if(myNextSampleSize == 0)
  {
    // Call GetNextSampleProperties before calling this function
    return E_FAIL;
  }

  DWORD ret = ::SetFilePointer(myFileHandle, myNextSampleSize, NULL, FILE_CURRENT);
  if(ret < 0)
  {
    // unexpected end-of-file
    hr = E_FAIL;
  }
  return hr;
}